/* vi: set sw=4 ts=4: */
/*
 * Licensed under GPLv2 or later, see file LICENSE in this tarball for details.
 */

#include "libbb.h"
#include <syslog.h>
#include <utmp.h>
#include <sys/resource.h>
#include <net/if.h>

#if ENABLE_SELINUX
#include <selinux/selinux.h>  /* for is_selinux_enabled()  */
#include <selinux/get_context_list.h> /* for get_default_context() */
#include <selinux/flask.h> /* for security class definitions  */
#endif

//for draytek telnet command
#include "testfunc.h"
#include <asm/unistd.h>
#include <signal.h>



#if ENABLE_PAM
/* PAM may include <locale.h>. We may need to undefine bbox's stub define: */
#undef setlocale
/* For some obscure reason, PAM is not in pam/xxx, but in security/xxx.
 * Apparently they like to confuse people. */
#include <security/pam_appl.h>
#include <security/pam_misc.h>
static const struct pam_conv conv = {
	misc_conv,
	NULL
};
#endif

enum {
	TIMEOUT = 60,
	EMPTY_USERNAME_COUNT = 10,
	USERNAME_SIZE = 32,
	TTYNAME_SIZE = 32,
};

static char* short_tty;

#define CUSTOM_DHCP_OPTION 1

#if ENABLE_FEATURE_UTMP
/* vv  Taken from tinylogin utmp.c  vv */
/*
 * read_or_build_utent - see if utmp file is correct for this process
 *
 *	System V is very picky about the contents of the utmp file
 *	and requires that a slot for the current process exist.
 *	The utmp file is scanned for an entry with the same process
 *	ID.  If no entry exists the process exits with a message.
 *
 *	The "picky" flag is for network and other logins that may
 *	use special flags.  It allows the pid checks to be overridden.
 *	This means that getty should never invoke login with any
 *	command line flags.
 */

static void read_or_build_utent(struct utmp *utptr, int picky)
{
	struct utmp *ut;
	pid_t pid = getpid();

	setutent();

	/* First, try to find a valid utmp entry for this process.  */
	while ((ut = getutent()))
		if (ut->ut_pid == pid && ut->ut_line[0] && ut->ut_id[0] &&
		(ut->ut_type == LOGIN_PROCESS || ut->ut_type == USER_PROCESS))
			break;

	/* If there is one, just use it, otherwise create a new one.  */
	if (ut) {
		*utptr = *ut;
	} else {
		if (picky)
			bb_error_msg_and_die("no utmp entry found");

		memset(utptr, 0, sizeof(*utptr));
		utptr->ut_type = LOGIN_PROCESS;
		utptr->ut_pid = pid;
		strncpy(utptr->ut_line, short_tty, sizeof(utptr->ut_line));
		/* This one is only 4 chars wide. Try to fit something
		 * remotely meaningful by skipping "tty"... */
		strncpy(utptr->ut_id, short_tty + 3, sizeof(utptr->ut_id));
		strncpy(utptr->ut_user, "LOGIN", sizeof(utptr->ut_user));
		utptr->ut_tv.tv_sec = time(NULL);
	}
	if (!picky)	/* root login */
		memset(utptr->ut_host, 0, sizeof(utptr->ut_host));
}

/*
 * write_utent - put a USER_PROCESS entry in the utmp file
 *
 *	write_utent changes the type of the current utmp entry to
 *	USER_PROCESS.  the wtmp file will be updated as well.
 */
static void write_utent(struct utmp *utptr, const char *username)
{
	utptr->ut_type = USER_PROCESS;
	strncpy(utptr->ut_user, username, sizeof(utptr->ut_user));
	utptr->ut_tv.tv_sec = time(NULL);
	/* other fields already filled in by read_or_build_utent above */
	setutent();
	pututline(utptr);
	endutent();
#if ENABLE_FEATURE_WTMP
	if (access(bb_path_wtmp_file, R_OK|W_OK) == -1) {
		close(creat(bb_path_wtmp_file, 0664));
	}
	updwtmp(bb_path_wtmp_file, utptr);
#endif
}
#else /* !ENABLE_FEATURE_UTMP */
#define read_or_build_utent(utptr, picky) ((void)0)
#define write_utent(utptr, username) ((void)0)
#endif /* !ENABLE_FEATURE_UTMP */

#if ENABLE_FEATURE_NOLOGIN
static void die_if_nologin(void)
{
	FILE *fp;
	int c;

	if (access("/etc/nologin", F_OK))
		return;

	fp = fopen_for_read("/etc/nologin");
	if (fp) {
		while ((c = getc(fp)) != EOF)
			bb_putchar((c=='\n') ? '\r' : c);
		fflush(stdout);
		fclose(fp);
	} else
		puts("\r\nSystem closed for routine maintenance\r");
	exit(EXIT_FAILURE);
}
#else
static ALWAYS_INLINE void die_if_nologin(void) {}
#endif

#if ENABLE_FEATURE_SECURETTY && !ENABLE_PAM
static int check_securetty(void)
{
	char *buf = (char*)"/etc/securetty"; /* any non-NULL is ok */
	parser_t *parser = config_open2("/etc/securetty", fopen_for_read);
	while (config_read(parser, &buf, 1, 1, "# \t", PARSE_NORMAL)) {
		if (strcmp(buf, short_tty) == 0)
			break;
		buf = NULL;
	}
	config_close(parser);
	/* buf != NULL here if config file was not found, empty
	 * or line was found which equals short_tty */
	return buf != NULL;
}
#else
static ALWAYS_INLINE int check_securetty(void) { return 1; }
#endif

static void get_username_or_die(char *buf, int size_buf)
{
	int c, cntdown;

	cntdown = EMPTY_USERNAME_COUNT;
 prompt:
	print_login_prompt();
	/* skip whitespace */
	do {
		c = getchar();
		if (c == EOF) exit(EXIT_FAILURE);
		if (c == '\n') {
			if (!--cntdown) exit(EXIT_FAILURE);
			goto prompt;
		}
	} while (isspace(c));

	*buf++ = c;
	if (!fgets(buf, size_buf-2, stdin))
		exit(EXIT_FAILURE);
	if (!strchr(buf, '\n'))
		exit(EXIT_FAILURE);
	while (isgraph(*buf)) buf++;
	*buf = '\0';
}

static void motd(void)
{
	int fd;

	fd = open(bb_path_motd_file, O_RDONLY);
	if (fd >= 0) {
		fflush(stdout);
		bb_copyfd_eof(fd, STDOUT_FILENO);
		close(fd);
	}
}

/**************************Draytek **********************************/

//vigorfly telnet command function, Bruce Hsu 2009/12/30
#define TEL_BUF_LEN 128

/*
 * TUI's command structure
 */
typedef struct cmds {
    char   *name;               /* Name of command */
    int     (*func)(char *argv, int cmd_level);
                                /* Function to execute command */
    int     cmd_lvls;            /* command levels for further use */
} cmds_t;

struct cmd_result{
	int 	ret;
	int		level;
	char	*dump;
};

/* return value for telnet command */
#define		CMD_OK			0
#define		CMD_BREAK		1
#define		CMD_FAIL		2

#define CHUNK 256    // i think Chunk size=256 is quite enough   //bruce hsu
static char *slurp(FILE *p)
{
    size_t off = 0, tot = CHUNK, n;
    char *buf = malloc(CHUNK+1); /* Zero terminate */
    if (!buf)
        return NULL;
    while((n = fread(buf+off, 1, CHUNK, p)) > 0) {
        off += n, tot += n;
        buf = realloc(buf, tot);
    }
    buf[off] = '\0';  /* Zero terminate - we have room */
    return buf;
}

char *run_command(const char *cmd)
{
    FILE *p = popen(cmd, "r");
    char *buf = NULL;
    if(p) {
        buf = slurp(p);
        pclose(p);
    } else {
        perror(cmd);
    }
	if (buf && isspace(buf[strlen(buf)-1]))	// remove '0x0a' at end of buf
		buf[strlen(buf)-1] = '\0';
    return buf;
}

int do_exec(char *buff, int adv_cmd)
{
	if(buff){
		system(buff);
	}
	return CMD_OK;
}

int do_exit(char *buff, int adv_cmd)
{
	exit(0);
	return CMD_OK;
}

int do_hw_nat(char *buff, int adv_cmd)
{
    char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+7;  // 7 is length of "hw_nat "
    unsigned short value=0;
    unsigned char *uci1=NULL;
    unsigned char *uci2=NULL;
    if(strlen(buff)<9){        // 7 is length of "hw_nat xx"
        printf("Command format should be: hw_nat [on/off/show]\n");
    }else{      //set wan_phy mode
        //mode
        if(strncmp(ptr,"on",2)==0){ //hw_nat on
            system("echo !!!! HW NAT function is only for test in this version !!!!");
            system("[ -f \/lib\/modules\/2.6.21\/kernel\/net\/nat\/hw_nat\/hw_nat.ko ] && insmod hw_nat > dev/null");
	        sleep(1);
            uci1=run_command("lsmod | grep hw_nat");
            if(uci1){
                if(strlen(uci1)>0)
                    system("echo HW_NAT ON");
                free(uci1);
            }
        }else if(strncmp(ptr,"off",3)==0){ //hw_nat off
            system("echo !!!! HW NAT function is only for test in this version !!!!");
            uci1=run_command("lsmod | grep hw_nat");
            if(uci1){
                if(strlen(uci1)>0){
                    system("rmmod hw_nat");
                }
                system("echo HW_NAT OFF");
                free(uci1);
            }
        }else if(strncmp(ptr,"show",4)==0){ //hw_nat show
            system("echo !!!! HW NAT function is only for test in this version !!!!");
            uci1=run_command("lsmod | grep hw_nat");
            if(uci1){
                if(strlen(uci1)>0){
                    system("hw_nat -g");
                }else
                    system("echo HW_NAT not enabled");
                free(uci1);
            }
        }
    }

	return CMD_OK;
}

int do_arp(char *buff, int adv_cmd)
{
    char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+4;  // 4 is length of "arp "
    if(strlen(buff)<6){        // 6 is length of "arp xx"
        printf("Command format should be: arp [flush/status]\n");
    }else{      //set wan_phy mode
        //mode
        if(strncmp(ptr,"flush",5)==0){ //arp flush
        	system("ifconfig br0 down");
	        sleep(1);
	        system("ifconfig br0 up");
	        sleep(1);
	        system("goahead static_arp");
        }else if(strncmp(ptr,"status",6)==0){ //arp status
            system("arp -n -a -i br0");
        }
    }

	return CMD_OK;
}

int do_reboot(char *buff, int adv_cmd)
{

    system("reboot");

	return CMD_OK;
}

#if CUSTOM_DHCP_OPTION == 1
static int setargs(char *args, char **argv)
{
   int count = 0;

   while (isspace(*args))
      ++args;
   while (*args) {
       if (argv)
           argv[count] = args;
       while (*args && !isspace(*args))
           ++args;
       if (argv && *args){
           *args++ = '\0';}
       while (isspace(*args))
           ++args;
       count++;
   }

   return count;
}

char **parsedargs(char *args, int *argc)
{
   char **argv = NULL;
   int argn = 0;

   if (args && *args
    && (args = strdup(args))
    && (argn = setargs(args, NULL))
    && (argv = malloc((argn+1) * sizeof(char *)))) {
       *argv++ = args;
       argn = setargs(args,argv);
   }

   if (args && !argv)
       free(args);

   *argc = argn;
   return argv;
}

void freeparsedargs(char **argv)
{
  if (argv) {
      free(argv[-1]);
      free(argv-1);
  }
}

void do_dhcp_option_help(void)
{
    printf("ip dhcpc option\r\n");
    printf("Usage:\r\n");
    printf("-c: set option number: 0~255\r\n");
    printf("-d: delete custom dhcp client option by index number\r\n");
    printf("-e: enable/disable option feature, 1:enable, 0:disable\r\n");
    printf("-h: display help\r\n");
    printf("-i: set index number: 0~5\r\n");
    printf("-l: list all custom set DHCP options\r\n");
    printf("-v: set option value by string\r\n");
    printf("-x: set option value by raw byte (hex)\r\n\r\n");

    printf("If you want to get help:\r\n");
    printf("Ex: ip dhcpc option -h\r\n\r\n");

    printf("If you want to list all DHCP Options:\r\n");
    printf("Ex: ip dhcpc option -l\r\n\r\n");

    printf("If you want to delete a DHCP Option:\r\n");
    printf("ip dhcpc option -d [idx]\r\n");
    printf("Ex: ip dhcpc option -d 1\r\n\r\n");

    printf("If you want to set a DHCP Option value by string:\r\n");
    printf("ip dhcpc option -e [1 or 0] -i [index unmber] -c [option number] -v [option value]\r\n");
    printf("Ex: ip dhcpc option -e 1 -i 1 -c 12 -v draytek\r\n\r\n");

    printf("If you want to set a DHCP Option value by raw byte (hex):\r\n");
    printf("ip dhcpc option -e [1 or 0] -i [index unmber] -c [option number] -x [option value]\r\n");
    printf("Ex: ip dhcpc option -e 1 -i 1 -c 61 -x 01112233445566\r\n\r\n");
}

int do_modify_dhcp_option(unsigned char *buff, int adv_cmd)
{
    char *op_val_str = 0, *op_val_hex = 0, *list = 0;
    char cmd[200];
    char en[5], data_type[10], numb[10], val[200];
    int  m = 0;
    int argc = 0;
    char **argv;
    int result;
    int op_index = -1;
    int del_num = -1;
    int op_num = -1;
    int enable = -1;

    int flag_c = 0;
    int flag_d = 0;
    int flag_e = 0;
    int flag_i = 0;
    int flag_v = 0;
    int flag_x = 0;

    argv = parsedargs(buff, &argc);

    optind = 0;

    while( (result = getopt(argc, argv, "c:d:e:hi:lv:x:")) != -1 )
    {
        switch(result)
        {
            case 'c':
                if (optarg == NULL) {
                    printf("Option -c value is empty.\r\n");
                    printf("If you want to add a DHCP Option:\r\n");
                    printf("Ex: ip dhcpc option -e 1 -i 1 -c 12 -v draytek\r\n\r\n");
                    goto _close;
                }
                op_num = atoi(optarg);
                if (op_num > 255 || op_num < 0) {
                    printf("The option number must be between 0 and 255.");
                    goto _close;
                }
                flag_c = 1;
                break;
            case 'd':
                if (optarg == NULL) {
                    printf("Option -d value is empty.\r\n");
                    printf("If you want to delete a DHCP Option:\r\n");
                    printf("Ex: ip dhcpc option -d 1\r\n\r\n");
                    goto _close;
                }
                del_num = atoi(optarg);
                if (del_num > 6 || del_num < 0) {
                    printf("The index number must be between 0 and 5.");
                    goto _close;
                }
                flag_d = 1;
                break;
            case 'e':
                if (optarg == NULL) {
                    printf("Option -e value is empty.\r\n");
                    printf("If you want to add a DHCP Option:\r\n");
                    printf("Ex: ip dhcpc option -e 1 -i 1 -c 12 -v draytek\r\n\r\n");
                    goto _close;
                }
                enable = atoi(optarg);
                if (enable > 1 || enable < 0) {
                    printf("The enable number must be 1 or 0.");
                    goto _close;
                }
                flag_e = 1;
                break;
            case 'h':
                do_dhcp_option_help();
                goto _close;
            case 'i':
                if (optarg == NULL) {
                    printf("Option -i value is empty.\r\n");
                    printf("If you want to add a DHCP Option:\r\n");
                    printf("Ex: ip dhcpc option -e 1 -i 1 -c 12 -v draytek\r\n\r\n");
                    goto _close;
                }
                op_index = atoi(optarg);
                if (op_index > 5 || op_index < 0) {
                    printf("The index number must be between 0 and 5.\r\n\r\n");
                    goto _close;
                }
                flag_i = 1;
                break;
            case 'l':
                printf("List of custom set DHCPC Options:\r\n");
                for (m = 0; m < 6; m++) {
                    sprintf(cmd, "nvram_get 2860 dhcpc_option_%d", m);
                    list = run_command(cmd);
                    if (strchr(list, ';') != NULL) {
                        getNthValueSafe(0, list, ';', en, sizeof(en));
                        getNthValueSafe(1, list, ';', data_type, sizeof(data_type));
                        getNthValueSafe(2, list, ';', numb, sizeof(numb));
                        getNthValueSafe(3, list, ';', val, sizeof(val));

                        printf("DHCPC Option %d: Enable = %s, Value type = %s, Number = %s, Value = %s\r\n", m, en, data_type, numb, val);
                    }
                    if(list) {
                        free(list);
                    }
                }
                goto _close;
            case 'v':
                if (!flag_v) {
                    if (optarg == NULL) {
                        printf("Option -v value is empty.\r\n");
                        printf("If you want to set a DHCP Option value by string:\r\n");
                        printf("Ex: ip dhcpc option -e 1 -i 1 -c 12 -v draytek\r\n\r\n");
                        goto _close;
                    }
                    op_val_str = strdup(optarg);
                    flag_v = 1;
                }
                break;
            case 'x':
                if (!flag_x) {
                    if (strlen(optarg) > 126) {//63 x 2 = 126 ,"00112233 ..."
                        printf("Option -x value is too long.  Maximum length is 126.\r\n\r\n");
                        goto _close;
                    } else if (optarg == NULL) {
                        printf("Option -x value is empty.\r\n");
                        printf("If you want to set a DHCP Option value by raw byte (hex):\r\n");
                        printf("Ex: ip dhcpc option -e 1 -i 1 -c 61 -x 01112233445566\r\n\r\n");
                        goto _close;
                    }
                    op_val_hex = strdup(optarg);
                    flag_x = 1;
                }
                break;
            default:
                do_dhcp_option_help();
                goto _close;
         }

    }

    if (flag_v && flag_c && flag_i && flag_e && op_index >= 0 && op_index < 6) {
        sprintf(cmd, "nvram_set dhcpc_option_%d \"%d;String;%d;%s\"", op_index, enable, op_num, op_val_str);
        system(cmd);
        printf("Set idx %d success!\r\n", op_index);
    } else if (flag_x && flag_c && flag_i && flag_e && op_index >= 0 && op_index < 6) {
        sprintf(cmd, "nvram_set dhcpc_option_%d \"%d;Raw;%d;%s\"", op_index, enable, op_num, op_val_hex);
        system(cmd);
        printf("Set idx %d success!\r\n", op_index);
    } else if (flag_d && del_num >= 0 && del_num < 6) {
        sprintf(cmd, "nvram_set dhcpc_option_%d \"%s\"", del_num, "");
        system(cmd);
        printf("Delete idx %d success!\r\n", del_num);
    } else {
        do_dhcp_option_help();
    }

_close:

    if(op_val_str)
        free(op_val_str);

    if(op_val_hex)
        free(op_val_hex);

    freeparsedargs(argv);

	return CMD_OK;
}
#endif

int do_restart_web(char *buff, int adv_cmd)
{
	char temp[30];
	sprintf(temp, "killall -%d goahead", (SIGRTMIN+3));
	system(temp);
	return CMD_OK;
}

int do_switch_reg(char *buff, int adv_cmd)
{
    int reg=0;
    int reg_cmd[TEL_BUF_LEN]={0};

	for(reg=0;reg<=0xfc;reg+=4 ){
		memset(reg_cmd, 0, TEL_BUF_LEN);
		sprintf(reg_cmd,"switch reg r %02X", reg);
		system(reg_cmd);
	}
	return CMD_OK;
}

int do_3ginfo(char *buff, int adv_cmd)
{
	system("[ -f /var/3g/wan_3gdev ]&& cat /var/3g/wan_3gdev");
	system("[ -f /var/3g/wan_dev ]&& cat /var/3g/wan_dev");
	system("[ -f /var/3g.chat ]&& cat /var/3g.chat");
	system("cat /proc/bus/usb/devices");
	return CMD_OK;
}

int do_reset_usb(char *buff, int adv_cmd)
{
	system("gpio u");
	sleep(1);
	system("/sbin/usb_led_ctrl.sh");
	return CMD_OK;
}

int do_force_3g(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char iface[20]={0};
    char *ptr=buff+11;  // 11 is length of "force_3gif "

    if(strlen(buff)<=15){        // 15 is length of "force_3gif /xxx"
        printf("#force_3gif [Interface]\n");
        printf("#EX: force_3gif /dev/ttyUSB0\n");
        ptr=run_command("[ -f /var/3g/force_dev ]&& cat /var/3g/force_dev");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{      //set force 3G device
        sscanf(ptr, "%s",iface);
        sprintf(cmd,"echo %s > /var/3g/force_dev",iface);
        system(cmd);
        //restart wan_det.sh, do not reset 3G wan
        system("killall wan_det.sh");
        system("/sbin/wan_det.sh force_3g &");
    }
    return CMD_OK;
}

int do_force_usb(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+11;  // 11 is length of "force_3gid "
    unsigned short v_id=0, p_id=0, r_id=0;
    if(strlen(buff)<=17){        // 17 is length of "force_3gid /x/x/x"
        printf("#force_3gid [Devce ID]\n");
        printf("#EX: force_3gid 1234/5678/0\n\n");
    }else{      //set force 3G device
        //step1. unload usb serial driver
        system("rmmod usbserial");
        sleep(3);
        //step2. get device vid, pid.
        sscanf(ptr, "%x\/%x\/%x",&v_id,&p_id,&r_id);
        //step3. insert usb serial driver
        sprintf(cmd, "insmod usbserial vendor=0x%04x product=0x%04x", v_id, p_id);
        system(cmd);
        printf("%s\n", cmd);
    }
	return CMD_OK;
}

int do_force_3gcmd(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char switch_cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+12;  // 12 is length of "force_3gcmd "

    if(strlen(buff)<=70){        // 70 is length of "force_3gcmd xxxxxxxxx..."
        printf("#force_3gcmd [modeswitch command in hex]\n");
        printf("#EX: force_3gcmd 55534243000000000000000000000011060000000000000000000000000000\n");
        ptr=run_command("[ -f /var/3g/force_3gcmd ]&& cat /var/3g/force_3gcmd");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{       //set force 3G modeswitch command
        sscanf(ptr, "%s",switch_cmd);
        sprintf(cmd,"echo %s > /var/3g/force_3gcmd",switch_cmd);
        system(cmd);
        //restart wan_det.sh, do not reset 3G wan
        system("killall wan_det.sh");
        system("/sbin/wan_det.sh force_3g &");
    }
	return CMD_OK;
}

int do_extend_3g(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+12;  // 12 is length of "extend_3gif "

    if(strlen(buff)<=20){        // 20 is length of "extend_3gif /dev/tty"
        printf("#extend_3gif [Interface]\n");
        printf("#EX: extend_3gif /dev/ttyUSB0\n");
        ptr=run_command("nvram_get 2860 extend_3gif");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{       //save config
        sprintf(cmd,"nvram_set 2860 extend_3gif %s",ptr);
        system(cmd);
        system("killall -34 goahead");  //refresh goahead nvram config
    }
	return CMD_OK;
}

int do_extend_usb(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+12;  // 12 is length of "extend_3gid "

    if(strlen(buff)<=18){        // 20 is length of "extend_3gid /x/x/x"
        printf("#extend_3gid [Devce ID]\n");
        printf("#EX: extend_3gid 1234/5678/0\n");
        ptr=run_command("nvram_get 2860 extend_3gid");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{       //save config
        sprintf(cmd,"nvram_set 2860 extend_3gid %s",ptr);
        system(cmd);
        system("killall -34 goahead");  //refresh goahead nvram config
    }
	return CMD_OK;
}

int do_extend_stor(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+14;  // 12 is length of "extend_storid "

    if(strlen(buff)<=20){        // 20 is length of "extend_storid /x/x/x"
        printf("#extend_storid [Devce ID]\n");
        printf("#EX: extend_storid 1234/5678/0\n");
        ptr=run_command("nvram_get 2860 extend_stor_3gid");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{       //save config
        sprintf(cmd,"nvram_set 2860 extend_stor_3gid %s",ptr);
        system(cmd);
        system("killall -34 goahead");  //refresh goahead nvram config
    }
	return CMD_OK;
}

int do_extend_3gcmd(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+13;  // 12 is length of "extend_3gcmd "

    if(strlen(buff)<=70){         // 70 is length of "extend_3gcmd xxxxxxxxx..."
        printf("#extend_3gcmd [modeswitch command in hex]\n");
        printf("#EX: extend_3gcmd 55534243000000000000000000000011060000000000000000000000000000\n");
        ptr=run_command("nvram_get 2860 extend_3gcmd");
        if(ptr){
            printf("#Current value: %s\n\n",ptr);
            sleep(1);
            free(ptr);
        }else{
            printf("#Current value: \n\n");
        }
    }else{       //save config
        sprintf(cmd,"nvram_set 2860 extend_3gcmd %s",ptr);
        system(cmd);
        system("killall -34 goahead");  //refresh goahead nvram config
    }
	return CMD_OK;
}

int do_wan_phy_mode(char *buff, int adv_cmd)
{
	char cmd[TEL_BUF_LEN]={0};
    char *ptr=buff+8;  // 8 is length of "wan_phy "
    unsigned short value=0;
    unsigned short reg=0;
    unsigned short cfg=0;
    unsigned char *uci1=NULL;
    unsigned char *uci2=NULL;
    if(strlen(buff)<11){        // 11 is length of "wan_phy 10h"
        printf("Command format should be: wan_phy [Type]\n");
        printf("EX: wan_phy 100f\n");
        printf("Type:\n");
        printf("auto: Auto-Negotiation\n");
        printf("100f: 100Mbps, Full Duplex\n");
        printf("100h: 100Mbps, Half Duplex\n");
        printf("10f:  10Mbps, Full Duplex\n");
        printf("10h:  10Mbps, Half Duplex\n");
    }else{      //set wan_phy mode
        //mode
        if(strncmp(ptr,"auto",4)==0){ //auto mode
            value=0x3100;
            cfg=0;
        }else if(strncmp(ptr,"100f",4)==0){ //auto mode
            value=0x2100;
            cfg=1;
        }else if(strncmp(ptr,"100h",4)==0){ //100 half
            value=0x2000;
            cfg=2;
        }else if(strncmp(ptr,"10f",3)==0){ //10 full
            value=0x0100;
            cfg=3;
        }else if(strncmp(ptr,"10h",3)==0){ //10half
            value=0x0000;
            cfg=4;
        }else{  //unknown command
            return CMD_OK;
        }
        //check fiber type
        uci1=run_command("mii_mgr -g -p 17 -r 2 | grep 0243");
        uci2=run_command("mii_mgr -g -p 17 -r 3 | grep 0c54");
        if((uci1 == NULL) ||(uci2 == NULL)){    //check fail
            if(uci1!=NULL)
                free(uci1);
            if(uci2!=NULL)
                free(uci2);
            printf("Fiber check fail\n");
            return CMD_OK;
        }else{
            //register format should be: "Get: phy[17].reg[2] = 0243" or ""
            if((strlen(uci1)!=0) && (strlen(uci2)!=0)){
                reg=17;
                //printf("Fiber type\n");
            }else{
                reg=1;
                //printf("Nornal type\n");
            }
            free(uci1);
            free(uci2);
        }

        //set command
        sprintf(cmd,"mii_mgr -s -p %d -r 0 -v 0x%04x",reg,value);
        system(cmd);
        printf("%s\n",cmd);

        //save config
        sprintf(cmd,"nvram_set 2860 wan_phy %d",cfg);
        system(cmd);

    }
	return CMD_OK;
}

#if CONFIG_USER_IPV6_P1_TEST == 1
int do_ipv6_p1(char *buff, int adv_cmd)
{
	int i;
	char cmd[32];
	char *ptr=buff+8;  // 8 is length of "ipv6_p1 "
	if(strlen(buff)<9 || strlen(buff) > (9+8)){        // 9 is length of "ipv6_p1 0"
		printf("Command format should be: ipv6_p1 [Option]\n");
		printf("EX: ipv6_p1 1\n");
	}else{
		for (i = 0; ptr[i] != 0; i++) {
			if ((ptr[i] < '0' || ptr[i] > '9') && ptr[i] != '.') {
				ptr[i] = 0;
				break;
			}
		}
		snprintf(cmd, sizeof(cmd), "/sbin/ipv6_p1.sh %s", ptr);
		cmd[sizeof(cmd)-1] = 0;
		system(cmd);
	}
	return CMD_OK;
}
#endif

#if CONFIG_USER_CHT_FW

int do_cht(char *buff, int adv_cmd)
{
    int reg_cmd[TEL_BUF_LEN]={0};

	memset(reg_cmd, 0, TEL_BUF_LEN);
	sprintf(reg_cmd, "/sbin/cht_cmd.sh %s", buff);
	system(reg_cmd);
	return CMD_OK;
}

#endif

int do_all_wimax(char *buff, int adv_cmd)
{
	if(strstr(buff, "on")){
		printf("Turn on all wimax support\n");
		system("nvram_set 2860 comstar_wimax 1 > /dev/null 2>&1");
		printf("Switch on usb power again. Please wait!\n");
		system("gpio u");
	}else if(strstr(buff, "off")){
		printf("Turn off comstar wimax support\n");
		system("nvram_set 2860 comstar_wimax 0 > /dev/null 2>&1");
		printf("Switch on usb power again. Please wait!\n");
		system("gpio u");
	}else{
		printf("allwimaxfly200 <on/off>\n");
	}
}



int do_help(char *buff, int adv_cmd);

#define		NORMAL_MODE					0
#define		RD_DEBUG_MODE				1
#define		SOLNET_MODE					2
#define		SUPER_MODE					4 // this command could be used in every mode



/* Command table : Command, function, level */
const cmds_t Cmds[] = {
	{"allwimaxfly200", do_all_wimax, SUPER_MODE}, // Allow comstar wimax dongle
#define	MAX_SECURE_CMD_CNT						1 // setup command count as SECURE_MODE, it would not be displayed in normal mode.
	/* normal command */
    {"help",        do_help,         SUPER_MODE}, // super mode command can be executed in every level
    {"sys ver",		do_exec,		 NORMAL_MODE},
    {"ping",		do_exec,		 NORMAL_MODE},
    {"tr069_client",do_exec,		 NORMAL_MODE},
	{"arp",	        do_arp,	         NORMAL_MODE},
	{"wan_phy",		do_wan_phy_mode, SUPER_MODE},
	{"reboot",	    do_reboot,	     NORMAL_MODE},
#if CUSTOM_DHCP_OPTION == 1
	{"ip dhcpc option",	do_modify_dhcp_option, NORMAL_MODE},
#endif
#if CONFIG_USER_IPV6_P1_TEST == 1
	{"ipv6_p1",		do_ipv6_p1,		SUPER_MODE},
#endif
	//{"hw_nat",	    do_hw_nat,	 NORMAL_MODE},
	{"exit",		do_exit,		 SUPER_MODE},
#if CONFIG_USER_CHT_FW
    {"tracert",		do_exec,		 NORMAL_MODE},
	{"wan",			do_cht,			 NORMAL_MODE},
	{"3gbp",		do_cht,		 	 NORMAL_MODE},
	{"3gbp_log",	do_cht,	 		 NORMAL_MODE},
	{"lbip",		do_cht,		 	 NORMAL_MODE},
#endif

    /* rddebug command */
	{"kill",		do_exec,		 RD_DEBUG_MODE},
	{"killall",		do_exec,		 RD_DEBUG_MODE},
	{"route",		do_exec,		 RD_DEBUG_MODE},
	{"ifconfig",	do_exec,		 RD_DEBUG_MODE},
	{"ps",			do_exec,		 RD_DEBUG_MODE},
	{"iptables",	do_exec,		 RD_DEBUG_MODE},
	{"model",		do_exec,         RD_DEBUG_MODE},
	{"sys eeprom",	do_exec,         RD_DEBUG_MODE},
	{"sys fwup",	do_exec,         RD_DEBUG_MODE},
	{"echo",		do_exec,         RD_DEBUG_MODE},
	{"limitdebug",	do_exec,         RD_DEBUG_MODE},
	{"restartweb",	do_restart_web,  RD_DEBUG_MODE},
	{"switchreg",	do_switch_reg,	 RD_DEBUG_MODE},
	{"3ginfo",		do_3ginfo,		 RD_DEBUG_MODE},
	{"resetusb",	do_reset_usb,	 RD_DEBUG_MODE},
	{"gpio",		do_exec,         RD_DEBUG_MODE},
	{"force_3gif",	do_force_3g,     RD_DEBUG_MODE},
    {"force_3gid",	do_force_usb,    RD_DEBUG_MODE},
    {"force_3gcmd",	do_force_3gcmd,    RD_DEBUG_MODE},
    {"extend_3gif",	do_extend_3g,     RD_DEBUG_MODE},
    {"extend_3gid",	do_extend_usb,    RD_DEBUG_MODE},
    {"extend_storid", do_extend_stor,    RD_DEBUG_MODE},
    {"extend_3gcmd",do_extend_3gcmd,    RD_DEBUG_MODE},
};


int do_help(char *buff, int adv_cmd)
{
	int cmd_count, i, result, j=0;

	cmd_count = sizeof(Cmds) / sizeof(cmds_t);
	//printf("cmd_count = %d\r\n", cmd_count);
	for(i=MAX_SECURE_CMD_CNT; i<cmd_count; i++){
		if((Cmds[i].cmd_lvls == adv_cmd) || (Cmds[i].cmd_lvls == SUPER_MODE)){
			j++;
			printf("%-15s", Cmds[i].name);
			if(j % 4 == 0){
				printf("\n");
			}
		}
	}
	printf("\n");
	return CMD_OK;
}


//this is vigorfly200 telnet command function
static void vigortelnetcmd()
{
    char buff[TEL_BUF_LEN]={0};
    char dump[256]={0};
	struct cmd_result query;
    int adv_cmd=NORMAL_MODE;
	int i, cmd_count, result;
	void *ptr[2];

    printf("\nWelcome to %s!!!\n", DRAYTEK_MODEL_NAME);
    printf("Enter 'help' for a list of built-in commands\n");
	query.dump = &dump;
	ptr[0] = (void *)buff;
	ptr[1] = (void *)&query;

    while(1){
        printf("%s#", DRAYTEK_MODEL_NAME);
        memset(buff,0,TEL_BUF_LEN);
        /*
         * Here we should prevent one special case,
         * If user type CTRL+D (strlen=0), that means "closes the connection and exits",
         * so we check if strlen=0, exit the program.
         *
         * Why i use "fgets()" instead of "gets()" ??
         * Because "gets()" will kill "\n".
         *
         */
        fgets(buff, TEL_BUF_LEN, stdin);
        if(0==strlen(buff)){
            exit(0);
        }
		result = CMD_OK;

		query.level = adv_cmd;
		syscall(__NR_user_config, 13, 5, ptr);
		adv_cmd = query.level;
		if(query.ret == CMD_BREAK){
			break;
		}else if(query.ret == CMD_FAIL){
			if(dump[0] != '\0'){
				printf("%s", dump);
			}

			cmd_count = sizeof(Cmds) / sizeof(cmds_t);
			for(i=0; i<cmd_count; i++){
				if(!strncmp(Cmds[i].name, buff, strlen(Cmds[i].name))){
					/* Command would be executed only when level is correct. Secure command could be executed in every level. */
					if(Cmds[i].cmd_lvls == adv_cmd || Cmds[i].cmd_lvls == SUPER_MODE){
						result = Cmds[i].func(buff, adv_cmd);
						break;
					}
				}
			}

			if(result == CMD_BREAK){
				break;
			}
		}else if(query.ret == CMD_OK){
			if(dump[0] != '\0'){
				printf("%s", dump);
			}
		}

		#if 0	// (Yota special firmware) set WLAN enable minute, nancy
		if (strncmp(buff,"yota special ",13)==0){
			char buf[40];
			int x = atoi(&(buff[13]));
			if (x > 0 && x <= 600) {
				sprintf(buf, "nvram_set 2860 yotaSpecialMinute %d", x);
				system(buf);
				system("reboot");
			}
			continue;
		}
		#endif
    }
}

static void alarm_handler(int sig UNUSED_PARAM)
{
	/* This is the escape hatch!  Poor serial line users and the like
	 * arrive here when their connection is broken.
	 * We don't want to block here */
	ndelay_on(1);
	printf("\r\nLogin timed out after %d seconds\r\n", TIMEOUT);
	fflush(stdout);
	/* unix API is brain damaged regarding O_NONBLOCK,
	 * we should undo it, or else we can affect other processes */
	ndelay_off(1);
	_exit(EXIT_SUCCESS);
}

static int is_a_wan_ip(unsigned long ip)
{
	struct ifreq ifr;
	int skfd = 0;
	unsigned long lan_ip = 0, lan_mask = 0;

	if (!ip)
		return 0;

	if((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return 0;
	strncpy(ifr.ifr_name, "br0", IF_NAMESIZE);
	if (ioctl(skfd, SIOCGIFADDR, &ifr) < 0) {
		close(skfd);
		return 0;
	}
	lan_ip = (unsigned long)(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr);
	if (ioctl(skfd, SIOCGIFNETMASK, &ifr) < 0) {
		close(skfd);
		return 0;
	}
	lan_mask = (unsigned long)(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr);
	close(skfd);

	if ((lan_ip & lan_mask) != (ip & lan_mask))
		return 1;
	return 0;
}

int login_main(int argc, char **argv) MAIN_EXTERNALLY_VISIBLE;
int login_main(int argc UNUSED_PARAM, char **argv)
{
	enum {
		LOGIN_OPT_f = (1<<0),
		LOGIN_OPT_h = (1<<1),
		LOGIN_OPT_p = (1<<2),
	};
	char *fromhost;
	char username[USERNAME_SIZE];
	const char *tmp;
	int amroot;
	unsigned opt;
	int count = 0;
	struct passwd *pw;
	char *opt_host = opt_host; /* for compiler */
	char *opt_user = opt_user; /* for compiler */
	char full_tty[TTYNAME_SIZE];
	char *cmd;
	USE_SELINUX(security_context_t user_sid = NULL;)
	USE_FEATURE_UTMP(struct utmp utent;)
#if ENABLE_PAM
	int pamret;
	pam_handle_t *pamh;
	const char *pamuser;
	const char *failed_msg;
	struct passwd pwdstruct;
	char pwdbuf[256];
#endif
	int isFromWan = 0;

	short_tty = full_tty;
	username[0] = '\0';
	signal(SIGALRM, alarm_handler);
	alarm(TIMEOUT);

	/* More of suid paranoia if called by non-root */
	amroot = !sanitize_env_if_suid(); /* Clear dangerous stuff, set PATH */

	/* Mandatory paranoia for suid applet:
	 * ensure that fd# 0,1,2 are opened (at least to /dev/null)
	 * and any extra open fd's are closed.
	 * (The name of the function is misleading. Not daemonizing here.) */
	bb_daemonize_or_rexec(DAEMON_ONLY_SANITIZE | DAEMON_CLOSE_EXTRA_FDS, NULL);

	opt = getopt32(argv, "f:h:p", &opt_user, &opt_host);
	if (opt & LOGIN_OPT_f) {
		if (!amroot)
			bb_error_msg_and_die("-f is for root only");
		safe_strncpy(username, opt_user, sizeof(username));
	}
	argv += optind;
	if (argv[0]) {
		if (!strncmp(argv[0], "clientIp", 9)) {
			/* client IP */
			if (argv[1])	// check whether the client is from WAN
				isFromWan = is_a_wan_ip((unsigned long)atoi(argv[1]));
		} else {
			/* user from command line (getty) */
			safe_strncpy(username, argv[0], sizeof(username));
		}
	}

	/* Let's find out and memorize our tty */
	if (!isatty(0) || !isatty(1) || !isatty(2))
		return EXIT_FAILURE;		/* Must be a terminal */
	safe_strncpy(full_tty, "UNKNOWN", sizeof(full_tty));
	tmp = ttyname(0);
	if (tmp) {
		safe_strncpy(full_tty, tmp, sizeof(full_tty));
		if (strncmp(full_tty, "/dev/", 5) == 0)
			short_tty = full_tty + 5;
	}

	read_or_build_utent(&utent, !amroot);

	if (opt & LOGIN_OPT_h) {
		USE_FEATURE_UTMP(
			safe_strncpy(utent.ut_host, opt_host, sizeof(utent.ut_host));
		)
		fromhost = xasprintf(" on '%s' from '%s'", short_tty, opt_host);
	} else
		fromhost = xasprintf(" on '%s'", short_tty);

	/* Was breaking "login <username>" from shell command line: */
	/*bb_setpgrp();*/

	openlog(applet_name, LOG_PID | LOG_CONS | LOG_NOWAIT, LOG_AUTH);

	while (1) {
		/* flush away any type-ahead (as getty does) */
		ioctl(0, TCFLSH, TCIFLUSH);

		if (!username[0])
			get_username_or_die(username, sizeof(username));

#if ENABLE_PAM
		pamret = pam_start("login", username, &conv, &pamh);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "start";
			goto pam_auth_failed;
		}
		/* set TTY (so things like securetty work) */
		pamret = pam_set_item(pamh, PAM_TTY, short_tty);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "set_item(TTY)";
			goto pam_auth_failed;
		}
		pamret = pam_authenticate(pamh, 0);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "authenticate";
			goto pam_auth_failed;
			/* TODO: or just "goto auth_failed"
			 * since user seems to enter wrong password
			 * (in this case pamret == 7)
			 */
		}
		/* check that the account is healthy */
		pamret = pam_acct_mgmt(pamh, 0);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "acct_mgmt";
			goto pam_auth_failed;
		}
		/* read user back */
		pamuser = NULL;
		/* gcc: "dereferencing type-punned pointer breaks aliasing rules..."
		 * thus we cast to (void*) */
		if (pam_get_item(pamh, PAM_USER, (void*)&pamuser) != PAM_SUCCESS) {
			failed_msg = "get_item(USER)";
			goto pam_auth_failed;
		}
		if (!pamuser || !pamuser[0])
			goto auth_failed;
		safe_strncpy(username, pamuser, sizeof(username));
		/* Don't use "pw = getpwnam(username);",
		 * PAM is said to be capable of destroying static storage
		 * used by getpwnam(). We are using safe(r) function */
		pw = NULL;
		getpwnam_r(username, &pwdstruct, pwdbuf, sizeof(pwdbuf), &pw);
		if (!pw)
			goto auth_failed;
		pamret = pam_open_session(pamh, 0);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "open_session";
			goto pam_auth_failed;
		}
		pamret = pam_setcred(pamh, PAM_ESTABLISH_CRED);
		if (pamret != PAM_SUCCESS) {
			failed_msg = "setcred";
			goto pam_auth_failed;
		}
		break; /* success, continue login process */

 pam_auth_failed:
		bb_error_msg("pam_%s call failed: %s (%d)", failed_msg,
					pam_strerror(pamh, pamret), pamret);
		safe_strncpy(username, "UNKNOWN", sizeof(username));
#else /* not PAM */
		pw = getpwnam(username);
		if (!pw) {
			strcpy(username, "UNKNOWN");
			goto fake_it;
		}

		if (pw->pw_passwd[0] == '!' || pw->pw_passwd[0] == '*')
			goto auth_failed;

		if (opt & LOGIN_OPT_f)
			break; /* -f USER: success without asking passwd */

		if (pw->pw_uid == 0 && !check_securetty())
			goto auth_failed;

		/* Don't check the password if password entry is empty (!) */
		if (!pw->pw_passwd[0])
			break;
 fake_it:
		/* authorization takes place here */
		if (correct_password(pw))
			break;
#endif /* ENABLE_PAM */
 auth_failed:
		opt &= ~LOGIN_OPT_f;
		bb_do_delay(FAIL_DELAY);
		/* TODO: doesn't sound like correct English phrase to me */
		puts("Login incorrect");
		if (++count == 3) {
			syslog(LOG_WARNING, "invalid password for '%s'%s",
						username, fromhost);
			return EXIT_FAILURE;
		}
		username[0] = '\0';
	}

	alarm(0);
	if (!amroot)
		die_if_nologin();

	write_utent(&utent, username);

#if ENABLE_SELINUX
	if (is_selinux_enabled()) {
		security_context_t old_tty_sid, new_tty_sid;

		if (get_default_context(username, NULL, &user_sid)) {
			bb_error_msg_and_die("cannot get SID for %s",
					username);
		}
		if (getfilecon(full_tty, &old_tty_sid) < 0) {
			bb_perror_msg_and_die("getfilecon(%s) failed",
					full_tty);
		}
		if (security_compute_relabel(user_sid, old_tty_sid,
					SECCLASS_CHR_FILE, &new_tty_sid) != 0) {
			bb_perror_msg_and_die("security_change_sid(%s) failed",
					full_tty);
		}
		if (setfilecon(full_tty, new_tty_sid) != 0) {
			bb_perror_msg_and_die("chsid(%s, %s) failed",
					full_tty, new_tty_sid);
		}
	}
#endif
	/* Try these, but don't complain if they fail.
	 * _f_chown is safe wrt race t=ttyname(0);...;chown(t); */
	fchown(0, pw->pw_uid, pw->pw_gid);
	fchmod(0, 0600);

	/* We trust environment only if we run by root */
	if (ENABLE_LOGIN_SCRIPTS && amroot) {
		char *t_argv[2];

		t_argv[0] = getenv("LOGIN_PRE_SUID_SCRIPT");
		if (t_argv[0]) {
			t_argv[1] = NULL;
			xsetenv("LOGIN_TTY", full_tty);
			xsetenv("LOGIN_USER", pw->pw_name);
			xsetenv("LOGIN_UID", utoa(pw->pw_uid));
			xsetenv("LOGIN_GID", utoa(pw->pw_gid));
			xsetenv("LOGIN_SHELL", pw->pw_shell);
			spawn_and_wait(t_argv); /* NOMMU-friendly */
			unsetenv("LOGIN_TTY"  );
			unsetenv("LOGIN_USER" );
			unsetenv("LOGIN_UID"  );
			unsetenv("LOGIN_GID"  );
			unsetenv("LOGIN_SHELL");
		}
	}

	change_identity(pw);
	tmp = pw->pw_shell;
	if (!tmp || !*tmp)
		tmp = DEFAULT_SHELL;
	/* setup_environment params: shell, clear_env, change_env, pw */
	setup_environment(tmp, !(opt & LOGIN_OPT_p), 1, pw);

	motd();

	if (pw->pw_uid == 0)
		syslog(LOG_INFO, "root login%s", fromhost);
#if ENABLE_SELINUX
	/* well, a simple setexeccon() here would do the job as well,
	 * but let's play the game for now */
	set_current_security_context(user_sid);
#endif

    //run mail alert for telnet login
    cmd = xasprintf("/sbin/mail-alert.sh %s telnet &", pw->pw_name);
    system(cmd);
	free(cmd);

	// util-linux login also does:
	// /* start new session */
	// setsid();
	// /* TIOCSCTTY: steal tty from other process group */
	// if (ioctl(0, TIOCSCTTY, 1)) error_msg...
	// BBox login used to do this (see above):
	// bb_setpgrp();
	// If this stuff is really needed, add it and explain why!

	/* set signals to defaults */
	signal(SIGALRM, SIG_DFL);
	/* Is this correct? This way user can ctrl-c out of /etc/profile,
	 * potentially creating security breach (tested with bash 3.0).
	 * But without this, bash 3.0 will not enable ctrl-c either.
	 * Maybe bash is buggy?
	 * Need to find out what standards say about /bin/login -
	 * should it leave SIGINT etc enabled or disabled? */
	signal(SIGINT, SIG_DFL);
    /* run draytek command first*/
	do{
	    vigortelnetcmd();
	} while(isFromWan);	// if client is from WAN, run draytek command only
	/* Exec login shell with no additional parameters */
	run_shell(tmp, 1, NULL, NULL);

	/* return EXIT_FAILURE; - not reached */
}

