#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include "nvram.h"

#include <linux/autoconf.h>

#define DEFAULT_FLASH_ZONE_NAME "2860"

int set_usage(char *aout)
{
#ifndef CONFIG_RT2860V2_AP_MEMORY_OPTIMIZATION
	int i, num;

	printf("Usage example: \n");
	for (i = 0; i < getNvramNum(); i++){
		printf("\t%s %s ", aout, getNvramName(i));
		printf("lan_ipaddr 1.2.3.4\n");
	}
#endif
	return -1;
}

int ra_nv_set(int argc,char **argv)
{
	int index, rc;
	char *fz, *key, *value;

	if (argc == 1 || argc > 5)
		return set_usage(argv[0]);

	if (argc == 2) {
		fz = DEFAULT_FLASH_ZONE_NAME;
		key = argv[1];
		value = "";
	} else if (argc == 3) {
		fz = DEFAULT_FLASH_ZONE_NAME;
		key = argv[1];
		value = argv[2];
	} else if (argc == 4) {
		fz = argv[1];
		key = argv[2];
		value = argv[3];
	}

	if ((index = getNvramIndex(fz)) == -1) {
		printf("%s: Error: \"%s\" flash zone not existed\n", argv[0], fz);
		return set_usage(argv[0]);
	}

	nvram_init(index);
	rc = nvram_set(index, key, value);
	nvram_close(index);
	return rc;
}

int get_usage(char *aout)
{
#ifndef CONFIG_RT2860V2_AP_MEMORY_OPTIMIZATION
	int i;

	printf("Usage: \n");
	for (i = 0; i < getNvramNum(); i++){
		printf("\t%s %s ", aout, getNvramName(i));
		printf("lan_ipaddr\n");
	}
#endif
	return -1;
}

int ra_nv_get(int argc, char *argv[])
{
	char *fz;
	char *key;
	char *value;

	int index;

	if (argc != 3 && argc != 2)
		return get_usage(argv[0]);

	if (argc == 2) {
		fz = DEFAULT_FLASH_ZONE_NAME;
		key = argv[1];
	} else {
		fz = argv[1];
		key = argv[2];
	}

	if ((index = getNvramIndex(fz)) == -1) {
		printf("%s: Error: \"%s\" flash zone not existed\n", argv[0], fz);
		return get_usage(argv[0]);
	}

	nvram_init(index);
	printf("%s\n", nvram_bufget(index, key));
	nvram_close(index);
	return 0;
}

void usage(char *cmd)
{
#ifndef CONFIG_RT2860V2_AP_MEMORY_OPTIMIZATION
	printf("Usage:\n");
	printf("  %s <command> [<platform>] [<file>]\n\n", cmd);
	printf("command:\n");
	printf("  rt2860_nvram_show - display rt2860 values in nvram\n");
	printf("  inic_nvram_show   - display inic values in nvram\n");
#ifdef CONFIG_DUAL_IMAGE
	printf("  uboot_nvram_show - display uboot parameter values\n");
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	printf("  rt2561_nvram_show - display rt2561 values in nvram\n");
#endif
	printf("  getmac  - display wan or lan mac address\n");	// nancy
	printf("  show    - display values in nvram for <platform>\n");
	printf("  gen     - generate config file from nvram for <platform>\n");
	printf("  renew   - replace nvram values for <platform> with <file>\n");
	printf("  clear	  - clear all entries in nvram for <platform>\n");
	printf("platform:\n");
	printf("  2860    - rt2860\n");
	printf("  inic    - intelligent nic\n");
#ifdef CONFIG_DUAL_IMAGE
	printf("  uboot    - uboot parameter\n");
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	printf("  2561    - rt2561\n");
#endif
	printf("file:\n");
	printf("          - file name for renew command\n");
#endif
	exit(0);
}

int main(int argc, char *argv[])
{
	char *cmd;

	if (argc < 2)
		usage(argv[0]);

    nvram_lock_init();

	//call nvram_get or nvram_set
	if ((cmd = strrchr(argv[0], '/')) != NULL)
		cmd++;
	else
		cmd = argv[0];
	if (!strncmp(argv[0], "nvram_get", 10))
		return ra_nv_get(argc, argv);
	else if (!strncmp(cmd, "nvram_set", 10))
		return ra_nv_set(argc, argv);

	if (argc == 2) {
		if (!strncmp(argv[1], "rt2860_nvram_show", 18))
			nvram_show(RT2860_NVRAM);
#if defined (CONFIG_RTDEV) || \
	defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE) || \
	defined (CONFIG_RTDEV_PLC)
		else if (!strncmp(argv[1], "rtdev_nvram_show", 17))
			nvram_show(RTDEV_NVRAM);
#endif
		else if (!strncmp(argv[1], "inic_nvram_show", 16))
			nvram_show(RTINIC_NVRAM);
#ifdef CONFIG_DUAL_IMAGE
		else if (!strncmp(argv[1], "uboot_nvram_show", 17))
			nvram_show(UBOOT_NVRAM);
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
		else if (!strncmp(argv[1], "rt2561_nvram_show", 18))
			nvram_show(RT2561_NVRAM);
#endif
		else
			usage(argv[0]);
	} else if (argc == 3) {
		/* TODO: <cmd> gen 2860ap */
		if (!strncasecmp(argv[1], "gen", 4) ||
		    !strncasecmp(argv[1], "make_wireless_config", 21)) {
			if (!strncmp(argv[2], "2860", 5) ||
			    !strncasecmp(argv[2], "rt2860", 7)) //b-compatible
				gen_config(RT2860_NVRAM);
#if defined (CONFIG_RTDEV) || \
	defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE) || \
	defined (CONFIG_RTDEV_PLC)
			else if (!strncasecmp(argv[2], "rtdev", 6))
				gen_config(RTDEV_NVRAM);
#endif
#if CONFIG_RT3572_AP_MODULE == 1
			else if (!strncmp(argv[2], "3572", 5) ||!strncasecmp(argv[2], "rt3572", 7))
				gen_config(RT3572_NVRAM);
#endif
			else if (!strncasecmp(argv[2], "inic", 5))
				gen_config(RTINIC_NVRAM);
#ifdef CONFIG_DUAL_IMAGE
			else if (!strncasecmp(argv[2], "uboot", 6))
				printf("No support of gen command of uboot parameter.\n");
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
			else if (!strncmp(argv[2], "2561", 5) ||
			    	 !strncasecmp(argv[2], "rt2561", 7))
				gen_config(RT2561_NVRAM);
#endif
			else
				usage(argv[0]);
		} else if (!strncasecmp(argv[1], "show", 5)) {
			if (!strncmp(argv[2], "2860", 5) ||
			    !strncasecmp(argv[2], "rt2860", 7)) //b-compatible
				nvram_show(RT2860_NVRAM);
#if defined (CONFIG_RTDEV) || \
	defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE) || \
	defined (CONFIG_RTDEV_PLC)
			else if (!strncasecmp(argv[2], "rtdev", 6))
				nvram_show(RTDEV_NVRAM);
#endif
			else if (!strncasecmp(argv[2], "inic", 5))
				nvram_show(RTINIC_NVRAM);
#ifdef CONFIG_DUAL_IMAGE
			else if (!strncasecmp(argv[2], "uboot", 6))
				nvram_show(UBOOT_NVRAM);
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
			else if (!strncmp(argv[2], "2561", 5) ||
			    	 !strncasecmp(argv[2], "rt2561", 7))
				nvram_show(RT2561_NVRAM);
#endif
			else
				usage(argv[0]);
		} else if(!strncasecmp(argv[1], "clear", 6)) {
			if (!strncmp(argv[2], "2860", 5) ||
			    !strncasecmp(argv[2], "rt2860", 7)) //b-compatible
				nvram_clear(RT2860_NVRAM);
#if defined (CONFIG_RTDEV) || \
	defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE) || \
	defined (CONFIG_RTDEV_PLC)
			else if (!strncasecmp(argv[2], "rtdev", 6))
				nvram_clear(RTDEV_NVRAM);
#endif
			else if (!strncasecmp(argv[2], "inic", 5))
				nvram_clear(RTINIC_NVRAM);
#ifdef CONFIG_DUAL_IMAGE
			else if (!strncasecmp(argv[2], "uboot", 6))
				nvram_clear(UBOOT_NVRAM);
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
			else if (!strncmp(argv[2], "2561", 5) ||
				 !strncasecmp(argv[2], "rt2561", 7))
				nvram_clear(RT2561_NVRAM);
#endif
			else
				usage(argv[0]);
		} else if (!strncmp(argv[1], "getmac", 6)) {	// nancy
			if (!strncmp(argv[2], "lan", 3))
				nvram_getmac(1);
			else
				nvram_getmac(0);
		} else
			usage(argv[0]);
	} else if (argc == 4) {
		if (!strncasecmp(argv[1], "renew", 6)) {
			if (!strncmp(argv[2], "2860", 5) ||
			    !strncasecmp(argv[2], "rt2860", 7)) //b-compatible
				renew_nvram(RT2860_NVRAM, argv[3]);
#if defined (CONFIG_RTDEV) || \
	defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE) || \
	defined (CONFIG_RTDEV_PLC)
			else if (!strncasecmp(argv[2], "rtdev", 6))
				renew_nvram(RTDEV_NVRAM, argv[3]);
#endif
			else if (!strncasecmp(argv[2], "inic", 5))
				renew_nvram(RTINIC_NVRAM, argv[3]);
#ifdef CONFIG_DUAL_IMAGE
			else if (!strncasecmp(argv[2], "uboot", 6))
				printf("No support of renew command of uboot parameter.\n");
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
			else if (!strncmp(argv[2], "2561", 5) ||
			    	 !strncasecmp(argv[2], "rt2561", 7))
				renew_nvram(RT2561_NVRAM, argv[3]);
#endif
		} else
			usage(argv[0]);
	} else
		usage(argv[0]);
	return 0;
}

#define DHCP_RELAY_MODE         2
#define LAN_A                   1
#define LAN_B                   2
#define AP_P2P                  1
#define AP_P2MP                 2
#define AP_MODE                 0

#define MAC_ADDR_LEN            6

int getLanMac(char *if_hw)
{
    int ret = 0;
    unsigned char mac[MAC_ADDR_LEN] = {0};

    ret = flash_read_mac(mac); // LAN MAC
    if (ret == MAC_ADDR_LEN) {
        //mac[5]-=2; // set to LAN MAC, Modify flash_read_mac() to read LAN MAC
        sprintf(if_hw, "%02x:%02x:%02x:%02x:%02x:%02x", mac[0]&0xff, mac[1]&0xff, mac[2]&0xff, mac[3]&0xff, mac[4]&0xff, mac[5]&0xff);
        //fprintf(stderr, "===>getLanMac: %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0]&0xff, mac[1]&0xff, mac[2]&0xff, mac[3]&0xff, mac[4]&0xff, mac[5]&0xff);
    }
    return ret;
}

void gen_dhcp_relay_config_by_ssid(int nvram_mode, char* buf, int ssid_num, int vlan_idx, char* src, char* mac)
{
    char cfg_name[16] = {0};
    int i, vlan, dhcpEnabled, dhcp2Enabled, ap_mode, wl_vlan;
    char hw_addr[18] = {0};

    dhcpEnabled = atoi(nvram_get(nvram_mode, "dhcpEnabled"));
    dhcp2Enabled = atoi(nvram_get(nvram_mode, "dhcp2Enabled"));
    ap_mode = atoi(nvram_get(nvram_mode, "800_ap_mode"));
    wl_vlan = atoi(nvram_get(nvram_mode, "WL_VLAN"));

    /* Get LANA & LANB hardware address. */
    if (ap_mode == AP_MODE && (dhcpEnabled == DHCP_RELAY_MODE || dhcp2Enabled == DHCP_RELAY_MODE)) {
        getLanMac(hw_addr);
    }

    for (i = 0; i < ssid_num; i++) {
        if (wl_vlan == 0) {
             if (dhcpEnabled == DHCP_RELAY_MODE) {
                sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcpRelay"));
                if (ap_mode == AP_MODE) {
                    sprintf(src+strlen(src), "%s;", nvram_bufget(nvram_mode, "lan_ipaddr"));
                    sprintf(mac+strlen(mac), "%s;", hw_addr);
                    continue;
                }
             } else {
                sprintf(buf+strlen(buf), ";");
             }
             sprintf(src+strlen(src), ";");
             sprintf(mac+strlen(mac), ";");
        } else {
            sprintf(cfg_name, "SSID_VLAN%d", (i + vlan_idx + 1));
            vlan = atoi(nvram_get(nvram_mode, cfg_name));
            switch (ap_mode) {
                case AP_P2P:
                case AP_P2MP:
                    if (dhcpEnabled == DHCP_RELAY_MODE) {
                        sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcpRelay"));
                    } else {
                        sprintf(buf+strlen(buf), ";");
                    }
                    sprintf(src+strlen(src), ";");
                    sprintf(mac+strlen(mac), ";");
                    break;

                default:
                    if (vlan == LAN_A && dhcpEnabled == DHCP_RELAY_MODE) {
                        sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcpRelay"));
                        if (ap_mode == AP_MODE) {
                            sprintf(src+strlen(src), "%s;", nvram_bufget(nvram_mode, "lan_ipaddr"));
                            sprintf(mac+strlen(mac), "%s;", hw_addr);
                            continue;
                        }
                    } else if (vlan == LAN_B && dhcp2Enabled == DHCP_RELAY_MODE) {
                        sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcp2Relay"));
                        if (ap_mode == AP_MODE) {
                            sprintf(src+strlen(src), "%s;", nvram_bufget(nvram_mode, "lan2_ipaddr"));
                            sprintf(mac+strlen(mac), "%s;", hw_addr);
                            continue;
                        }
                    } else {
                        sprintf(buf+strlen(buf), ";");
                    }
                    sprintf(src+strlen(src), ";");
                    sprintf(mac+strlen(mac), ";");
                    break;
            }
        }
    }
}

void gen_dhcp_relay_config_by_lan(int nvram_mode, char* buf)
{
     int dhcpEnabled, dhcp2Enabled, ap_mode;

     ap_mode = atoi(nvram_get(nvram_mode, "800_ap_mode"));

     dhcpEnabled = atoi(nvram_get(nvram_mode, "dhcpEnabled"));
     if ((dhcpEnabled == DHCP_RELAY_MODE) && (ap_mode != AP_MODE)) {
        sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcpRelay"));
     } else {
        sprintf(buf+strlen(buf), ";");
     }

     dhcp2Enabled = atoi(nvram_get(nvram_mode, "dhcp2Enabled"));
     if ((dhcp2Enabled == DHCP_RELAY_MODE) && (ap_mode != AP_MODE)) {
        sprintf(buf+strlen(buf), "%s;", nvram_bufget(nvram_mode, "dhcp2Relay"));
     } else {
        sprintf(buf+strlen(buf), ";");
     }
}

#ifdef ACL_256_SUPPORT
int save_acl_to_file(int mode, int idx, char* name, char* val)
{
    FILE *fp;
    char path[128] = {0};

    if (mode == RT2860_NVRAM) {
        sprintf(path,"/etc/Wireless/RT2860/RT2860_ACL%d", idx);
		fp = fopen(path, "w+");
#if CONFIG_RT3572_AP_MODULE == 1
	} else if (mode == RT3572_NVRAM) {
	    sprintf(path,"/etc/Wireless/RT2870AP/RT2870AP_ACL%d", idx);
		fp = fopen(path, "w+");
#endif
    } else
		return 0;

    if (fp) {
        fprintf(fp, "#The word of \"Default\" must not be removed\n");
    	fprintf(fp, "Default\n");
        fprintf(fp, "%s=%s\n", name, nvram_bufget(RT2860_NVRAM, val));
        fclose(fp);
    }
}
#endif

char dhcpr_ssid[128], dhcpr_lan[64], dhcpr_ssid_src[128], dhcpr_ssid_mac[128];
int gen_config(int mode)
{
	FILE *fp, *fp1;
	int  i, ssid_num = 1;
	char tx_rate[16], wmm_enable[16];
	int wmm = 0, op_mode = 0;


	nvram_init(mode);

	/*
	nvram_bufset(mode, "SystemName", "RalinkAP");
	nvram_bufset(mode, "ModuleName", "RT2860");
	nvram_commit(mode);
	*/
	unsigned char temp[2], buf[20];
    int ret=0;
	flash_read_NicConf(buf);

    //check if we need to set the value, we don't want to set these values every time
    //P.S. these values is related with wireless function.      //Bruce Hsu20091201
    //printf("Check0: %x, %x\n", buf[1], atoi(nvram_bufget(mode,"RFICType")));
    if (buf[1] != strtoul(nvram_bufget(mode,"RFICType"), 0, 16)) {
        sprintf(temp, "%x", buf[1]);
        //printf("Set RFICType: %s\n", temp);
	    nvram_bufset(mode, "RFICType", temp);
        ret=1;
    }
    //printf("Check1: %x, %x\n", (buf[1]&0xf0>>4), atoi(nvram_bufget(mode,"TXPath")));
    if ((buf[1]&0xf0>>4) != strtoul(nvram_bufget(mode,"TXPath"), 0, 16)) {
        sprintf(temp, "%x", buf[1]&0xf0>>4);
        //printf("Set TXPath: %s\n", temp);
	    nvram_bufset(mode, "TXPath", temp);
        ret=1;
    }
    //printf("Check2: %x, %x\n", (buf[0]&0x0f), atoi(nvram_bufget(mode,"RXPath")));
    if ((buf[0]&0x0f) != strtoul(nvram_bufget(mode,"RXPath"), 0, 16)) {
	    sprintf(temp, "%x", buf[0]&0x0f);
        //printf("Set RXPath: %s\n", temp);
	    nvram_bufset(mode, "RXPath", temp);
        ret=1;
    }
    if(ret != 0){
        nvram_commit(mode);
    }

	system("mkdir -p /etc/Wireless/RT2860");
	if (mode == RT2860_NVRAM) {
		fp = fopen("/etc/Wireless/RT2860/RT2860.dat", "w+");
#if CONFIG_RT3572_AP_MODULE == 1
	} else if (mode == RT3572_NVRAM) {
		system("mkdir -p /etc/Wireless/RT2870AP");
		fp = fopen("/etc/Wireless/RT2870AP/RT2870AP.dat", "w+");
#endif
//	} else if (mode == RTINIC_NVRAM) { // FIXME James
//		system("mkdir -p /etc/Wireless/iNIC");
//		fp = fopen("/etc/Wireless/iNIC/iNIC_ap.dat", "w+");
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	} else if (mode == RT2561_NVRAM) {
		system("mkdir -p /etc/Wireless/RT2561");
		fp = fopen("/etc/Wireless/RT2561/RT2561.dat", "w+");
#endif
	} else
		return 0;

	fprintf(fp, "#The word of \"Default\" must not be removed\n");
	fprintf(fp, "Default\n");

#define FPRINT_NUM(x) fprintf(fp, #x"=%d\n", atoi(nvram_bufget(mode, #x)));
#define FPRINT_STR(x) fprintf(fp, #x"=%s\n", nvram_bufget(mode, #x));

	if ((RT2860_NVRAM == mode) //|| (RTINIC_NVRAM == mode) // FIXME James
//#if defined (CONFIG_RTDEV)
//		|| (RTDEV_NVRAM == mode)
//#endif
		) {
		FPRINT_NUM(ClientLimit_Enable);
		FPRINT_NUM(ClientLimit);
		FPRINT_NUM(TrafficLimit_Enable);
		FPRINT_NUM(UploadLimit);
		FPRINT_NUM(DownloadLimit);
		FPRINT_NUM(IdleTime);
		FPRINT_NUM(Rssi);
		FPRINT_NUM(CountryRegion);
		FPRINT_NUM(CountryRegionABand);
		FPRINT_STR(CountryCode);
		FPRINT_NUM(BssidNum);
		ssid_num = atoi(nvram_get(mode, "BssidNum"));

		FPRINT_STR(SSID1);
		FPRINT_STR(SSID2);
		FPRINT_STR(SSID3);
		FPRINT_STR(SSID4);
		FPRINT_STR(SSID5);
		FPRINT_STR(SSID6);
		FPRINT_STR(SSID7);
		FPRINT_STR(SSID8);

		FPRINT_NUM(WirelessMode);
		FPRINT_STR(FixedTxMode);

		//TxRate(FixedRate)
		bzero(tx_rate, sizeof(char)*16);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(tx_rate+strlen(tx_rate), "%d",
					atoi(nvram_bufget(mode, "TxRate")));
			sprintf(tx_rate+strlen(tx_rate), "%c", ';');
		}
		tx_rate[strlen(tx_rate) - 1] = '\0';
		fprintf(fp, "TxRate=%s\n", tx_rate);

		FPRINT_NUM(Channel);
		FPRINT_NUM(BasicRate);
		FPRINT_NUM(BeaconPeriod);
		FPRINT_NUM(DtimPeriod);
		FPRINT_NUM(TxPower);
		FPRINT_NUM(DisableOLBC);
		FPRINT_NUM(BGProtection);
		fprintf(fp, "TxAntenna=\n");
		fprintf(fp, "RxAntenna=\n");
		FPRINT_NUM(TxPreamble);
		FPRINT_NUM(RTSThreshold  );
		FPRINT_NUM(FragThreshold  );

		wmm = atoi(nvram_bufget(mode, "WmmCapable"));
		if (wmm)
			fprintf(fp, "TxBurst=0\n");
		else
			FPRINT_NUM(TxBurst);

		FPRINT_NUM(PktAggregate);
		FPRINT_NUM(AutoProvisionEn);
		FPRINT_NUM(FreqDelta);
		fprintf(fp, "TurboRate=0\n");

		//WmmCapable
		bzero(wmm_enable, sizeof(char)*16);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(wmm_enable+strlen(wmm_enable), "%d",
					atoi(nvram_bufget(mode, "WmmCapable")));
			sprintf(wmm_enable+strlen(wmm_enable), "%c", ';');
		}
		wmm_enable[strlen(wmm_enable) - 1] = '\0';
		fprintf(fp, "WmmCapable=%s\n", wmm_enable);

		FPRINT_STR(APAifsn);
		FPRINT_STR(APCwmin);
		FPRINT_STR(APCwmax);
		FPRINT_STR(APTxop);
		FPRINT_STR(APACM);
		FPRINT_STR(BSSAifsn);
		FPRINT_STR(BSSCwmin);
		FPRINT_STR(BSSCwmax);
		FPRINT_STR(BSSTxop);
		FPRINT_STR(BSSACM);
		FPRINT_STR(AckPolicy);
		FPRINT_STR(APSDCapable);
		FPRINT_STR(DLSCapable);
		FPRINT_STR(NoForwarding);
		//Enable default no forwarding between MBSSIDs. --Albert
		//FPRINT_NUM(NoForwardingBTNBSSID);
		fprintf(fp, "NoForwardingBTNBSSID=1\n");

		FPRINT_STR(HideSSID);
		FPRINT_NUM(ShortSlot);
		FPRINT_NUM(AutoChannelSelect);

		FPRINT_STR(IEEE8021X);
		//FPRINT_NUM(IEEE80211H);
		// James.Lin: change for DFS function
		fprintf(fp, "IEEE80211H=1\n");

		FPRINT_NUM(ITxBfEn);
		FPRINT_STR(PreAntSwitch);
		FPRINT_NUM(PhyRateLimit);
		FPRINT_NUM(DebugFlags);
		FPRINT_NUM(ETxBfEnCond);
		FPRINT_NUM(ITxBfTimeout);
		FPRINT_NUM(ETxBfTimeout);
		FPRINT_NUM(ETxBfNoncompress);
		FPRINT_NUM(ETxBfIncapable);
		FPRINT_NUM(FineAGC);
		FPRINT_NUM(StreamMode);
		FPRINT_STR(StreamModeMac0);
		FPRINT_STR(StreamModeMac1);
		FPRINT_STR(StreamModeMac2);
		FPRINT_STR(StreamModeMac3);


		FPRINT_NUM(CSPeriod);
		//FPRINT_STR(RDRegion);
		// James.Lin: change for DFS function
		fprintf(fp, "RDRegion=CE\n");
		FPRINT_NUM(StationKeepAlive);
		FPRINT_NUM(DfsLowerLimit);
		FPRINT_NUM(DfsUpperLimit);
		FPRINT_NUM(DfsOutdoor);
		FPRINT_NUM(SymRoundFromCfg);
        FPRINT_NUM(BusyIdleFromCfg);
        FPRINT_NUM(DfsRssiHighFromCfg);
        FPRINT_NUM(DfsRssiLowFromCfg);
		FPRINT_NUM(DFSParamFromConfig);
		FPRINT_STR(FCCParamCh0);
		FPRINT_STR(FCCParamCh1);
		FPRINT_STR(FCCParamCh2);
		FPRINT_STR(FCCParamCh3);
		FPRINT_STR(CEParamCh0);
		FPRINT_STR(CEParamCh1);
		FPRINT_STR(CEParamCh2);
		FPRINT_STR(CEParamCh3);
		FPRINT_STR(JAPParamCh0);
		FPRINT_STR(JAPParamCh1);
		FPRINT_STR(JAPParamCh2);
		FPRINT_STR(JAPParamCh3);
		FPRINT_STR(JAPW53ParamCh0);
		FPRINT_STR(JAPW53ParamCh1);
		FPRINT_STR(JAPW53ParamCh2);
		FPRINT_STR(JAPW53ParamCh3);
		FPRINT_NUM(FixDfsLimit);
		FPRINT_NUM(LongPulseRadarTh);
		FPRINT_NUM(AvgRssiReq);
		FPRINT_NUM(DFS_R66);
		FPRINT_STR(BlockCh);

		FPRINT_NUM(GreenAP);
		FPRINT_STR(PreAuth);
		FPRINT_STR(AuthMode);
		FPRINT_STR(EncrypType);
        /*kurtis: WAPI*/
		FPRINT_STR(WapiPsk1);
		FPRINT_STR(WapiPsk2);
		FPRINT_STR(WapiPsk3);
		FPRINT_STR(WapiPsk4);
		FPRINT_STR(WapiPsk5);
		FPRINT_STR(WapiPsk6);
		FPRINT_STR(WapiPsk7);
		FPRINT_STR(WapiPsk8);
		FPRINT_STR(WapiPskType);
		FPRINT_STR(Wapiifname);
		FPRINT_STR(WapiAsCertPath);
		FPRINT_STR(WapiUserCertPath);
		FPRINT_STR(WapiAsIpAddr);
		FPRINT_STR(WapiAsPort);

		FPRINT_NUM(BssidNum);

		FPRINT_STR(RekeyMethod);
		FPRINT_NUM(RekeyInterval);
		FPRINT_STR(PMKCachePeriod);

		FPRINT_NUM(MeshAutoLink);
		FPRINT_STR(MeshAuthMode);
		FPRINT_STR(MeshEncrypType);
		FPRINT_NUM(MeshDefaultkey);
		FPRINT_STR(MeshWEPKEY);
		FPRINT_STR(MeshWPAKEY);
		FPRINT_STR(MeshId);

		//WPAPSK
		FPRINT_STR(WPAPSK1);
		FPRINT_STR(WPAPSK2);
		FPRINT_STR(WPAPSK3);
		FPRINT_STR(WPAPSK4);
		FPRINT_STR(WPAPSK5);
		FPRINT_STR(WPAPSK6);
		FPRINT_STR(WPAPSK7);
		FPRINT_STR(WPAPSK8);

		FPRINT_STR(DefaultKeyID);
		FPRINT_STR(Key1Type);
		FPRINT_STR(Key1Str1);
		FPRINT_STR(Key1Str2);
		FPRINT_STR(Key1Str3);
		FPRINT_STR(Key1Str4);
		FPRINT_STR(Key1Str5);
		FPRINT_STR(Key1Str6);
		FPRINT_STR(Key1Str7);
		FPRINT_STR(Key1Str8);

		FPRINT_STR(Key2Type);
		FPRINT_STR(Key2Str1);
		FPRINT_STR(Key2Str2);
		FPRINT_STR(Key2Str3);
		FPRINT_STR(Key2Str4);
		FPRINT_STR(Key2Str5);
		FPRINT_STR(Key2Str6);
		FPRINT_STR(Key2Str7);
		FPRINT_STR(Key2Str8);

		FPRINT_STR(Key3Type);
		FPRINT_STR(Key3Str1);
		FPRINT_STR(Key3Str2);
		FPRINT_STR(Key3Str3);
		FPRINT_STR(Key3Str4);
		FPRINT_STR(Key3Str5);
		FPRINT_STR(Key3Str6);
		FPRINT_STR(Key3Str7);
		FPRINT_STR(Key3Str8);

		FPRINT_STR(Key4Type);
		FPRINT_STR(Key4Str1);
		FPRINT_STR(Key4Str2);
		FPRINT_STR(Key4Str3);
		FPRINT_STR(Key4Str4);
		FPRINT_STR(Key4Str5);
		FPRINT_STR(Key4Str6);
		FPRINT_STR(Key4Str7);
		FPRINT_STR(Key4Str8);

		FPRINT_NUM(HSCounter);

		FPRINT_NUM(HT_HTC);
		if (wmm)
			fprintf(fp, "HT_RDG=0\n");
		else
			FPRINT_NUM(HT_RDG);
		FPRINT_NUM(HT_LinkAdapt);
		FPRINT_NUM(HT_OpMode);
		FPRINT_NUM(HT_MpduDensity);
		FPRINT_NUM(HT_EXTCHA);
		FPRINT_NUM(HT_BW);
		FPRINT_NUM(VHT_BW);
		FPRINT_NUM(HT_AutoBA);
		FPRINT_NUM(HT_BADecline);
		FPRINT_NUM(HT_AMSDU);
		FPRINT_NUM(HT_BAWinSize);
		FPRINT_NUM(HT_GI);
		FPRINT_NUM(HT_STBC);
		//FPRINT_NUM(HT_MCS);
		FPRINT_STR(HT_MCS);
		FPRINT_NUM(HT_TxStream);
		FPRINT_NUM(HT_RxStream);
		FPRINT_NUM(HT_PROTECT);
		FPRINT_NUM(HT_DisallowTKIP);
		FPRINT_NUM(HT_BSSCoexistence);
		FPRINT_NUM(GreenAP);

		FPRINT_NUM(WscConfMode);

		//WscConfStatus
		if (atoi(nvram_bufget(mode, "WscConfigured")) == 0)
			fprintf(fp, "WscConfStatus=%d\n", 1);
		else
			fprintf(fp, "WscConfStatus=%d\n", 2);
		//if (strcmp(nvram_bufget(mode, "WscVendorPinCode"), "") != 0)
			//FPRINT_STR(WscVendorPinCode);
		FPRINT_NUM(WCNTest);

        #ifdef ACL_256_SUPPORT
		FPRINT_NUM(AccessPolicy0);
		save_acl_to_file(mode, 0, "AccessControlList0", "AccessControlList0");
		FPRINT_NUM(AccessPolicy1);
		save_acl_to_file(mode, 1, "AccessControlList1", "AccessControlList1");
		FPRINT_NUM(AccessPolicy2);
		save_acl_to_file(mode, 2, "AccessControlList2", "AccessControlList2");
		FPRINT_NUM(AccessPolicy3);
		save_acl_to_file(mode, 3, "AccessControlList3", "AccessControlList3");
		FPRINT_NUM(AccessPolicy4);
		save_acl_to_file(mode, 4, "AccessControlList4", "AccessControlList4");
		FPRINT_NUM(AccessPolicy5);
		save_acl_to_file(mode, 5, "AccessControlList5", "AccessControlList5");
		FPRINT_NUM(AccessPolicy6);
		save_acl_to_file(mode, 6, "AccessControlList6", "AccessControlList6");
		FPRINT_NUM(AccessPolicy7);
		save_acl_to_file(mode, 7, "AccessControlList7", "AccessControlList7");
        #else
		FPRINT_NUM(AccessPolicy0);
		FPRINT_STR(AccessControlList0);
		FPRINT_NUM(AccessPolicy1);
		FPRINT_STR(AccessControlList1);
		FPRINT_NUM(AccessPolicy2);
		FPRINT_STR(AccessControlList2);
		FPRINT_NUM(AccessPolicy3);
		FPRINT_STR(AccessControlList3);
		FPRINT_NUM(AccessPolicy4);
		FPRINT_STR(AccessControlList4);
		FPRINT_NUM(AccessPolicy5);
		FPRINT_STR(AccessControlList5);
		FPRINT_NUM(AccessPolicy6);
		FPRINT_STR(AccessControlList6);
		FPRINT_NUM(AccessPolicy7);
		FPRINT_STR(AccessControlList7);
        #endif

		FPRINT_NUM(WdsEnable);
		FPRINT_STR(WdsPhyMode);
		FPRINT_STR(WdsEncrypType);
		FPRINT_STR(WdsList);
		FPRINT_STR(WdsKey);
		FPRINT_STR(Wds0Key);
		FPRINT_STR(Wds1Key);
		FPRINT_STR(Wds2Key);
		FPRINT_STR(Wds3Key);
		FPRINT_STR(RADIUS_Server);
		FPRINT_STR(RADIUS_Port);
		//FPRINT_STR(RADIUS_Key);
		FPRINT_STR(RADIUS_Key1);
		FPRINT_STR(RADIUS_Key2);
		FPRINT_STR(RADIUS_Key3);
		FPRINT_STR(RADIUS_Key4);
		FPRINT_STR(RADIUS_Key5);
		FPRINT_STR(RADIUS_Key6);
		FPRINT_STR(RADIUS_Key7);
		FPRINT_STR(RADIUS_Key8);
		FPRINT_STR(RADIUS_Acct_Server);
		FPRINT_NUM(RADIUS_Acct_Port);
		FPRINT_STR(RADIUS_Acct_Key);
		FPRINT_STR(own_ip_addr);
		FPRINT_STR(Ethifname);
		FPRINT_STR(EAPifname);
		FPRINT_STR(PreAuthifname);
		FPRINT_NUM(session_timeout_interval);
		FPRINT_NUM(idle_timeout_interval);
		FPRINT_NUM(WiFiTest);
		FPRINT_NUM(TGnWifiTest);

		//AP Client parameters
		FPRINT_NUM(ApCliEnable);
		FPRINT_STR(ApCliSsid);
		FPRINT_STR(ApCliBssid);
		FPRINT_STR(ApCliAuthMode);
		FPRINT_STR(ApCliEncrypType);
		FPRINT_STR(ApCliWPAPSK);
		//FPRINT_NUM(ApCliDefaultKeyId);
		FPRINT_NUM(ApCliDefaultKeyID);
		FPRINT_NUM(ApCliKey1Type);
		FPRINT_STR(ApCliKey1Str);
		FPRINT_NUM(ApCliKey2Type);
		FPRINT_STR(ApCliKey2Str);
		FPRINT_NUM(ApCliKey3Type);
		FPRINT_STR(ApCliKey3Str);
		FPRINT_NUM(ApCliKey4Type);
		FPRINT_STR(ApCliKey4Str);
		//FPRINT_NUM(TempComp);
		//FPRINT_NUM(EfuseBufferMode);
		//FPRINT_NUM(E2pAccessMode);

		//Radio On/Off
		if (atoi(nvram_bufget(mode, "RadioOff")) == 1)
			fprintf(fp, "RadioOn=0\n");
		else
			fprintf(fp, "RadioOn=1\n");

		//Wireless Roaming On/Off
		FPRINT_NUM(WirelessRoaming_en);
		FPRINT_STR(STA_CTL);
		
		//EDCCA Support
		FPRINT_NUM(EDCCA_AP_STA_TH);
		FPRINT_NUM(EDCCA_AP_AP_TH);
		FPRINT_NUM(EDCCA_AP_RSSI_TH);
		FPRINT_NUM(EDCCA_ED_TH);
		FPRINT_NUM(EDCCA_FALSE_CCA_TH);
		FPRINT_NUM(EDCCA_BLOCK_CHECK_TH);

        //DHCP RELAY
        bzero(dhcpr_ssid, sizeof(char)*128);
        bzero(dhcpr_ssid_src, sizeof(char)*128);
        bzero(dhcpr_ssid_mac, sizeof(char)*128);
        bzero(dhcpr_lan, sizeof(char)*64);
        op_mode = atoi(nvram_get(mode, "OperationMode"));
        if (op_mode == 0) { // AP mode
            gen_dhcp_relay_config_by_ssid(mode, dhcpr_ssid, ssid_num, 0, dhcpr_ssid_src, dhcpr_ssid_mac);
            gen_dhcp_relay_config_by_lan(mode, dhcpr_lan);
        } else {
            for (i = 0; i < ssid_num; i++) {
    			sprintf(dhcpr_ssid+strlen(dhcpr_ssid), ";");
    		}

            sprintf(dhcpr_lan+strlen(dhcpr_lan), ";;");
        }
        dhcpr_ssid[strlen(dhcpr_ssid) - 1] = '\0';
        fprintf(fp, "DHCPRelaySSID=%s\n", dhcpr_ssid);
        dhcpr_lan[strlen(dhcpr_lan) - 1] = '\0';
        fprintf(fp, "DHCPRelayLAN=%s\n", dhcpr_lan);
        dhcpr_ssid_src[strlen(dhcpr_ssid_src) - 1] = '\0';
        fprintf(fp, "DHCPRelaySSIDSrcIP=%s\n", dhcpr_ssid_src);
        dhcpr_ssid_mac[strlen(dhcpr_ssid_mac) - 1] = '\0';
        fprintf(fp, "DHCPRelaySSIDSrcMAC=%s\n", dhcpr_ssid_mac);
		/*
		 * There are no SSID/WPAPSK/Key1Str/Key2Str/Key3Str/Key4Str anymore since driver1.5 , but
		 * STA WPS still need these entries to show the WPS result(That is the only way i know to get WPAPSK key) and
		 * so we create empty entries here.   --YY
		 */
		fprintf(fp, "SSID=\nWPAPSK=\nKey1Str=\nKey2Str=\nKey3Str=\nKey4Str=\n");
	}

//#if CONFIG_RT3572_AP_MODULE == 1
#define FPRINT_NUM_3572(x,s) fprintf(fp1, #x"=%d\n", atoi(nvram_bufget(RT2860_NVRAM, #s)));
#define FPRINT_STR_3572(x,s) fprintf(fp1, #x"=%s\n", nvram_bufget(RT2860_NVRAM, #s));

	//if (RTDEV_NVRAM == mode) {
#if defined (CONFIG_RTDEV)
		system("mkdir -p /etc/Wireless/iNIC");
		fp1 = fopen("/etc/Wireless/iNIC/iNIC_ap.dat", "w+");
#endif

		fprintf(fp1, "#The word of \"Default\" must not be removed\n");
		fprintf(fp1, "Default\n");

		/*Wireless LAN 5G WirelessRoaming_en*/
		FPRINT_NUM_3572(WirelessRoaming5G_en,WirelessRoaming5G_en);
		FPRINT_STR_3572(STA_CTL_5g,STA_CTL_5g);
		
		FPRINT_NUM_3572(ClientLimit_Enable, ClientLimit5G_Enable_3572);
		FPRINT_NUM_3572(ClientLimit, ClientLimit5G_3572);
		FPRINT_NUM_3572(CountryRegion, CountryRegion_3572);
		FPRINT_NUM_3572(CountryRegionABand, CountryRegionABand_3572);
		FPRINT_STR_3572(CountryCode, CountryCode_3572);

		FPRINT_NUM_3572(BssidNum, BssidNum_3572);
		ssid_num = atoi(nvram_get(mode, "BssidNum"));

		FPRINT_STR_3572(SSID1,  SSID1_3572);
		FPRINT_STR_3572(SSID2,  SSID2_3572);
		FPRINT_STR_3572(SSID3,  SSID3_3572);
		FPRINT_STR_3572(SSID4, SSID4_3572);
		FPRINT_STR_3572(SSID5, SSID5_3572);
		FPRINT_STR_3572(SSID6, SSID6_3572);
		FPRINT_STR_3572(SSID7, SSID7_3572);
		FPRINT_STR_3572(SSID8, SSID8_3572);

		FPRINT_NUM_3572(WirelessMode, WirelessMode_3572);
		FPRINT_STR_3572(FixedTxMode, FixedTxMode_3572);

		//TxRate(FixedRate)
		bzero(tx_rate, sizeof(char)*16);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(tx_rate+strlen(tx_rate), "%d",
					atoi(nvram_bufget(mode, "TxRate_3572")));
			sprintf(tx_rate+strlen(tx_rate), "%c", ';');
		}
		tx_rate[strlen(tx_rate) - 1] = '\0';
		fprintf(fp1, "TxRate=%s\n", tx_rate);

		FPRINT_NUM_3572(Channel, Channel_3572);
		FPRINT_NUM_3572(BasicRate, BasicRate_3572);
		FPRINT_NUM_3572(BeaconPeriod, BeaconPeriod_3572);
		FPRINT_NUM_3572(DtimPeriod, DtimPeriod_3572);
		FPRINT_NUM_3572(TxPower, TxPower_3572);
		FPRINT_NUM_3572(DisableOLBC, DisableOLBC_3572);
		FPRINT_NUM_3572(BGProtection, BGProtection_3572);
		FPRINT_NUM_3572(TxAntenna, TxAntenna_3572);
		FPRINT_NUM_3572(RxAntenna, RxAntenna_3572);
		FPRINT_NUM_3572(TxPreamble, TxPreamble_3572);
		FPRINT_NUM_3572(RTSThreshold, RTSThreshold_3572);
		FPRINT_NUM_3572(FragThreshold, FragThreshold_3572);

		wmm = atoi(nvram_bufget(mode, "WmmCapable"));
		if (wmm)
			fprintf(fp1, "TxBurst=0\n");
		else
			FPRINT_NUM_3572(TxBurst, TxBurst_3572);

		FPRINT_NUM_3572(PktAggregate, PktAggregate_3572);
		FPRINT_NUM_3572(AutoProvisionEn, AutoProvisionEn_3572);
		FPRINT_NUM_3572(FreqDelta, FreqDelta_3572);
		fprintf(fp1, "TurboRate=0\n");

		//WmmCapable
		bzero(wmm_enable, sizeof(char)*16);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(wmm_enable+strlen(wmm_enable), "%d",
					atoi(nvram_bufget(mode, "WmmCapable")));
			sprintf(wmm_enable+strlen(wmm_enable), "%c", ';');
		}
		wmm_enable[strlen(wmm_enable) - 1] = '\0';
		fprintf(fp1, "WmmCapable=%s\n", wmm_enable);

		FPRINT_STR_3572(APAifsn, APAifsn_3572);
		FPRINT_STR_3572(APCwmin, APCwmin_3572);
		FPRINT_STR_3572(APCwmax, APCwmax_3572);
		FPRINT_STR_3572(APTxop, APTxop_3572);
		FPRINT_STR_3572(APACM, APACM_3572);
		FPRINT_STR_3572(BSSAifsn, BSSAifsn_3572);
		FPRINT_STR_3572(BSSCwmin, BSSCwmin_3572);
		FPRINT_STR_3572(BSSCwmax, BSSCwmax_3572);
		FPRINT_STR_3572(BSSTxop, BSSTxop_3572);
		FPRINT_STR_3572(BSSACM, BSSACM_3572);
		FPRINT_STR_3572(AckPolicy, AckPolicy_3572);
		FPRINT_STR_3572(APSDCapable, APSDCapable_3572);
		FPRINT_STR_3572(DLSCapable, DLSCapable_3572);
		FPRINT_STR_3572(NoForwarding, NoForwarding_3572);
		//Enable default no forwarding between MBSSIDs. (5G) --Albert
		//FPRINT_NUM(NoForwardingBTNBSSID);
		fprintf(fp1, "NoForwardingBTNBSSID=1\n");

		FPRINT_STR_3572(HideSSID, HideSSID_3572);
		FPRINT_NUM_3572(ShortSlot, ShortSlot_3572);
		FPRINT_NUM_3572(AutoChannelSelect, AutoChannelSelect_3572);

		FPRINT_STR_3572(IEEE8021X, IEEE8021X_3572);
		//FPRINT_NUM_3572(IEEE80211H, IEEE80211H_3572);
		// James.Lin: change for DFS function
		fprintf(fp1, "IEEE80211H=1\n");
		FPRINT_NUM_3572(CarrierDetect, CarrierDetect_3572);

		FPRINT_NUM_3572(ITxBfEn, ITxBfEn_3572);
		FPRINT_STR_3572(PreAntSwitch, PreAntSwitch_3572);
		FPRINT_NUM_3572(PhyRateLimit, PhyRateLimit_3572);
		FPRINT_NUM_3572(DebugFlags, DebugFlags_3572);
		FPRINT_NUM_3572(ETxBfEnCond, ETxBfEnCond_3572);
		FPRINT_NUM_3572(ITxBfTimeout, ITxBfTimeout_3572);
		FPRINT_NUM_3572(ETxBfTimeout, ETxBfTimeout_3572);
		FPRINT_NUM_3572(ETxBfNoncompress, ETxBfNoncompress_3572);
		FPRINT_NUM_3572(ETxBfIncapable, ETxBfIncapable_3572);
		FPRINT_NUM_3572(FineAGC, FineAGC_3572);
		FPRINT_NUM_3572(StreamMode, StreamMode_3572);
		FPRINT_STR_3572(StreamModeMac0, StreamModeMac0_3572);
		FPRINT_STR_3572(StreamModeMac1, StreamModeMac1_3572);
		FPRINT_STR_3572(StreamModeMac2, StreamModeMac2_3572);
		FPRINT_STR_3572(StreamModeMac3, StreamModeMac3_3572);


		FPRINT_NUM_3572(CSPeriod, CSPeriod_3572);
		//FPRINT_STR_3572(RDRegion, RDRegion_3572);
		// James.Lin: change for DFS function
		fprintf(fp1, "RDRegion=CE\n");
		// James.Lin: Add for 5600~5650 weather radar channel detection rate too low issue
		FPRINT_NUM_3572(DfsDeclareThres, DfsDeclareThres_3572);
		FPRINT_NUM_3572(StationKeepAlive, StationKeepAlive_3572);
		FPRINT_NUM_3572(DfsLowerLimit, DfsLowerLimit_3572);
		FPRINT_NUM_3572(DfsUpperLimit, DfsUpperLimit_3572);
		FPRINT_NUM_3572(DfsOutdoor, DfsOutdoor_3572);
		FPRINT_NUM_3572(SymRoundFromCfg, SymRoundFromCfg_3572);
        FPRINT_NUM_3572(BusyIdleFromCfg, BusyIdleFromCfg_3572);
        FPRINT_NUM_3572(DfsRssiHighFromCfg, DfsRssiHighFromCfg_3572);
        FPRINT_NUM_3572(DfsRssiLowFromCfg, DfsRssiLowFromCfg_3572);
		FPRINT_NUM_3572(DFSParamFromConfig, DFSParamFromConfig_3572);
		FPRINT_STR_3572(FCCParamCh0, FCCParamCh0_3572);
		FPRINT_STR_3572(FCCParamCh1, FCCParamCh1_3572);
		FPRINT_STR_3572(FCCParamCh2, FCCParamCh2_3572);
		FPRINT_STR_3572(FCCParamCh3, FCCParamCh3_3572);
		FPRINT_STR_3572(CEParamCh0, CEParamCh0_3572);
		FPRINT_STR_3572(CEParamCh1, CEParamCh1_3572);
		FPRINT_STR_3572(CEParamCh2, CEParamCh2_3572);
		FPRINT_STR_3572(CEParamCh3, CEParamCh3_3572);
		FPRINT_STR_3572(JAPParamCh0, JAPParamCh0_3572);
		FPRINT_STR_3572(JAPParamCh1, JAPParamCh1_3572);
		FPRINT_STR_3572(JAPParamCh2, JAPParamCh2_3572);
		FPRINT_STR_3572(JAPParamCh3, JAPParamCh3_3572);
		FPRINT_STR_3572(JAPW53ParamCh0, JAPW53ParamCh0_3572);
		FPRINT_STR_3572(JAPW53ParamCh1, JAPW53ParamCh1_3572);
		FPRINT_STR_3572(JAPW53ParamCh2, JAPW53ParamCh2_3572);
		FPRINT_STR_3572(JAPW53ParamCh3, JAPW53ParamCh3_3572);
		FPRINT_NUM_3572(FixDfsLimit, FixDfsLimit_3572);
		FPRINT_NUM_3572(LongPulseRadarTh, LongPulseRadarTh_3572);
		FPRINT_NUM_3572(AvgRssiReq, AvgRssiReq_3572);
		FPRINT_NUM_3572(DFS_R66, DFS_R66_3572);
		FPRINT_STR_3572(BlockCh, BlockCh_3572);

		FPRINT_NUM_3572(GreenAP, GreenAP_3572);
		FPRINT_STR_3572(PreAuth, PreAuth_3572);
		FPRINT_STR_3572(AuthMode, AuthMode_3572);
		FPRINT_STR_3572(EncrypType, EncrypType_3572);
        /*kurtis: WAPI*/
		FPRINT_STR_3572(WapiPsk1, WapiPsk1_3572);
		FPRINT_STR_3572(WapiPsk2, WapiPsk2_3572);
		FPRINT_STR_3572(WapiPsk3, WapiPsk3_3572);
		FPRINT_STR_3572(WapiPsk4, WapiPsk4_3572);
		FPRINT_STR_3572(WapiPsk5, WapiPsk5_3572);
		FPRINT_STR_3572(WapiPsk6, WapiPsk6_3572);
		FPRINT_STR_3572(WapiPsk7, WapiPsk7_3572);
		FPRINT_STR_3572(WapiPsk8, WapiPsk8_3572);
		FPRINT_STR_3572(WapiPskType, WapiPskType_3572);
		FPRINT_STR_3572(Wapiifname, Wapiifname_3572);
		FPRINT_STR_3572(WapiAsCertPath, WapiAsCertPath_3572);
		FPRINT_STR_3572(WapiUserCertPath, WapiUserCertPath_3572);
		FPRINT_STR_3572(WapiAsIpAddr, WapiAsIpAddr_3572);
		FPRINT_STR_3572(WapiAsPort, WapiAsPort_3572);

		FPRINT_NUM_3572(BssidNum, BssidNum_3572);

		FPRINT_STR_3572(RekeyMethod, RekeyMethod_3572);
		FPRINT_NUM_3572(RekeyInterval, RekeyInterval_3572);
		FPRINT_STR_3572(PMKCachePeriod, PMKCachePeriod_3572);

		FPRINT_NUM_3572(MeshAutoLink, MeshAutoLink_3572);
		FPRINT_STR_3572(MeshAuthMode, MeshAuthMode_3572);
		FPRINT_STR_3572(MeshEncrypType, MeshEncrypType_3572);
		FPRINT_NUM_3572(MeshDefaultkey, MeshDefaultkey_3572);
		FPRINT_STR_3572(MeshWEPKEY, MeshWEPKEY_3572);
		FPRINT_STR_3572(MeshWPAKEY, MeshWPAKEY_3572);
		FPRINT_STR_3572(MeshId, MeshId_3572);

		//WPAPSK
		FPRINT_STR_3572(WPAPSK1, WPAPSK1_3572);
		FPRINT_STR_3572(WPAPSK2, WPAPSK2_3572);
		FPRINT_STR_3572(WPAPSK3, WPAPSK3_3572);
		FPRINT_STR_3572(WPAPSK4, WPAPSK4_3572);
		FPRINT_STR_3572(WPAPSK5, WPAPSK5_3572);
		FPRINT_STR_3572(WPAPSK6, WPAPSK6_3572);
		FPRINT_STR_3572(WPAPSK7, WPAPSK7_3572);
		FPRINT_STR_3572(WPAPSK8, WPAPSK8_3572);

		FPRINT_STR_3572(DefaultKeyID, DefaultKeyID_3572);
		FPRINT_STR_3572(Key1Type, Key1Type_3572);
		FPRINT_STR_3572(Key1Str1, Key1Str1_3572);
		FPRINT_STR_3572(Key1Str2, Key1Str2_3572);
		FPRINT_STR_3572(Key1Str3, Key1Str3_3572);
		FPRINT_STR_3572(Key1Str4, Key1Str4_3572);
		FPRINT_STR_3572(Key1Str5, Key1Str5_3572);
		FPRINT_STR_3572(Key1Str6, Key1Str6_3572);
		FPRINT_STR_3572(Key1Str7, Key1Str7_3572);
		FPRINT_STR_3572(Key1Str8, Key1Str8_3572);

		FPRINT_STR_3572(Key2Type, Key2Type_3572);
		FPRINT_STR_3572(Key2Str1, Key2Str1_3572);
		FPRINT_STR_3572(Key2Str2, Key2Str2_3572);
		FPRINT_STR_3572(Key2Str3, Key2Str3_3572);
		FPRINT_STR_3572(Key2Str4, Key2Str4_3572);
		FPRINT_STR_3572(Key2Str5, Key2Str5_3572);
		FPRINT_STR_3572(Key2Str6, Key2Str6_3572);
		FPRINT_STR_3572(Key2Str7, Key2Str7_3572);
		FPRINT_STR_3572(Key2Str8, Key2Str8_3572);

		FPRINT_STR_3572(Key3Type, Key3Type_3572);
		FPRINT_STR_3572(Key3Str1, Key3Str1_3572);
		FPRINT_STR_3572(Key3Str2, Key3Str2_3572);
		FPRINT_STR_3572(Key3Str3, Key3Str3_3572);
		FPRINT_STR_3572(Key3Str4, Key3Str4_3572);
		FPRINT_STR_3572(Key3Str5, Key3Str5_3572);
		FPRINT_STR_3572(Key3Str6, Key3Str6_3572);
		FPRINT_STR_3572(Key3Str7, Key3Str7_3572);
		FPRINT_STR_3572(Key3Str8, Key3Str8_3572);

		FPRINT_STR_3572(Key4Type, Key4Type_3572);
		FPRINT_STR_3572(Key4Str1, Key4Str1_3572);
		FPRINT_STR_3572(Key4Str2, Key4Str2_3572);
		FPRINT_STR_3572(Key4Str3, Key4Str3_3572);
		FPRINT_STR_3572(Key4Str4, Key4Str4_3572);
		FPRINT_STR_3572(Key4Str5, Key4Str5_3572);
		FPRINT_STR_3572(Key4Str6, Key4Str6_3572);
		FPRINT_STR_3572(Key4Str7, Key4Str7_3572);
		FPRINT_STR_3572(Key4Str8, Key4Str8_3572);

		FPRINT_NUM_3572(HSCounter, HSCounter_3572);

		FPRINT_NUM_3572(HT_HTC, HT_HTC_3572);
		if (wmm)
			fprintf(fp1, "HT_RDG=0\n");
		else
			fprintf(fp1, "HT_RDG=1\n");
		FPRINT_NUM_3572(HT_LinkAdapt, HT_LinkAdapt_3572);
		FPRINT_NUM_3572(HT_OpMode, HT_OpMode_3572);
		FPRINT_NUM_3572(HT_MpduDensity, HT_MpduDensity_3572);
		FPRINT_STR_3572(HT_EXTCHA, HT_EXTCHA_3572);
		FPRINT_NUM_3572(HT_BW, HT_BW_3572);
		FPRINT_NUM_3572(VHT_BW, VHT_BW_3572);
		FPRINT_NUM_3572(HT_AutoBA, HT_AutoBA_3572);
		FPRINT_NUM_3572(HT_BADecline, HT_BADecline_3572);
		FPRINT_NUM_3572(HT_AMSDU, HT_AMSDU_3572);
		FPRINT_NUM_3572(HT_BAWinSize, HT_BAWinSize_3572);
		FPRINT_NUM_3572(HT_GI, HT_GI_3572);
		FPRINT_NUM_3572(HT_STBC, HT_STBC_3572);
		FPRINT_STR_3572(HT_MCS, HT_MCS_3572);
		FPRINT_NUM_3572(HT_TxStream, HT_TxStream_3572);
		FPRINT_NUM_3572(HT_RxStream, HT_RxStream_3572);
		FPRINT_NUM_3572(HT_PROTECT, HT_PROTECT_3572);
		FPRINT_NUM_3572(HT_DisallowTKIP, HT_DisallowTKIP_3572);
		FPRINT_NUM_3572(HT_BSSCoexistence, HT_BSSCoexistence_3572);
		FPRINT_NUM_3572(GreenAP, GreenAP_3572);

		//WPS Parameters
		FPRINT_NUM_3572(WscConfMode, WscConfMode_3572);

		//WscConfStatus
		if (atoi(nvram_bufget(mode, "WscConfigured")) == 0)
			fprintf(fp1, "WscConfStatus=%d\n", 1);
		else
			fprintf(fp1, "WscConfStatus=%d\n", 2);
		//if (strcmp(nvram_bufget(mode, "WscVendorPinCode"), "") != 0)
			//FPRINT_STR(WscVendorPinCode);
		fprintf(fp1, "WCNTest=0\n");

        /*#ifdef ACL_256_SUPPORT
		FPRINT_NUM(AccessPolicy0);
		save_acl_to_file(mode, 0, "AccessControlList0", "AccessControlList0");
		FPRINT_NUM(AccessPolicy1);
		save_acl_to_file(mode, 1, "AccessControlList1", "AccessControlList1");
		FPRINT_NUM(AccessPolicy2);
		save_acl_to_file(mode, 2, "AccessControlList2", "AccessControlList2");
		FPRINT_NUM(AccessPolicy3);
		save_acl_to_file(mode, 3, "AccessControlList3", "AccessControlList3");
		FPRINT_NUM(AccessPolicy4);
		save_acl_to_file(mode, 4, "AccessControlList4", "AccessControlList4");
		FPRINT_NUM(AccessPolicy5);
		save_acl_to_file(mode, 5, "AccessControlList5", "AccessControlList5");
		FPRINT_NUM(AccessPolicy6);
		save_acl_to_file(mode, 6, "AccessControlList6", "AccessControlList6");
		FPRINT_NUM(AccessPolicy7);
		save_acl_to_file(mode, 7, "AccessControlList7", "AccessControlList7");
        #else*/
		FPRINT_NUM_3572(AccessPolicy0, AccessPolicy0_3572);
		FPRINT_STR_3572(AccessControlList0, AccessControlList0_3572);
		FPRINT_NUM_3572(AccessPolicy1, AccessPolicy1_3572);
		FPRINT_STR_3572(AccessControlList1, AccessControlList1_3572);
		FPRINT_NUM_3572(AccessPolicy2, AccessPolicy2_3572);
		FPRINT_STR_3572(AccessControlList2, AccessControlList2_3572);
		FPRINT_NUM_3572(AccessPolicy3, AccessPolicy3_3572);
		FPRINT_STR_3572(AccessControlList3, AccessControlList3_3572);
		FPRINT_NUM_3572(AccessPolicy4, AccessPolicy4_3572);
		FPRINT_STR_3572(AccessControlList4, AccessControlList4_3572);
		FPRINT_NUM_3572(AccessPolicy5, AccessPolicy5_3572);
		FPRINT_STR_3572(AccessControlList5, AccessControlList5_3572);
		FPRINT_NUM_3572(AccessPolicy6, AccessPolicy6_3572);
		FPRINT_STR_3572(AccessControlList6, AccessControlList6_3572);
		FPRINT_NUM_3572(AccessPolicy7, AccessPolicy7_3572);
		FPRINT_STR_3572(AccessControlList7, AccessControlList7_3572);
        //#endif

		// WDS Parameters
		FPRINT_NUM_3572(WdsEnable, WdsEnable_3572);
		FPRINT_STR_3572(WdsPhyMode, WdsPhyMode_3572);
		FPRINT_STR_3572(WdsEncrypType, WdsEncrypType_3572);
		FPRINT_STR_3572(WdsList, WdsList_3572);
		//FPRINT_STR_3572(WdsKey, WdsKey_3572);
		FPRINT_STR_3572(Wds0Key, Wds0Key_3572);
		FPRINT_STR_3572(Wds1Key, Wds1Key_3572);
		FPRINT_STR_3572(Wds2Key, Wds2Key_3572);
		FPRINT_STR_3572(Wds3Key, Wds3Key_3572);
		FPRINT_STR_3572(RADIUS_Server, RADIUS_Server_3572);
		FPRINT_STR_3572(RADIUS_Port, RADIUS_Port_3572);
		//FPRINT_STR(RADIUS_Key);
		FPRINT_STR_3572(RADIUS_Key1, RADIUS_Key1_3572);
		FPRINT_STR_3572(RADIUS_Key2, RADIUS_Key2_3572);
		FPRINT_STR_3572(RADIUS_Key3, RADIUS_Key3_3572);
		FPRINT_STR_3572(RADIUS_Key4, RADIUS_Key4_3572);
		FPRINT_STR_3572(RADIUS_Key5, RADIUS_Key5_3572);
		FPRINT_STR_3572(RADIUS_Key6, RADIUS_Key6_3572);
		FPRINT_STR_3572(RADIUS_Key7, RADIUS_Key7_3572);
		FPRINT_STR_3572(RADIUS_Key8, RADIUS_Key8_3572);
		FPRINT_STR_3572(RADIUS_Acct_Server, RADIUS_Acct_Server_3572);
		FPRINT_NUM_3572(RADIUS_Acct_Port, RADIUS_Acct_Port_3572);
		FPRINT_STR_3572(RADIUS_Acct_Key, RADIUS_Acct_Key_3572);
		FPRINT_STR_3572(own_ip_addr, own_ip_addr_3572);
		FPRINT_STR_3572(Ethifname, Ethifname_3572);
		FPRINT_STR_3572(EAPifname, EAPifname_3572);
		FPRINT_STR_3572(PreAuthifname, PreAuthifname_3572);
		FPRINT_NUM_3572(session_timeout_interval, session_timeout_interval_3572);
		FPRINT_NUM_3572(idle_timeout_interval, idle_timeout_interval_3572);
		FPRINT_NUM_3572(WiFiTest, WiFiTest_3572);
		FPRINT_NUM_3572(TGnWifiTest, TGnWifiTest_3572);

		//AP Client parameters
		FPRINT_NUM_3572(ApCliEnable, ApCliEnable_3572);
		FPRINT_STR_3572(ApCliSsid, ApCliSsid_3572);
		FPRINT_STR_3572(ApCliBssid, ApCliBssid_3572);
		FPRINT_STR_3572(ApCliAuthMode, ApCliAuthMode_3572);
		FPRINT_STR_3572(ApCliEncrypType, ApCliEncrypType_3572);
		FPRINT_STR_3572(ApCliWPAPSK, ApCliWPAPSK_3572);
		FPRINT_NUM_3572(ApCliDefaultKeyID, ApCliDefaultKeyID_3572);
		FPRINT_NUM_3572(ApCliKey1Type, ApCliKey1Type_3572);
		FPRINT_STR_3572(ApCliKey1Str, ApCliKey1Str_3572);
		FPRINT_NUM_3572(ApCliKey2Type, ApCliKey2Type_3572);
		FPRINT_STR_3572(ApCliKey2Str, ApCliKey2Str_3572);
		FPRINT_NUM_3572(ApCliKey3Type, ApCliKey3Type_3572);
		FPRINT_STR_3572(ApCliKey3Str, ApCliKey3Str_3572);
		FPRINT_NUM_3572(ApCliKey4Type, ApCliKey4Type_3572);
		FPRINT_STR_3572(ApCliKey4Str, ApCliKey4Str_3572);

		//Radio On/Off
		if (atoi(nvram_bufget(RT2860_NVRAM, "RadioOff_3572")) == 1)
			fprintf(fp1, "RadioOn=0\n");
		else
			fprintf(fp1, "RadioOn=1\n");

		fclose(fp1);
	//}
//#endif

#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	//if (RT2561_NVRAM == mode) {
	if (RTDEV_NVRAM == mode) {
		FPRINT_NUM(CountryRegion);
		FPRINT_NUM(CountryRegionABand);
		FPRINT_STR(CountryCode);
		FPRINT_NUM(BssidNum);
		ssid_num = atoi(nvram_get(mode, "BssidNum"));
		FPRINT_STR(SSID);
		FPRINT_NUM(WirelessMode);
		//TxRate(FixedRate)
		bzero(tx_rate, sizeof(char)*12);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(tx_rate+strlen(tx_rate), "%d",
			atoi(nvram_bufget(mode, "TxRate")));
			sprintf(tx_rate+strlen(tx_rate), "%c", ';');
		}
		tx_rate[strlen(tx_rate) - 1] = '\0';
		fprintf(fp, "TxRate=%s\n", tx_rate);

		FPRINT_NUM(Channel);
		FPRINT_NUM(BasicRate);
		FPRINT_NUM(BeaconPeriod);
		FPRINT_NUM(DtimPeriod);
		FPRINT_NUM(TxPower);
		FPRINT_NUM(DisableOLBC);
		FPRINT_NUM(BGProtection);
		fprintf(fp, "TxAntenna=\n");
		fprintf(fp, "RxAntenna=\n");
		FPRINT_NUM(TxPreamble);
		FPRINT_NUM(RTSThreshold  );
		FPRINT_NUM(FragThreshold  );
		FPRINT_NUM(TxBurst);
		FPRINT_NUM(PktAggregate);
		fprintf(fp, "TurboRate=0\n");

		//WmmCapable
		bzero(wmm_enable, sizeof(char)*8);
		for (i = 0; i < ssid_num; i++)
		{
			sprintf(wmm_enable+strlen(wmm_enable), "%d",
			atoi(nvram_bufget(mode, "WmmCapable")));
			sprintf(wmm_enable+strlen(wmm_enable), "%c", ';');
		}
		wmm_enable[strlen(wmm_enable) - 1] = '\0';
		fprintf(fp, "WmmCapable=%s\n", wmm_enable);

		FPRINT_STR(APAifsn);
		FPRINT_STR(APCwmin);
		FPRINT_STR(APCwmax);
		FPRINT_STR(APTxop);
		FPRINT_STR(APACM);
		FPRINT_STR(BSSAifsn);
		FPRINT_STR(BSSCwmin);
		FPRINT_STR(BSSCwmax);
		FPRINT_STR(BSSTxop);
		FPRINT_STR(BSSACM);
		FPRINT_STR(AckPolicy);
		FPRINT_STR(APSDCapable);
		FPRINT_STR(DLSCapable);
		FPRINT_STR(NoForwarding);
		FPRINT_NUM(NoForwardingBTNBSSID);
		FPRINT_STR(HideSSID);
		FPRINT_NUM(ShortSlot);
		FPRINT_NUM(AutoChannelSelect);
		FPRINT_NUM(MaxTxPowerLevel);
		FPRINT_STR(IEEE8021X);
		FPRINT_NUM(IEEE80211H);
		FPRINT_NUM(CSPeriod);
		FPRINT_STR(RDRegion);
		FPRINT_STR(PreAuth);
		FPRINT_STR(AuthMode);
		FPRINT_STR(EncrypType);
        /*kurtis: WAPI*/
		FPRINT_STR(WapiPsk1);
		FPRINT_STR(WapiPskType);
		FPRINT_STR(Wapiifname);
		FPRINT_STR(WapiAsCertPath);
		FPRINT_STR(WapiUserCertPath);
		FPRINT_STR(WapiAsIpAddr);
		FPRINT_STR(WapiAsPort);

		FPRINT_NUM(RekeyInterval);
		FPRINT_STR(RekeyMethod);
		FPRINT_STR(PMKCachePeriod);
		FPRINT_STR(WPAPSK);
		FPRINT_STR(DefaultKeyID);
		FPRINT_STR(Key1Type);
		FPRINT_STR(Key1Str);
		FPRINT_STR(Key2Type);
		FPRINT_STR(Key2Str);
		FPRINT_STR(Key3Type);
		FPRINT_STR(Key3Str);
		FPRINT_STR(Key4Type);
		FPRINT_STR(Key4Str);
		FPRINT_NUM(HSCounter);
		FPRINT_NUM(AccessPolicy0);
		FPRINT_STR(AccessControlList0);
		FPRINT_NUM(AccessPolicy1);
		FPRINT_STR(AccessControlList1);
		FPRINT_NUM(AccessPolicy2);
		FPRINT_STR(AccessControlList2);
		FPRINT_NUM(AccessPolicy3);
		FPRINT_STR(AccessControlList3);
		FPRINT_NUM(WdsEnable);
		FPRINT_STR(WdsPhyMode);
		FPRINT_STR(WdsEncrypType);
		FPRINT_STR(WdsList);
		FPRINT_STR(WdsKey);
		FPRINT_STR(RADIUS_Server);
		FPRINT_STR(RADIUS_Port);
		FPRINT_STR(RADIUS_Key);
		FPRINT_STR(own_ip_addr);
		FPRINT_STR(Ethifname);
		//AP Client parameters
		FPRINT_NUM(ApCliEnable);
		FPRINT_STR(ApCliSsid);
		FPRINT_STR(ApCliBssid);
		FPRINT_STR(ApCliAuthMode);
		FPRINT_STR(ApCliEncrypType);
		FPRINT_STR(ApCliWPAPSK);
		FPRINT_NUM(ApCliDefaultKeyID);
		FPRINT_NUM(ApCliKey1Type);
		FPRINT_STR(ApCliKey1Str);
		FPRINT_NUM(ApCliKey2Type);
		FPRINT_STR(ApCliKey2Str);
		FPRINT_NUM(ApCliKey3Type);
		FPRINT_STR(ApCliKey3Str);
		FPRINT_NUM(ApCliKey4Type);
		FPRINT_STR(ApCliKey4Str);
		FPRINT_NUM(WscConfMode);
		//WscConfStatus
		if (atoi(nvram_bufget(mode, "WscConfigured")) == 0)
			fprintf(fp, "WscConfStatus=%d\n", 1);
		else
			fprintf(fp, "WscConfStatus=%d\n", 2);
	}
#elif defined (CONFIG_RTDEV_PLC)
	if (RTDEV_NVRAM == mode) {
		FPRINT_STR(MacAddress);
		FPRINT_STR(NPW);
		FPRINT_STR(DPW);
		FPRINT_NUM(BackupCCo);
		FPRINT_NUM(CCo);
		FPRINT_NUM(UnAssoCCo);
		FPRINT_NUM(STA);
		FPRINT_NUM(UnAssoSTA);
	}
#endif

	nvram_close(mode);
	fclose(fp);
	return 0;
}

#define BUFSZ 5120
unsigned char nv_buf[BUFSZ] = {0};
int renew_nvram(int mode, char *fname)
{
	FILE *fp;
	unsigned char *buf, *p;
	unsigned char wan_mac[32];
	int found = 0, need_commit = 0;

    buf = nv_buf;

	if (NULL == (fp = fopen(fname, "ro"))) {
		perror("fopen");
		return -1;
	}

	//find "Default" first
	while (NULL != fgets(buf, BUFSZ, fp)) {
		if (buf[0] == '\n' || buf[0] == '#')
			continue;
		if (!strncmp(buf, "Default\n", 8)) {
			found = 1;
			break;
		}
	}
	if (!found) {
		printf("file format error!\n");
		fclose(fp);
		return -1;
	}

	nvram_init(mode);
	while (NULL != fgets(buf, BUFSZ, fp)) {
		if (buf[0] == '\n' || buf[0] == '#')
			continue;
		if (NULL == (p = strchr(buf, '='))) {
			if (need_commit)
				nvram_commit(mode);
			printf("%s file format error!\n", fname);
			fclose(fp);
			return -1;
		}
		buf[strlen(buf) - 1] = '\0'; //remove carriage return
		*p++ = '\0'; //seperate the string
		//printf("bufset %d '%s'='%s'\n", mode, buf, p);
		nvram_bufset(mode, buf, p);
		need_commit = 1;
	}

	need_commit = 1;

	if (need_commit)
		nvram_commit(mode);

	nvram_close(mode);
	fclose(fp);
	return 0;
}

int nvram_show(int mode)
{
	char *buffer, *p;
	int rc;
	int crc;
	unsigned int len = 0x4000;

	nvram_init(mode);
	len = getNvramBlockSize(mode);
	buffer = malloc(len);
	if (buffer == NULL) {
		fprintf(stderr, "nvram_show: Can not allocate memory!\n");
		return -1;
	}
	flash_read(buffer, getNvramOffset(mode), len);
	memcpy(&crc, buffer, 4);

	fprintf(stderr, "crc = %x\n", crc);
	p = buffer + 4;
	while (*p != '\0') {
		printf("%s\n", p);
		p += strlen(p) + 1;
	}

	free(buffer);
	return 0;
}

int nvram_getmac(int lan)
{
	unsigned char buf[6];

	//Get wan port mac address, please refer to eeprom format doc
	//0x30000=user configure, 0x32000=rt2860 parameters, 0x40000=RF parameter
	flash_read_mac(buf);	// Modify flash_read_mac() to reads LAN MAC
	//if (lan)
		//buf[5]-=2;; 		//set to LAN MAC, Modify flash_read_mac() to reads LAN MAC
	printf("%0X:%0X:%0X:%0X:%0X:%0X\n",buf[0],buf[1],buf[2],buf[3],buf[4],buf[5]);
	return 0;
}

