/**

  EAP (Extensible Authentication Protocol) functionality.

  File: ssheap.h

  <keywords EAP (Extensible Authentication Protocol), Extensible
  Authentication Protocol (EAP), authentication/EAP>

  Extensible Authentication Protocol (EAP) is a protocol for
  encapsulating an authentication protocol, within a defined
  framework, which provides simple "negotiation" of the protocol to
  use for authentication.

  The EAP library defines three types of objects: SshEapConfiguration,
  SshEapConnection and SshEap.

  SshEapConfiguration contains values for all configuration parameters
  of a SshEap instance, which are common to all EAP protocols.  A
  SshEapConfiguration instance can be shared among several SshEap
  instances. An SshEapConfiguration instance must not be destroyed
  before all of the SshEap instances using it have been destroyed.

  SshEapConnection provides a common interface to the SshEap library
  for passing packets to and from it. One SshEapConnection object can
  be associated with only one SshEap instance. The SshEapConnection
  instance must not be destroyed before the SshEap instance using it
  is destroyed.

  The SshEap object is the workhorse of the EAP library, which handles
  tasks common to all EAP protocols, and routes messages and signals
  between EAP authentication protocols and the caller.

  The signaling and message-passing with the EAP library is based on
  simple callback functions, with some restrictions.

  The following constraints apply to using the library:

  1. A SshEap object must be destroyed before a SshEapConfiguration
     or SshEapConnection object it is using is destroyed.

  2. All message passing with the library uses SshBuffer's. Any SshBuffer
     passed to the library is "the property" of that library, and must
     not be modified by the caller unless it is passed back via a callback.

  3. The EAP library uses a set of defined signals to inform what it
     expects the user to do next. Several of the functions present can
     only be called in response to a certain signal and calling them at
     an inappropriate time will result in an assertion failing. See the
     descriptions of the individual functions and signals for more
     information.

  4. The EAP protocol is a request-response protocol, which assumes only
     one request-response process to be active at any time on one connection.
     Thus calling any EAP library function, which initiates the sending of a
     EAP message will implicitly discard any replies which have not been
     received to any previously sent request. The caller of an SshEap
     instance should therefore call the functions
     ssh_eap_send_notification_request, ssh_eap_send_identification_request
     and ssh_eap_authenticate only after:

     - The reception of a signal which denotes a timeout or reception of a
       response.

     - The calling of ssh_eap_reset.

     - The creation of the SshEap instance using ssh_eap_create_server.

  5. An EAP protocol exchange should always begin with an identity request,
     and hence a call to ssh_eap_send_identification_request should
     be performed before a call to ssh_eap_authenticate

  6. If the owner of a SshEap instance has an EAP message queued for output,
     it should discard this earlier message in favor of the later one if it
     must discard an EAP message in such a situation.

  7. The only EAP library functions functions which can be called from SshEap
     or SshEapConnection callbacks are:

     + ssh_eap_destroy
     + ssh_eap_connection_destroy
     + ssh_eap_reset
     + ssh_eap_token
     + ssh_eap_accept_auth
     + ssh_eap_accept_auth_none
     + ssh_eap_configure_protocol

     The exception is callbacks of type SshEapAsyncSignalCB, from which any
     EAP library function can be called.

  8. The EAP signals of type SshEapSignal are all generated by function
     calls into the EAP library, either via a SshEapConnection instance or
     a SshEap instance. The EAP signals of type SshEapAsyncSignal are
     generated by asynchronous events, such as timeouts.

  9. An SshEap instance is acting either as an authenticator, or as a peer,
     using RFC 3748 terminology. This mode is set at creation, and cannot
     be changed.


  * Signals And Link Authentication *

  The basic interplay of signals and link authentication state is as follows:

  1. Initially, the peer must assume that the link is unauthenticated.

  2. The authenticator will begin the authentication with an EAP Identity
     message, which when received by the peer, will result in an
     SSH_EAP_SIGNAL_NEED_TOKEN signal with SshEapTokenType
     SSH_EAP_TOKEN_USERNAME. The peer should respond with its identity
     supplied in the token. The authenticator will then receive the peer's
     identity via a SSH_EAP_SIGNAL_IDENTITY signal. The peer can now assume
     that either:

     a. No authentication method is in use, and merely authenticating with
        a user name is sufficient. If this is the case, the peer must assume
        that authentication is OK.

     b. An authentication protocol will commence, and authentication has not
        yet happened.

  3. The authentication phase (if the identity response is not sufficient),
     will then begin. The phase may include several request/response packets,
     and the peer may have an idea of when the phase ends (after it has sent
     the "final" response). The SshEap peer instance will then emit a
     SSH_EAP_SIGNAL_AUTH_PEER_MAYBE_OK signal.

  4. The authenticating peer then must respond with a EAP Success or Failure
     packet, but they are not retransmitted. Upon reception, the SshEap
     instance will emit either a SSH_EAP_SIGNAL_AUTH_PEER_OK or
     SSH_EAP_SIGNAL_AUTH_FAIL_REPLY signal.

  5. The Success and Failure packets are not resent, and RFC 3748 explicitly
     forbids resending of EAP response packets, and hence if the underlying
     transport is unreliable the Success or Failure packets may be lost.

     The SshEap peer instance, must therefore, when operating over an
     unreliable transport, assume that the link is authenticated after it
     receives the SSH_EAP_SIGNAL_AUTH_PEER_MAYBE_OK signal (case 2b) or
     after it has called ssh_eap_send_identification_reply (case 2a),
     until it receives contradicting information (either from the EAP
     library or from an external source).

  @copyright
  Copyright (c) 2002 - 2007 SFNT Finland Oy, all rights reserved.

*/

#ifndef SSH_EAP_H
#define SSH_EAP_H 1

#include "sshbuffer.h"

#ifdef SSHDIST_RADIUS
#include "sshradius.h"
#endif /* SSHDIST_RADIUS */



#include "sshcrypt.h"
#ifdef SSHDIST_CERT
#include "cmi.h"
#endif /* SSHDIST_CERT */

/* --------------------------------------------------------------------*/
/*  EAP assigned numbers                                               */
/* --------------------------------------------------------------------*/

/*  EAP types for requests and replies. One of these values is passed
    as the "type" parameter to the SshEapSignalCB callback. See
    the IETF RFC 2284 document, section 3 for details. */

#define SSH_EAP_TYPE_NONE 		 0 /** Unspecified. */
#define SSH_EAP_TYPE_IDENTITY 		 1 /** Query peer identity. */
#define SSH_EAP_TYPE_NOTIFICATION 	 2 /** Displayable message. */
#define SSH_EAP_TYPE_NAK 		 3 /** Unacceptable auth type. */
#define SSH_EAP_TYPE_MD5_CHALLENGE 	 4 /** PPP Chap challenge. */
#define SSH_EAP_TYPE_OTP 		 5 /** OTP challenge. */
#define SSH_EAP_TYPE_GENERIC_TOKEN_CARD  6 /** Card reference. */
#define SSH_EAP_TYPE_TLS                13 /** TLS */
#define SSH_EAP_TYPE_SIM 		18 /** 2G SIM authentication. */



#define SSH_EAP_TYPE_AKA 		23 /** 3G AKA authentication. */
#define SSH_EAP_TYPE_MSCHAP_V2 		26 /** MS CHAP challenge. */
#ifdef SSHDIST_EAP_AKA_DASH
#define SSH_EAP_TYPE_AKA_DASH 		50 /** 3G AKA' authentication. */
#endif /* SSHDIST_EAP_AKA_DASH */
/* --------------------------------------------------------------------*/
/*  Handles 	                                                      */
/* --------------------------------------------------------------------*/

/** The SshEap object is the concrete representation of an EAP
    instance.

    The EAP instance binds together the individual EAP protocols, their
    configuration and the I/O through the SshEapConnection. */
typedef struct SshEapRec *SshEap;

/** The SshEapConnection object abstracts away the mechanism used to pass
    packets across the external interface of the EAP instance.

    SshEapConnection is currently quite simple, and supports only
    callback based passing of SshBuffers. */
typedef struct SshEapConnectionRec *SshEapConnection;

/* --------------------------------------------------------------------*/
/*  Return values                                                      */
/* --------------------------------------------------------------------*/

/** Type used as a return value for several EAP library function
    calls which configure an SshEap instance.

    Packet and token handling errors are generally reported via
    signals, while configuration errors are reported via return
    values. */
typedef enum {
  /** Magic value not in use. */
  SSH_EAP_OPSTATUS_NONE = 0,
  /** The operation was succesful. */
  SSH_EAP_OPSTATUS_SUCCESS = 1,
  /** Insufficient resources. */
  SSH_EAP_OPSTATUS_INSUFFICIENT_RESOURCES = 2,
  /** EAP type not supported. */
  SSH_EAP_OPSTATUS_UNKNOWN_PROTOCOL = 3,
  /** General failure. */
  SSH_EAP_OPSTATUS_FAILURE = 4
} SshEapOpStatus;

/* --------------------------------------------------------------------*/
/*  Authentication tokens                                              */
/* --------------------------------------------------------------------*/


/** The SshEapTokenType denotes the type of an authentication token.

    A token used by the EAP library is an opaque data structure,
    which is passed around in SshBuffer's back and forth across the
    interface of the EAP library. The authentication token is not
    serializable as such, but contains pointers to external
    structures, such as strings. */
typedef enum {
  /** Invalid type. */
  SSH_EAP_TOKEN_NONE = 0,

  /** A shared secret associated with a certain user identification. */
  SSH_EAP_TOKEN_SHARED_SECRET = 1,

  /** A 32-bit counter value, must be incremented after each call and
      saved across instantiations of SshEap objects - the counter is
      assumed to be associated with a certain identification, and
      hence counters for different identifications should be
      separate. */
  SSH_EAP_TOKEN_COUNTER32 = 2,

  /** A SALT value to be used in authentication. */
  SSH_EAP_TOKEN_SALT = 3,






  /** A username. */
  SSH_EAP_TOKEN_USERNAME = 5,


#ifdef SSHDIST_EAP_SIM

  /** A SIM token to request the SIM to calculate Kc and SRES values
      from SIM using RAND values as input - when the EAP library requests
      for the SRES and Kc values, the input buffer is filled with 2 - 3
      RAND values; they are filled in the buffer of multiple of 16 octets;
      from the count of octets, one can calculate the number of RAND values.

     <CODE>
      -------------------------------------------------------
      | RAND1 16 octets | RAND2 16 octets | RAND3 16 octets |
      -------------------------------------------------------
     </CODE>

      As when returning the token to the library, the token data
      length is a multiple of 12 bytes (4B of SRES and 8B of KC). So the
      buffer looks like the following:

     <CODE>
      ---------------------------------
      | SRES1 4 octets | Kc1 8 octets |
      ---------------------------------
      | SRES2 4 octets | Kc2 8 octets |
      ---------------------------------
      | SRES3 4 octets | Kc3 8 octets |
      ---------------------------------
     </CODE>

     */
  SSH_EAP_TOKEN_SIM_CHALLENGE = 6,

#endif /* SSHDIST_EAP_SIM */


#ifdef SSHDIST_EAP_AKA
  /** An AKA token to request for calculating IK, CK and RES values -
      this signal is also used as a return value when the AUTN parameter was
      correctly verified. The input (RAND and AUTN) are encoded as follows:

     <CODE>
      -----------------------------------
      | RAND 16 octets | AUTN 16 octets |
      -----------------------------------
     </CODE>

     When returning the token to the library, the buffer is in the following
     format:

     <CODE>
      ---------------------------------------
      |              IK 16 octets           |
      ---------------------------------------
      |              CK 16 octets           |
      ---------------------------------------
      | RES_LEN 1 octet  | RES from 4 to    |
      ---------------------------------------
      | 16 octets   ....................... |
      ---------------------------------------
    </CODE>

      This token is returned only when the input AUTN parameter
      verification was successful, otherwise the token below is
      returned with an AUTS value.
  */
  SSH_EAP_TOKEN_AKA_CHALLENGE = 7,

  /** This token is used only towards the EAP library. This is a response
      for the AKA challenge token - when the AUTN value verification failed
      and synchronisation with the server is required, see above. In this
      case the returning token is in the following format:

     <CODE>
      ---------------------------------------
      |            AUTS 16 octets           |
      ---------------------------------------
    </CODE>
  */
  SSH_EAP_TOKEN_AKA_SYNCH_REQ = 8,

  /** AKA challenge reply when the AUTN verification fails and
      the authentication is rejected - no return buffer is required;
      this token causes the AKA layer to send an
      Authentication-Reject message to the server.
      */
  SSH_EAP_TOKEN_AKA_AUTH_REJECT = 9,
#endif /* SSHDIST_EAP_AKA */

  /** A private key used for various authentication methods.
      The private key must be returned as an SshPrivateKey type.  */ 
  SSH_EAP_TOKEN_PRIVATE_KEY =  10,

  /** Certificate authorities needed e.g. in EAP-TLS. */
  SSH_EAP_TOKEN_CERTIFICATE_AUTHORITY = 11,

#ifdef SSHDIST_EAP_AKA_DASH
  /** AKA-DASH challenge receives the network name from the
      AT_KDF_INPUT attribute. This network name contains an UTF-8
      string. This string is constructed as per the 3GPP [TS 24.302] for
      "Access Network Identity". This network name is verified with the
      access technology list maintained by the PolicyManager.

      Input to the PM will be "network_name" and "network_name_len".
      Output from PM to library will be Boolean type value either TRUE
      or FALSE
      */
  SSH_EAP_TOKEN_AKA_DASH_KDF_INPUT = 12
#endif /* SSHDIST_EAP_AKA_DASH */
} SshEapTokenType;

/** The SshEapToken structure should be considered as opaque by the
    user of the library, and should not be manipulated
    directly.

    Instead the methods below should be used. The structure is
    provided, so that the user can instantiate it simply on the stack,
    set the appropriate values, and pass it as a parameter to
    ssh_eap_token without having to separately allocate it. */

typedef struct SshEapTokenRec
{
  SshEapTokenType type;

  union {
    struct {
      void* dptr;
      unsigned long len;
    } buffer;
    struct SshMPIntRec *number;
    struct  {
      SshPrivateKey private_key;
      unsigned char *id_data;
      size_t id_data_size;
    } prvkey;
    SshUInt32 counter32;
    unsigned char **cas;
    Boolean success;
  } token;
} *SshEapToken, SshEapTokenStruct;




/** This function returns the authentication token type from an
    SshBuffer containing an SshEapToken instance, such as provided in
    association with a SSH_EAP_NEED_TOKEN signal. */
SshEapTokenType
ssh_eap_get_token_type_from_buf(SshBuffer buf);

/** This function returns the token type from an authentication
   token. */
SshEapTokenType
ssh_eap_get_token_type(SshEapToken t);

/** This function gets the token data buffer pointer and length from
    an SshBuffer containing an SshEapToken structure. */
void
ssh_eap_get_token_data_from_buf(SshBuffer buf, void **data, size_t *data_len);

/** This function returns the token data buffer pointer and length
   from an SshEapToken structure. */
void
ssh_eap_get_token_data(SshEapToken t, void **data, size_t *data_len);

/** This function sets type of the token to "type" and the
    authentication secret token in buf to point to the string "ptr" of
    length "len".

    The token points directly to the string, and hence the pointer
    must be valid until the token is no longer used. */
void
ssh_eap_init_token_typed(SshEapToken t,
			 SshEapTokenType type,
			 SshUInt8 *ptr,
			 unsigned long len);

/** This function sets the authentication token in buf to hold the
    32-bit counter value val. */
void
ssh_eap_init_token_counter32(SshEapToken t, SshUInt32 val);

/** This function sets the authentication secret token in buf to point
    to the string "ptr" of length "len".

    The token points directly to the string, and hence the pointer
    must be valid until the token is no longer used. This sets the
    SshEapTokenType of the token to SSH_EAP_TOKEN_SHARED_SECRET. */
void
ssh_eap_init_token_secret(SshEapToken t, SshUInt8 *ptr, unsigned long len);

/** This function sets the authentication token salt in buf to point
    to the string "ptr" of length "len".

    The token points directly to the string, and hence the pointer
    must be valid until the token is no longer used. This sets the
    SshEapTokenType of the token to SSH_EAP_TOKEN_SALT. */
void
ssh_eap_init_token_salt(SshEapToken t, SshUInt8 *ptr, unsigned long len);

/** This function sets the authentication username token in buf to
    point to the string "ptr" of length "len".

    The token points directly to the string, and hence the pointer
    must be valid until the token is no longer used. This sets the
    SshEapTokenType of the token to SSH_EAP_TOKEN_USERNAME. */
void
ssh_eap_init_token_username(SshEapToken t, SshUInt8 *ptr, unsigned long len);






void
ssh_eap_init_token_certificate_authority(SshEapToken t, unsigned char **ca);

void
ssh_eap_init_token_private_key(SshEapToken t, SshPrivateKey prvkey,
                               unsigned char *id_data, size_t id_data_size);

#ifdef SSHDIST_EAP_AKA_DASH
/** This function sets the result of "verification of kdfinput" to
    "success" field of token */
void
ssh_eap_init_token_kdf_input(SshEapToken t, Boolean value);

/** This function retrieves the verification result from the token */
Boolean
ssh_eap_get_token_kdf_input(SshEapToken t);
#endif /* SSHDIST_EAP_AKA_DASH */

/** This function initializes an authentication token. */
void
ssh_eap_init_token(SshEapToken t);

/** This function uninitializes an authentication token, and frees all
    buffers it contains, but not the SshEapTokenStruct itself. */
void
ssh_eap_uninit_token(SshEapToken t);

/** This function uninitializes and frees an authentication token. */
void
ssh_eap_free_token(SshEapToken t);

/** This function creates a new token with no specified content.

    @return
    If this fails due to insufficient resources, the function returns
    NULL. */
SshEapToken
ssh_eap_create_token(void);

/** This function allocates a new SshEapToken, and copies all content
    from SshEapToken src.

    All content data in the produced SshEapToken is stored in new
    allocated buffers, with no references to "src".

    @return
    If insufficient resources are available, the function returns
    NULL. */

SshEapToken
ssh_eap_dup_token(SshEapToken src);











































/*  Bit masks for MAC and hash algorithms. */
#define SSH_EAP_TRANSFORM_PRF_HMAC_SHA1    0x00000001 /** 160 bit key. */
#define SSH_EAP_TRANSFORM_PRF_HMAC_SHA256  0x00000002 /** 256 bit key.*/


/* --------------------------------------------------------------------*/
/*  Parameters for individual authentication protocols                 */
/* --------------------------------------------------------------------*/



























#ifdef SSHDIST_EAP_TLS
typedef struct SshEapTlsParamsRec
{
#ifdef SSHDIST_CERT
  /** A handle to the certificate manager */
  SshCMContext cm;

  SshUInt32 crl_check_pol;
#endif /* SSHDIST_CERT */
}SshEapTlsParamsStruct, *SshEapTlsParams;
#endif /* SSHDIST_EAP_TLS */

#ifdef SSHDIST_EAP_AKA
typedef struct SshEapAkaParamsRec
{
  /** transform for the EAP-AKA or EAP-AKA-DASH */
  SshUInt32 transform;
#ifdef SSHDIST_EAP_AKA_DASH  
  /* PM will decide whether to verify the network name
     obtained from AT_KDF_INPUT or not */
  Boolean verify_kdfinput;
#endif /* SSHDIST_EAP_AKA_DASH */
}SshEapAkaParamsStruct, *SshEapAkaParams;
#endif /* SSHDIST_EAP_AKA || SSHDIST_EAP_AKA_DASH */

/** This function is used for providing parameters to an authentication
    protocol.

    @param eap
    The authentication protocol.

    @param type
    The type for the parameter presented by 'ptr'.

    @param ptr
    The parameter to be provided to the authentication protocol.

    @param len
    The length of the parameter.

    */
SshEapOpStatus
ssh_eap_configure_protocol(SshEap eap, SshUInt8 type, void *ptr,
                           unsigned long len);

/* --------------------------------------------------------------------*/
/*  Signals                                                            */
/* --------------------------------------------------------------------*/


/** The SshEapSignal type contains the names for the individual
    signals passed from the EAP instance across the external interface
    to a SshEapSignalCB callback. These signals denote events
    generated by function calls into the EAP library. An SshBuffer buf
    may be associated with a signal to carry a message associated with
    the signal.

    These signals can also be delivered "late", meaning that the
    signal is delivered after the callback has returned. This happens
    most notably when the RADIUS support of the ssheap library is
    used. */

typedef enum SshEapSignalEnum {
  /** This type of signal is never passed, reception of it signals a
      fatal error condition. */
  SSH_EAP_SIGNAL_NONE = 0,

  /** This signal signifies that an authenticator EAP instance has
      accepted the authentication of the peer - the type of the protocol
      used for authentication is specified with the "type" parameter
      of the SshEapSignalCB function; the SshEap instance expects no
      specific action to be taken by the recipient of this signal. */
  SSH_EAP_SIGNAL_AUTH_AUTHENTICATOR_OK = 1,

  /** This signal signifies that the peer has received an EAP Success
      message, and therefore the authentication has accepted the
      authentication; the SshEap instance expects no specific action
      to be taken by the recipient of this signal - note that the peer
      does not necessarily receive this message even if the
      authentication is successful. */
  SSH_EAP_SIGNAL_AUTH_PEER_OK = 2,

  /** This signal denotes either the sending or reception of an EAP
      failure message - for an authenticator this means the peer sent
      an incorrect "response" to a "request" message and for the peer
      that it received the EAP failure message and should assume that
      authentication failed; The SshEap instance expects no specific
      action to be taken by the recipient of this signal. */
  SSH_EAP_SIGNAL_AUTH_FAIL_REPLY = 3,

  /** SSH_EAP_SIGNAL_AUTH_FAIL_NEGOTIATION specifies that
      authentication failed because no authentication protocol is
      configured that would be acceptable to both the authenticator
      and peer - this signal can be received only by the authenticator;
      the SshEAP instance expects no specific action to be taken by
      the recipient of this signal. */
  SSH_EAP_SIGNAL_AUTH_FAIL_NEGOTIATION = 4,

  /** SSH_EAP_SIGNAL_IDENTITY signals that the SshEap instance has
      received an EAP identity reply - this signal can only be
      received by an authenticator instance; the buffer associated
      with the signal will contain a UTF-8 encoded representation of
      the identity of the peer; an SshEap authenticator instance will
      lie dormant after this signal has been emitted; a call to
      ssh_eap_send_notification_request, ssh_eap_authenticate,
      ssh_eap_reset or ssh_eap_destroy may be appropriate, depending
      on the situation. */
  SSH_EAP_SIGNAL_IDENTITY = 5,

  /** The SSH_EAP_SIGNAL_NOTIFICATION signal signals that the SshEap
      instance has received either a EAP notification request or
      response - if the SshEap instance is configured as a peer, then
      the SshEAP instance expects no action from the peer; if a buffer
      buf passed with the callback is provided, it will contain a
      user-readable message which should be displayed to the user; a
      SshEap authenticator instance will lie dormant after this signal
      has been emitted; a call to ssh_eap_send_notification_request,
      ssh_eap_authenticate, ssh_eap_reset or ssh_eap_destroy may be
      appropriate, depending on the situation. */
  SSH_EAP_SIGNAL_NOTIFICATION = 6,

  /** The SSH_EAP_SIGNAL_NEED_TOKEN signal signals that the SshEap
      instance requires a token to proceed with authentication - this
      signal may be received both by a SshEap authenticator or peer
      instance; they both expect a call to ssh_eap_token to proceed
      with authentication; an authenticator can also call
      ssh_eap_authenticate with parameter SSH_EAP_AUTH_SUCCESS or
      SSH_EAP_AUTH_FAILURE; the SshBuffer buf passed with the signal
      contains an opaque structure, which defines the type of token
      required; the type can be obtained using the
      ssh_eap_get_token_type_from_buf function; the buffer will not
      be valid after after the call completes. */
  SSH_EAP_SIGNAL_NEED_TOKEN = 7,

  /** The SSH_EAP_SIGNAL_AUTH_PEER_MAYBE_OK signal signals that
      authentication may have completed - this signal is only emitted
      by a peer instance, and the recipient should assume that
      authentication has succeeded, unless the SshEapConnection is
      operating over a reliable transport. */
  SSH_EAP_SIGNAL_AUTH_PEER_MAYBE_OK = 8,

  /** The SSH_EAP_SIGNAL_AUTH_FAIL_AUTHENTICATOR signal signals that
      the peer has failed in authenticating the server - this signal is
      only delivered by a peer instance and only if a protocol which
      performs mutual authentication (such as SRP) is used. */
  SSH_EAP_SIGNAL_AUTH_FAIL_AUTHENTICATOR = 9,

  /** The SSH_EAP_SIGNAL_PACKET_DISCARDED signal signals that a packet
      has been discarded by the library - the packet is provided in the
      parameter buffer. */
  SSH_EAP_SIGNAL_PACKET_DISCARDED = 10,

  /** The SSH_EAP_SIGNAL_TOKEN_DISCARDED signal denotes that a token
      has been discarded by the library, without the library acting on
      the token. */
  SSH_EAP_SIGNAL_TOKEN_DISCARDED = 11,

  /** The SSH_EAP_SIGNAL_AUTH_FAIL_USERNAME signal denotes that the
      authentication failed due to the username being unknown - the
      username is provided in the parameter buffer. */
  SSH_EAP_SIGNAL_AUTH_FAIL_USERNAME = 12,

  /** The SSH_EAP_SIGNAL_AUTH_OK_USERNAME signal denotes that the
      authentication succeeded based on username only - the username is
      provided in the parameter buffer. */
  SSH_EAP_SIGNAL_AUTH_OK_USERNAME = 13,

  /** The SSH_EAP_FATAL_ERROR signal denotes that the EAP library has
      encountered a fatal error, such as an "out of memory" error - this
      implies that it is highly unlikely that the authentication
      protocol can be executed correctly, and hence the SshEap instance
      should be terminated. */
  SSH_EAP_SIGNAL_FATAL_ERROR = 14,

  /** The SSH_EAP_SIGNAL_AUTH_FAIL_TIMEOUT signal denotes the
      authentication failed due to a timeout - if the SshEap instance
      is an authenticator, this means that after the specified amount
      of resends (num_retransmit in SshEapConfiguration) at the
      specified interval of (retransmit_delay_sec in
      SshEapConfiguration) no response was received or the total
      authentication timeout has elapsed, without authentication
      completing (num_auth_timeout in SshEapConfiguration); if the
      SshEap instance is a peer, this means that either the maximum
      amount of resends to a certain request were sent (num_retransmit
      in SshEapConfiguration) or the total authentication timeout has
      elapsed; this signal may be received by both authenticator or
      peer; the SshEAP instance expects no specific action to be
      taken by the recipient of this signal. */
  SSH_EAP_SIGNAL_AUTH_FAIL_TIMEOUT = 15
  
} SshEapSignal;


/** The SshEapSignalCB delivers the signal "signal" from EAP
    instance "eap" and authentication protocol "protocol". The signal
    is generated by a function call into the EAP library (either via a
    SshEapConnection or SshEap instance).

    The EAP instance calling the callback owns the buffer, and may
    free it after the callback returns.  The EAP functions a recipient
    of this callback can call are limited, please see parameter
    descriptions for further information.

    @param eap
    SshEap eap is the instance passing the signal.

    @param type
    The EAP type which has caused the signal (may be
    SSH_EAP_TYPE_NONE).

    @param signal
    The signal to be delivered.

    @param buf
    If the signal carries with it an associated message, this is
    contained in SshBuffer buf. If no message is attached, then buf is
    NULL. SshBuffer buf contains any additional information

    @param ctx
    void* ctx is the callback context pointer passed to the
    ssh_eap_create_* functions.

    */
typedef void (*SshEapSignalCB)(SshEap eap,
			       SshUInt8 type,
			       SshEapSignal signal,
			       SshBuffer buf,
			       void *ctx);

#ifdef SSHDIST_RADIUS
/** Callback type to use for parsing RADIUS replies.

    This function is called from the SshRadiusClientRequestCB.

    The reply AVP's can be traversed using the appropriate functions
    from sshradius.h. This function is not called, if authentication would
    fail anyway.

    @return
    If this function returns FALSE, then the authentication will be
    failed (and appropriate signals sent) and if it returns TRUE, then
    authentication succeeds.

    */
typedef Boolean (*SshEapRadiusRequestCB)(SshEap eap,
                                         SshRadiusClientRequestStatus status,
                                         SshRadiusClientRequest request,
                                         SshRadiusOperationCode reply_code,
                                         void *context);
#endif /* SSHDIST_RADIUS */

/** The SshEapConfiguration object is a concrete structure created by
    the creator of the SshEap object, and can be shared among several
    EAP instances.

    The SshEapConfiguration contains parameters for the EAP object,
    which are common to all EAP protocols. The caller should create
    this object using ssh_eap_config_create and free it using
    ssh_eap_config_destroy, but is free to manipulate it's contents
    directly. */
typedef struct SshEapConfigurationRec
{
  /** num_retransmit is the amount of times a request or response is
      retransmitted, before it is assumed that the link is not working
      and there is a timeout - this is effective for both a peer or
      authenticating instance; for a peer instance, this implies the
      amount of times it is willing to respond to a resent EAP
      request, before signaling failure; if set to 0, then no
      retransmits will be sent or accepted. */
  unsigned long num_retransmit;

  /** retransmit_delay_sec is the delay in seconds before a request is
      retransmit, if a response is not received - this is effective
      only for an authenticating instance; if 0, then never
      retransmit. */
  unsigned long retransmit_delay_sec;

  /** Restart authentication after re_auth_delay_sec seconds - if 0,
      then never restart. */
  unsigned long re_auth_delay_sec;

  /** The time during which an authentication process has to complete,
      before it is timed out - this timeout is effective only for an
      authenticator and peer; the timer starts running from the first
      request sent or received, and stops after authentication has
      succeeded or failed; for a peer instance the timer also stops
      running at the emission of a SSH_EAP_SIGNAL_AUTH_PEER_MAYBE_OK
      signal */
  unsigned long auth_timeout_sec;

  /** The callback for passing signals and messages from the SshEap
      instance. */
  SshEapSignalCB signal_cb;

#ifdef SSHDIST_RADIUS
  /** This flag denotes that RADIUS can be supported on a per-user
      basis - if this flag is set, then after the reception of an EAP
      Identity Reply the message will be internally buffered until
      ssh_eap_continue; this allows the sender to configure RADIUS
      after receiving the identity and before calling
      ssh_eap_authenticate. */
  Boolean radius_buffer_identity;

  /** Callback function that is called for additional parsing of
      RADIUS requests, if so desired - this function is placed here
      (and not in SshEapRadiusConfiguration), as the purpose
      SshEapConfiguration is to define the embedding of SshEap, while
      SshEapRadiusConfiguration defines merely the RADIUS client and
      servers to use. */
  SshEapRadiusRequestCB radius_req_cb;
#endif /* SSHDIST_RADIUS */

  /** Reference count, for sanity checking only. */
  int refcount;
} SshEapConfigurationStruct, *SshEapConfiguration;

/** This function allocates an EAP configuration structure and
    initializes all the values to defaults.

    The structure should be freed using ssh_eap_params_destroy.

    @return
    If insufficient resources are available, NULL is returned.

    */
SshEapConfiguration
ssh_eap_config_create(void);

/** This function frees the SshEapConfiguration object previously
    allocated using ssh_eap_config_create */
void
ssh_eap_config_destroy(SshEapConfiguration params);



/* --------------------------------------------------------------------*/
/*  RADIUS                                                             */
/* --------------------------------------------------------------------*/

#ifdef SSHDIST_RADIUS

/** Structure used to pass RADIUS configuration to the EAP
    instance. */
typedef struct SshEapRadiusConfigurationRec
{
  /** The RADIUS client to use - this field MUST be valid.  */
  SshRadiusClient radius_client;

  /** Configuration of RADIUS servers to use for the client - this
      field MUST be valid. */
  SshRadiusClientServerInfo radius_servers;

  /** Default AVP's for EAP requests */
  SshRadiusUrlAvpSet default_avps;

  /** Should we ignore the Session-Timeout AVP contrary to RFC 2869. */
  Boolean ignore_radius_session_timeout;
} *SshEapRadiusConfiguration, SshEapRadiusConfigurationStruct;


/** Pass the RADIUS configuration to the EAP instance.

    If RADIUS is desirable to use for a certain authentication round,
    this function must be called after creation of the EAP instance or
    in the signal handler for SSH_EAP_SIGNAL_IDENTITY during the
    corresponding authentication round. Most notably, attaching a
    SshEapRadiusConfiguration after an EAP Identity Request/Reply
    round has completed will cause unexpected behaviour.

    It is not guaranteed that EAP will use the same RADIUS
    configuration, if it is not attached again during/before the round
    in question. If a RADIUS server has been configured. It should be
    reset using ssh_eap_radius_attach(eap, NULL).

    The EAP instance may keep referencing the RADIUS server until it
    is detached as shown above, the instance is destroyed or
    ssh_eap_reset is called.
    */
void
ssh_eap_radius_attach(SshEap eap,
                      SshEapRadiusConfiguration radius_config);

/** This function inputs the peer's identity to an EAP authenticator
    and sends the identity to the RADIUS server, thus starting the
    authentication round.

    The function should be called when the authenticator already knows
    the peer's identity, and the EAP Identity Request/Reply round can
    be skipped. This function has no effect when called for EAP peers.

    @param ptr
    The buffer 'ptr' contains a UTF-8 encoded representation of the
    identity of the peer.

    @param len
    The length of the buffer.

    */
void
ssh_eap_radius_input_peer_identity(SshEap eap,
				   SshUInt8 *ptr, unsigned long len);

#endif /* SSHDIST_RADIUS */
/* --------------------------------------------------------------------*/
/*  EAP Connection                                                     */
/* --------------------------------------------------------------------*/

/** A callback of the type SshEapConnectionOutputCB is used to pass an
    SshBuffer through an SshEapConnection object from an SshEap
    object.

    Note: The recipient MUST NOT modify or free the SshBuffer buf.

    @param con
    The connection used.

    @param ctx
    Contains the callback context value specified as the ctx parameter
    to ssh_eap_connection_create_cb.

    @param buf
    SshBuffer buf will contain an EAP packet as defined in RFC 3748.
    */
typedef void (*SshEapConnectionOutputCB)(SshEapConnection con,
                                         void *ctx,
                                         const SshBuffer buf);

/** Creates an SshEapConnection instance that represents an interface
    using function calls (and callbacks) to pass packets across it.

    @param cb
    Defines the function which is called with a packet to output.

    @param ctx
    Defines the context pointer passed to that cb along with the
    buffer containing the packet.

    @return
    If insufficient resources are available, NULL is returned. */
SshEapConnection
ssh_eap_connection_create_cb(SshEapConnectionOutputCB cb, void *ctx);

/** Cleans up and frees an SshEapConnection object previously
    allocated and created using ssh_eap_connection_create. */

void
ssh_eap_connection_destroy(SshEapConnection con);

/** Sets the maximum request unit size for the SshEapConnection.

    The EAP instance may use this value in constructing its packets.
    The EAP instance must be able to send packets of at least 1500
    bytes in size, even if this MRU is set lower. */
void
ssh_eap_connection_set_mru(SshEapConnection con, unsigned long mru);

/** This function is used to pass a packet to the SshEapConnection
    instance.

    The packet is contained in the SshBuffer buf, and the EAP
    connection will reference that buffer only until the callback
    returns. The ssh_eap_connection_input_packet will not free or
    modify the buffer. SshBuffer buf is assumed to contain one EAP
    packet as defined in RFC 3748. */
void
ssh_eap_connection_input_packet(SshEapConnection con, SshBuffer buf);

/** Disables an SshEapConnection object, forcing it to drop all packets
    until ssh_eap_connection_enable is called. */
void
ssh_eap_connection_disable(SshEapConnection con);

/** Enables again a SshEapConnection object, after it has been
    disabled using ssh_eap_connection_disable. */
void
ssh_eap_connection_enable(SshEapConnection con);

/* --------------------------------------------------------------------*/
/*  SshEap                                                             */
/* --------------------------------------------------------------------*/

/** Allocate and initialize either an EAP authenticator instance.

    The 'config' object may be shared by several EAP instances, but an
    SshEapConnection object may not.

    If a SshEap instance has not been configured to support any
    authentication protocols, it will fail all authentications.

    @see ssh_eap_accept_auth

    @return
    If insufficient resources are available, NULL is returned.

    @param ctx
    The callback context pointer to be used in
    message/signal-passing calls.

    @param config
    An SshEapConfiguration object - must be defined (not NULL), the
    instance cannot perform without this being defined sensibly.

    @param con
    An SshEapConnection object - must be defined (not NULL), the
    instance cannot perform without this being defined sensibly.

    */

SshEap
ssh_eap_create_server(void *ctx, SshEapConfiguration config,
                      SshEapConnection con);

/** Allocate and initialize an EAP peer instance.

    The 'config' object may be shared by several EAP instances,
    but an SshEapConnection object may not.

    If a SshEap instance has not been configured to support any
    authentication protocols, it will fail all authentications.

    @param ctx
    The callback context pointer to be used in message/signal-passing
    calls.

    @param config
    An SshEapConfiguration object - must be defined (not NULL), the
    instance cannot perform without this being defined sensibly.

    @param con
    An SshEapConnection object - must be defined (not NULL), the
    instance cannot perform without this being defined sensibly.

    @see ssh_eap_accept_auth

    @return
    If insufficient resources are available, NULL is returned.
    */
SshEap
ssh_eap_create_client(void *ctx, SshEapConfiguration config,
                      SshEapConnection con);

/** Destroy all the resources used up by an SshEap instance, and
    also cancel all of it's internal functions, e.g. timeouts, etc.

    The ssh_eap_destroy function may be called at anytime
    (after the reception of any signal or from any callback). */
void
ssh_eap_destroy(SshEap eap);

/** Reset the state of the SshEap instance and all
    protocols associated with it.

    Note: This does not remove the protocols configured with the
    ssh_eap_accept_auth* functions from the SshEap instance.
    To do that, use ssh_eap_accept_none.
    */
SshEapOpStatus
ssh_eap_reset(SshEap eap);

/** Specify that an SshEap instance should use the EAP
    authentication method identified by the EAP type "type" and with
    the indicated preference.

    The SshEap authenticator cycles through all configured protocols
    in the order of greatest preference fist, until it finds a
    suitable protocol when attempting to authenticate.

    @param eap
    The instance.

    @param type
    The EAP authentication method.

    @param preference
    The preference denotes in which order a server is to try all the
    authentication protocols it knows, until it finds a suitable one.

    */
SshEapOpStatus
ssh_eap_accept_auth(SshEap eap, SshUInt8 type, SshUInt8 preference);

/** Reset the authentication state associated with an
    SshEAP instance to no protocols accepted. */
void
ssh_eap_accept_auth_none(SshEap eap);

/** States for an EAP instance in packet/callback processing */
typedef enum {
  /** Illegal value */
  SSH_EAP_STATUS_NONE = 0,

  /** The EAP instance is "idle", waiting for packets or functions to
     trigger some action */
  SSH_EAP_STATUS_IDLE = 1,

  /** The EAP instance is waiting for a callback to proceed. */
  SSH_EAP_STATUS_WAITING = 2,

  /** The EAP instance is currently processing a packet or a token. */
  SSH_EAP_STATUS_PROCESSING = 3,

  /** The EAP instance is dying down. */
  SSH_EAP_STATUS_DYING = 4
} SshEapStatus;

/** Return the status of an EAP instance. */
SshEapStatus
ssh_eap_get_status(SshEap eap);


/* --------------------------------------------------------------------*/
/*  Identification                                                     */
/* --------------------------------------------------------------------*/

/** This function wsignals an SshEap authenticator instance (created
    using ssh_eap_create_server), to send an identification_request.

    The buffer is expected to contain a UTF-8 encoded string of length
    len (may be 0), which will be sent to the peer.  This effectively
    begins an authentication round, and also resets all EAP
    authentication protocols to an uninitialized state.

    The caller of this function will receive a SSH_EAP_SIGNAL_IDENTITY
    signal, when the peer has responded. */
void
ssh_eap_send_identification_request(SshEap eap,
                                    const unsigned char *buffer,
                                    unsigned long len);

/* --------------------------------------------------------------------*/
/*  Notification                                                       */
/* --------------------------------------------------------------------*/

/** This function is wused by the caller of a SshEap authenticator
    instance to send a UTF-8 encoded string buffer (length len bytes) to
    the peer.

    A SSH_EAP_SIGNAL_NOTIFICATION signal is sent to the
    authenticator after the response to this message has been
    received. */
void
ssh_eap_send_notification_request(SshEap eap,
                                  const unsigned char *buffer,
                                  unsigned long len);


/* --------------------------------------------------------------------*/
/*  Authentication                                                     */
/* --------------------------------------------------------------------*/

/** SshEapAuthStep is an enumeration of the parameters to
    ssh_eap_authenticate - see ssh_eap_authenticate for better
    descriptions. */
typedef enum {
  SSH_EAP_AUTH_SUCCESS = 1, 	/** Success. */
  SSH_EAP_AUTH_FAILURE = 2, 	/** Failure. */
  SSH_EAP_AUTH_CONTINUE = 3 	/** Continue. */
} SshEapAuthStep;

/** This function is used to signal an authenticator SshEap object to
    take any of the three steps: Fail the authentication, consider the
    authentication successful, or to continue with authenticating the
    peer as per the protocol.

    This function may only be called for a SshEap authenticator
    instance, and it's main use is to provide user-name specific
    settings for the authentication protocol to use.

    If step is set to SSH_EAP_AUTH_SUCCESS, then the authenticator
    should consider the authentication succesful and call the
    appropriate callbacks and send the appropriate messages.

    If step is set to SSH_EAP_AUTH_FAILURE, then the authenticator
    should consider that the authentication failed, and call the
    appropriate callbacks and send the appropriate messages.

    If step is set to SSH_EAP_AUTH_CONTINUE, then the authenticator
    should proceed with the authentication as configured.

    This function can be called in response to a
    SSH_EAP_SIGNAL_IDENTITY, SSH_EAP_SIGNAL_NOTIFICATION, or
    SSH_EAP_SIGNAL_NEED_TOKEN signal. */
void
ssh_eap_authenticate(SshEap eap, SshEapAuthStep step);

/** This function function is used by callers of both authenticator
    and peer instances to pass data structures required for
    authentication to the SshEap instance in response to an
    SSH_EAP_SIGNAL_NEED_TOKEN signal. The library assumes that token
    is not valid after the call returns, and the library will not
    modify it.

    This function can be called safely from a callback such as the
    SshEapSignalCB callback. Note that in this case the SshEap
    instance must take a temporary copy of the token, as it will
    handle the token later after processing of the current
    packet/token/etc has finished (implying that the token is
    processed after ssh_eap_token returns). */
void
ssh_eap_token(SshEap eap, SshUInt8 type, SshEapToken token);

/** This function returns the derived master session key.

    The function may be called by both the authenticator and peer and
    should be called after the EAP authentication has successfully
    completed, i.e. after receiving the
    SSH_EAP_SIGNAL_AUTH_AUTHENTICATOR_OK or
    SSH_EAP_SIGNAL_AUTH_PEER_OK signal.

    If the underlying EAP method outputs a master session key, then
    this key will be returned in 'buffer', the key length is returned
    in 'buffer_len'. If the underlying EAP method does output a master
    session key, or this function is called before the master session
    key has been derived then this function returns 'buffer' as NULL.
    The caller of this function must free 'buffer' when done.
    If the underlying EAP method supports session id, the session id
    can be retrieved in id_buffer. The length of session id is 
    returned in id_buffer_len. The argument id_buffer and id_buffer_len
    can be NULL, in which case the session id is not returned. 
    Also in case when EAP method does not support session id or 
    it is not available, the id_buffer is returned as NULL. */
void
ssh_eap_master_session_key(SshEap eap, unsigned char **buffer,
			   size_t *buffer_len,
                           unsigned char **id_buffer,
                           size_t *id_buffer_len);

/* --------------------------------------------------------------------*/
/*  Routing                                                            */
/* --------------------------------------------------------------------*/

/** The SshEapDestination enum describes whether a packet is destined
    for a peer instance or an authenticator instance, in the case that
    both are operating over the same transport. */

typedef enum {
  /** Unspecified destination. */
  SSH_EAP_NONE = 0,
  /** Packet is destined to the peer. */
  SSH_EAP_PEER = 1,
  /** Packet is destined to the authenticator. */
  SSH_EAP_AUTHENTICATOR = 2
} SshEapDestination;

/** Parse an alleged EAP packet, and decides whether it is destined
    for a peer instance or an authenticator instance, or if it is not an
    EAP packet at all. The packet is of length len and is provided in the
    passed buffer. */
SshEapDestination
ssh_eap_packet_destination(SshUInt8 *buffer, unsigned long len);

/* --------------------------------------------------------------------*/
/*  Misc                                                               */
/* --------------------------------------------------------------------*/

/** Converts EAP code to US English textual description. */
const char* ssh_eap_code_to_string(SshUInt8 code);

/** Converts EAP protocol identifier to US English textual
description. */
const char* ssh_eap_protocol_to_string(SshUInt8 type);

#endif /** not SSH_EAP_H */
