/*
 * engine_random.h
 *
 * Copyright:
 *       Copyright (c) 2002, 2003 SFNT Finland Oy.
 *       All rights reserved.
 *
 * A port of the "lib/sshcrypto/sshrandom/sshrand.c" to the engine,
 * for use in operations where stronger randomness is required, such
 * as TCP sequence number randomization.  The randomness generated by
 * this pool can be leaked directly to external parties.
 */

#include "sshincludes.h"
#include "engine_internal.h"

#ifdef SSH_ENGINE_PRNG

#include "md5.h"

#define SSH_DEBUG_MODULE "SshEngineRandom"

void
ssh_engine_random_init(SshEngine engine)
{
  SshEngineRandomState state = &engine->prng;
  memset(&engine->prng, 0, sizeof(engine->prng));

  state->add_position = 0;
  state->next_available_byte = sizeof(state->stir_key);
}


void
ssh_engine_random_uninit(SshEngine engine)
{
  ;
}

void
ssh_engine_random_stir(SshEngine engine)
{
  SshUInt32 iv[4];
  size_t i;
  SshEngineRandomState state = &engine->prng;

  /* Get IV from the beginning of the pool. */
  iv[0] = SSH_GET_32BIT(state->state + 0);
  iv[1] = SSH_GET_32BIT(state->state + 4);
  iv[2] = SSH_GET_32BIT(state->state + 8);
  iv[3] = SSH_GET_32BIT(state->state + 12);

  /* Get new key. */
  memcpy(state->stir_key, state->state, sizeof(state->stir_key));

  /* First pass. */
  for (i = 0; i < SSH_ENGINE_RANDOM_STATE_BYTES; i += 16)
    {
      ssh_md5_transform(iv, state->stir_key);
      iv[0] ^= SSH_GET_32BIT(state->state + i);
      SSH_PUT_32BIT(state->state + i, iv[0]);
      iv[1] ^= SSH_GET_32BIT(state->state + i + 4);
      SSH_PUT_32BIT(state->state + i + 4, iv[1]);
      iv[2] ^= SSH_GET_32BIT(state->state + i + 8);
      SSH_PUT_32BIT(state->state + i + 8, iv[2]);
      iv[3] ^= SSH_GET_32BIT(state->state + i + 12);
      SSH_PUT_32BIT(state->state + i + 12, iv[3]);
    }

  /* Get new key. */
  memcpy(state->stir_key, state->state, sizeof(state->stir_key));

  /* Second pass. */
  for (i = 0; i < SSH_ENGINE_RANDOM_STATE_BYTES; i += 16)
    {
      ssh_md5_transform(iv, state->stir_key);
      iv[0] ^= SSH_GET_32BIT(state->state + i);
      SSH_PUT_32BIT(state->state + i, iv[0]);
      iv[1] ^= SSH_GET_32BIT(state->state + i + 4);
      SSH_PUT_32BIT(state->state + i + 4, iv[1]);
      iv[2] ^= SSH_GET_32BIT(state->state + i + 8);
      SSH_PUT_32BIT(state->state + i + 8, iv[2]);
      iv[3] ^= SSH_GET_32BIT(state->state + i + 12);
      SSH_PUT_32BIT(state->state + i + 12, iv[3]);
    }

  memset(iv, 0, sizeof(iv));

  state->add_position = 0;

  /* Some data in the beginning is not returned to aboid giving an observer
     complete knowledge of the contents of our random pool. */
  state->next_available_byte = sizeof(state->stir_key);
}

void
ssh_engine_random_add_entropy(SshEngine engine,
                              const unsigned char *buf, size_t buflen)
{
  size_t pos;
  const unsigned char *input = buf;
  SshEngineRandomState state = &engine->prng;

  pos = state->add_position;

  SSH_DEBUG(SSH_D_MIDOK, ("adding %u bytes of entropy to pool", buflen));

  while (buflen > 0)
    {
      /* Only add noise to the part of the pool that is used to key MD5. */
      if (pos >= SSH_ENGINE_RANDOM_MD5_KEY_BYTES)
        {
          pos = 0;
          ssh_engine_random_stir(engine);
        }

      state->state[pos] ^= *input;
      input++;
      buflen--;
      pos++;
    }

  state->add_position = pos;
}

Boolean
ssh_engine_random_get_bytes(SshEngine engine,
                            unsigned char *buf, size_t buflen)
{
  int i;
  SshEngineRandomState state = &engine->prng;

  for (i = 0; i < buflen; i++)
    {
      if (state->next_available_byte >= SSH_ENGINE_RANDOM_STATE_BYTES)
        ssh_engine_random_stir(engine);

      if (state->next_available_byte >= SSH_ENGINE_RANDOM_STATE_BYTES)
        {
          return FALSE;
        }

      buf[i] = state->state[state->next_available_byte++];
    }

  return TRUE;
}

SshUInt8
ssh_engine_random_get_byte(SshEngine engine)
{
  SshUInt8 tmp = 0;

  ssh_engine_random_get_bytes(engine, &tmp, 1);
  return tmp;
}

#endif /* SSH_ENGINE_PRNG */
