/*
  sshppp_thread.h

  Author: Lauri Tarkkala <ltarkkal@ssh.fi>

  Copyright:
          Copyright (c) 2002, 2003 SFNT Finland Oy.
  All Rights Reserved.
*/

#define SSH_DEBUG_MODULE "SshPppThread"

#include "sshincludes.h"
#include "sshfsm.h"
#include "sshstream.h"
#include "sshtimeouts.h"
#include "sshcrypt.h"
#include "sshinet.h"
#include "sshbuffer.h"

#ifdef SSHDIST_EAP
#include "ssheap.h"
#endif /* SSHDIST_EAP */

#include "sshppp_linkpkt.h"
#include "sshppp_events.h"
#include "sshppp.h"
#include "sshppp_config.h"
#include "sshppp_flush.h"
#include "sshppp_auth.h"
#include "sshppp_internal.h"
#include "sshppp_timer.h"
#include "sshppp_thread.h"
#include "sshppp_protocol.h"
#include "sshppp_chap.h"
#include "sshppp_lcp_config.h"
#include "sshppp_lcp.h"
#include "sshppp_ipcp_config.h"
#include "sshppp_ipcp.h"
#include "sshppp_io_stream.h"

void
ssh_ppp_thread_enter_state(SshPppState gdata, SshPppThread tdata)
{
  SshIterationStatus status;

  /* Keep us in the same state, unless later otherwise directed */
  ssh_fsm_set_next(tdata->thread,
                   ssh_fsm_get_thread_current_state(tdata->thread));

  /* The SshStream callback may not have been activated */
  tdata->is_polling = FALSE;

  if (tdata->mux != NULL)
    {
      /* inform the MUX that we are currently not waiting for a buffer */
      ssh_ppp_flush_unwait(tdata->mux);

      /* if there is a buffer pending flushing, attempt to flush it */
      ssh_ppp_flush_run(gdata, tdata->mux);
    }

  /* if we have not been boot'd yet, do nothing */
  if (tdata->thread_status == SSH_PPP_INITIAL)
    tdata->thread_status = SSH_PPP_IDLE;

  if (tdata->thread_status != SSH_PPP_RUNNING)
    return;

  /* if a previous event is unhandled, do nothing */

  if (tdata->current_event != SSH_PPP_EVENT_NONE)
    return;

  /* check events generated by callbacks from this thread.
     these must be handled first. note that even if reading
     an event from this queue also frees an output
     slot for tdata->events_output, the eventq
     may be full because of "ssh_ppp_events_reserve()".

     Thus blocking on an callback message queue
     which is full means the following:

     - All slots in the queue have been reserved
       using ssh_ppp_events_reserve()

     - We wait untill one of these slots is filled
       with a message and also unreserved

     - We read that message into tdata->current_event,
       hence freeing up a slot in the message queue
  */

  tdata->current_event = ssh_ppp_events_get(tdata->events_internal_input);

  if (tdata->current_event != SSH_PPP_EVENT_NONE
      && gdata->fatal_error == 0)
    return;

  /* check events generated by the controlling ppp thread */

  tdata->current_event = ssh_ppp_events_get(tdata->events_cb_input);

  if (tdata->current_event != SSH_PPP_EVENT_DESTROY
      && gdata->fatal_error == 1)
    tdata->current_event = SSH_PPP_EVENT_FATAL_ERROR;

  if (tdata->current_event != SSH_PPP_EVENT_NONE)
    return;

  /* check timer */

  if (tdata->timer != NULL)
    {
      if (ssh_ppp_timer_check_timeout(tdata->timer))
        {
          tdata->current_event = SSH_PPP_EVENT_TIMEOUT;
          return;
        }
    }

  /* check I/O */

  SSH_ASSERT(tdata->input_pkt == NULL);

  if (tdata->mux != NULL)
    {
      status = ssh_ppp_flush_get_pdu(gdata, tdata->mux, &tdata->input_pkt);

      /* cache I/O packet for later handling */

      if (status == SSH_PPP_OK)
        {
          return;
        }
      tdata->current_event = SSH_PPP_EVENT_NONE;
    }
  else
    {
      status = SSH_PPP_EMPTY;
    }

  /* the SshStream callback has been activated only if status is EMPTY */

  if (status == SSH_PPP_EMPTY)
    {
      tdata->is_polling = TRUE;
    }

  return;
}

SshPppPktBuffer
ssh_ppp_thread_get_input_pkt(SshPppThread tdata)
{
  return tdata->input_pkt;
}

SshPppEvent
ssh_ppp_thread_get_event(SshPppState gdata, SshPppThread tdata)
{
  /* Block all functions doing misc stuff if SSH_PPP_EVENT_NONE
     is the current event from doing stuff if we are not
     running. */

  if (tdata->thread_status != SSH_PPP_RUNNING)
    return SSH_PPP_EVENT_SUSPENDED;

  /* Block untill we can send signals if we need to */

  if (ssh_ppp_events_isfull(tdata->events_cb_output)
      || ssh_ppp_events_isfull(tdata->events_output))
    {
      tdata->is_polling = TRUE;
      return SSH_PPP_EVENT_NONE;
    }

  /* Otherwise just proceed as normal */

  if (tdata->current_event != SSH_PPP_EVENT_NONE)
    {
      SSH_DEBUG(SSH_D_MIDOK,
                ("state %p processing event %s (%d)",
                 ssh_fsm_get_thread_current_state(tdata->thread),
                 ssh_ppp_event_to_string(tdata->current_event),
                 tdata->current_event));
    }

  return tdata->current_event;
}

void
ssh_ppp_thread_boot(SshPppThread thread)
{
  SSH_DEBUG(SSH_D_LOWOK, ("booting thread %p", thread));

  if (thread->thread_status == SSH_PPP_IDLE
      || thread->thread_status == SSH_PPP_INITIAL)
    {
      thread->thread_status = SSH_PPP_RUNNING;
    }

  ssh_ppp_thread_wakeup(thread);
}

void
ssh_ppp_thread_set_event(SshPppThread tdata, SshPppEvent ev)
{
  tdata->current_event = ev;
}

void
ssh_ppp_thread_cancel_current_event(SshPppThread tdata)
{

  if (tdata->mux != NULL)
    {
      ssh_ppp_flush_return_pdu(tdata->mux, tdata->input_pkt);
    }

  tdata->current_event = SSH_PPP_EVENT_NONE;
  tdata->input_pkt = NULL;
}

SshFSMStepStatus
ssh_ppp_thread_leave_suspend(SshPppState gdata, SshPppThread tdata)
{
  tdata->is_suspended = TRUE;
  return SSH_FSM_SUSPENDED;
}

SshFSMStepStatus
ssh_ppp_thread_leave_state(SshPppState gdata, SshPppThread tdata)
{
  SshPppEvent ev = tdata->current_event;

  if (tdata->thread_status != SSH_PPP_RUNNING)
    {

      /* If we are suspended, and we have a packet waiting for us, then
         if we are not soonishly unsuspended, then the whole PPP
         composite system will BLOCK, as no more input packets will
         be processed.

         Therefore we discard the event/packet here, as if it
         had already been filtered by ssh_ppp_flush (which it
         probably has been, anyway).

         Internally generated events are not discarded.
      */

      if (tdata->input_pkt != NULL)
        {
          ssh_ppp_thread_cancel_current_event(tdata);
        }

      tdata->is_suspended = TRUE;
      return SSH_FSM_SUSPENDED;
  }

  ssh_ppp_thread_cancel_current_event(tdata);

  if (tdata->is_polling == FALSE && ev != SSH_PPP_EVENT_FATAL_ERROR)
    return SSH_FSM_CONTINUE;

  /* Callbacks which have been called during the execution
     of a state may have resulted in events being added to the
     callback events queue. If we suspend now, it may
     result in unnecessary timeouts. */

  if (!ssh_ppp_events_isempty(tdata->events_cb_input)
      && (ev != SSH_PPP_EVENT_FATAL_ERROR || tdata == gdata->ppp_thread))
    return SSH_FSM_CONTINUE;

  tdata->is_suspended = TRUE;
  return SSH_FSM_SUSPENDED;
}

void
ssh_ppp_thread_wakeup(SshPppThread tdata)
{
  ssh_fsm_continue(tdata->thread);
  tdata->is_suspended = FALSE;
}

SshPppThread
ssh_ppp_thread_create(SshPppState state,
                      SshFSMThread thread,
                      SshPppEvents eventq,
                      const char* debug_name)
{
  SshPppThread t;

  t = ssh_malloc(sizeof(*t));

  if (t == NULL)
    return NULL;

  t->thread = thread;
  t->events_cb_output = NULL;
  t->events_output = NULL;
  t->timer = NULL;
  t->mux = NULL;

  t->events_cb_input = NULL;
  t->events_internal_input = NULL;

  t->events_cb = ssh_ppp_events_create();

  if (t->events_cb == NULL)
    goto fail;

  t->events_cb_input = ssh_ppp_events_attach_input(t->events_cb,t);

  t->events_internal = ssh_ppp_events_create();

  if (t->events_internal == NULL)
    goto fail;

  t->events_internal_input =
    ssh_ppp_events_attach_input(t->events_internal,t);

  t->events_cb_output = ssh_ppp_events_attach_output(t->events_cb,t);

  if (t->events_cb_output == NULL)
    goto fail;

  if (eventq != NULL)
    {
      t->events_output = ssh_ppp_events_attach_output(eventq,t);
      if (t->events_output == NULL)
        goto fail;
    }

  /* State of thread */

  t->thread_status = SSH_PPP_INITIAL;
  t->current_event = SSH_PPP_EVENT_NONE;
  t->is_suspended = FALSE;
  t->is_polling = FALSE;
  t->input_pkt = NULL;

  return t;
 fail:
  if (t->events_output != NULL)
    ssh_ppp_events_detach_output(t->events_output, t);

  if (t->events_cb_output != NULL)
    ssh_ppp_events_detach_output(t->events_cb_output, t);

  if (t->events_internal_input != NULL)
    ssh_ppp_events_detach_input(t->events_internal_input, t);

  if (t->events_cb_input != NULL)
    ssh_ppp_events_detach_input(t->events_cb_input, t);

  ssh_free(t);
  return NULL;
}

void
ssh_ppp_thread_destroy(SshPppThread t)
{
  ssh_ppp_events_detach_input(t->events_cb_input,t);
  ssh_ppp_events_detach_output(t->events_cb_output,t);

  ssh_ppp_events_detach_input(t->events_internal_input,t);

  ssh_ppp_events_detach_output(t->events_output,t);

  ssh_free(t);
}

void
ssh_ppp_thread_attach_mux(SshPppThread t, SshPppMuxProtocolStruct* mux)
{
  SSH_ASSERT(t->mux == NULL);

  t->mux = mux;
}

void
ssh_ppp_thread_attach_timer(SshPppThread t, SshPppTimer timer)
{
  SSH_ASSERT(t->timer == NULL);

  t->timer = timer;
}

SshPppEventsInput
ssh_ppp_thread_get_inputq(SshPppThread t)
{
  return t->events_internal_input;
}

SshPppEventsInput
ssh_ppp_thread_get_cb_inputq(SshPppThread t)
{
  return t->events_cb_input;
}

SshPppEventsOutput
ssh_ppp_thread_get_outputq(SshPppThread t)
{
  return t->events_output;
}

SshPppEventsOutput
ssh_ppp_thread_get_cb_outputq(SshPppThread t)
{
  return t->events_cb_output;
}
SshPppEvents
ssh_ppp_thread_get_events(SshPppThread t)
{
  return t->events_internal;
}

SshPppMuxProtocolStruct*
ssh_ppp_thread_get_mux(SshPppThread t)
{
  return t->mux;
}

SshPppTimer
ssh_ppp_thread_get_timer(SshPppThread t)
{
  return t->timer;
}

SshFSMThread
ssh_ppp_thread_get_thread(SshPppThread t)
{
  return t->thread;
}

void
ssh_ppp_thread_set_next(SshPppThread t, SshFSMStepCB cb)
{
  ssh_fsm_set_next(t->thread,cb);
}

void
ssh_ppp_thread_suspend(SshPppThread t)
{
  t->thread_status = SSH_PPP_SUSPEND;
}

void
ssh_ppp_thread_continue(SshPppThread t)
{
  if (t->thread_status == SSH_PPP_SUSPEND)
    {
      t->thread_status = SSH_PPP_RUNNING;
      ssh_ppp_thread_wakeup(t);
    }
}

void
ssh_ppp_thread_cancel_event(SshPppThread t, SshPppEvent ev)
{
  if (t->current_event == ev)
    {
      ssh_ppp_thread_cancel_current_event(t);
    }
}
