/*
  sshppp_events.h

  Author: Lauri Tarkkala <ltarkkal@ssh.fi>
  
  Copyright:
          Copyright (c) 2002, 2003 SFNT Finland Oy.
  All Rights Reserved.
*/

#ifndef SSHPPP_EVENTS_H

#define SSHPPP_EVENTS_H

/* These events are the PPP statemachine events from RFC1661 */

#define SSH_PPP_EVENT_UP 1
#define SSH_PPP_EVENT_DOWN 2
#define SSH_PPP_EVENT_OPEN 3
#define SSH_PPP_EVENT_CLOSE 4
#define SSH_PPP_EVENT_TOPLUS 5
#define SSH_PPP_EVENT_TOMINUS 6
#define SSH_PPP_EVENT_RCRPLUS 7
#define SSH_PPP_EVENT_RCRMINUS 8
#define SSH_PPP_EVENT_RCA 9
#define SSH_PPP_EVENT_RCN 10
#define SSH_PPP_EVENT_RTR 11
#define SSH_PPP_EVENT_RTA 12
#define SSH_PPP_EVENT_RUC 13
#define SSH_PPP_EVENT_RXJPLUS 14
#define SSH_PPP_EVENT_RXJMINUS 15
#define SSH_PPP_EVENT_RXR 16

/* "Magic" events used for control within PPP */

#define SSH_PPP_EVENT_NONE 0
#define SSH_PPP_EVENT_PENDING 17
#define SSH_PPP_EVENT_FAIL 18

/* Events to be used by authentication protocols */

/* Authentication protocol succeeded in authentication all parties */
#define SSH_PPP_EVENT_AUTH_OK 19
/* Authentication protocol failed to authenticate the peer. If
   authentication of both peer and the authenticating instance
   fails, then the SSH_PPP_EVENT_AUTH_PEER_FAIL event 
   must be generated by the authentication protocol. */
#define SSH_PPP_EVENT_AUTH_PEER_FAIL 20
/* This instance failed to authenticate to the peer. */
#define SSH_PPP_EVENT_AUTH_THIS_FAIL 21

/* Internal events for authentication protocols */
#define SSH_PPP_EVENT_CHALLENGE 22
#define SSH_PPP_EVENT_RESPONSE 23
#define SSH_PPP_EVENT_SECRET 24
#define SSH_PPP_EVENT_AUTH_THIS_FAIL_RECHALLENGE 25
#define SSH_PPP_EVENT_AUTH_THIS_FAIL_CHANGEPW 26

/* Second timer for CHAP authentication protocol */
#define SSH_PPP_EVENT_RESPONSE_RESEND_PLUS 27
#define SSH_PPP_EVENT_RESPONSE_RESEND_MINUS 28

/* Unknown protocol received */
#define SSH_PPP_EVENT_BAD_PROTOCOL 29

/* A protocol machine has "halted" */
#define SSH_PPP_EVENT_ISHALT 30

/* A configure request event which requests
   the current configuration. Used to optimize away
   the unnecessary "Link Down" transition */
#define SSH_PPP_EVENT_RCRNEUTRAL 31

/* User requests passed via callbacks */
#define SSH_PPP_EVENT_DESTROY 32
#define SSH_PPP_EVENT_HALT 33

/* A protocol reject message was received */
#define SSH_PPP_EVENT_PROTOCOL_REJECT 34

/* LCP Identification codes */
#define SSH_PPP_EVENT_IDENTITY_RECV 35

/* LCP Time-Remaining codes */
#define SSH_PPP_EVENT_SEND_TIME_REMAINING 36
#define SSH_PPP_EVENT_RECV_TIME_REMAINING 37

/* Generic timeout event */
#define SSH_PPP_EVENT_TIMEOUT 38

/* Renegotiate timeout */
#define SSH_PPP_EVENT_RENEGOTIATE 39

/* Magic event which should be handled as a DOWN event,
   and then transformed into an "UP" event. */
#define SSH_PPP_EVENT_DOWN_UP 40

/* Event which signifies to do nothing */
#define SSH_PPP_EVENT_SUSPENDED 41

/* Event which signals a fatal error */
#define SSH_PPP_EVENT_FATAL_ERROR 42

/* Event which signals reception of a MS-CHAP Change Passord packet */
#define SSH_PPP_EVENT_CHANGEPW 43

/* Reconfigure the PPP session */
#define SSH_PPP_EVENT_CUSTOM 64

typedef SshUInt8 SshPppEvent;

/*
  A traditional consumer/producer (without locking ;-) mechanism for 
  transferring events between FSM threads.
  
  Callbacks external to FSM MUST NOT add events to
  this queue as they have no mechanism for blocking
  if the queue is full.
*/

/* These structures do not exist, they merely provide
   some additional safety in the form of static
   type checking */
  
typedef struct SshPppEventsInputRec *SshPppEventsInput;
typedef struct SshPppEventsOutputRec *SshPppEventsOutput;

#define SSH_PPP_EVENT_QUEUE_MAX 4

typedef struct SshPppEventsRec {
  /* The thread listening for production events */
  struct SshPppThreadRec *listener_input;

  /* The threads listening to consumption events */
  struct SshPppThreadRec **listeners_output;

  /* Index of next event to be read */
  SshInt8 event_idx;
  /* Number of events in buffer */
  SshInt8 nevents;
  /* Number of reserved slots */
  SshInt8 reserved_events;
  /* Number of threads listening to consumption events */
  SshUInt8 num_listeners_output;

  /* Buffer for holding events */
  SshPppEvent event_buffer[SSH_PPP_EVENT_QUEUE_MAX];
} SshPppEventsStruct, *SshPppEvents;

Boolean 
ssh_ppp_events_isfull(SshPppEventsOutput outq);

Boolean 
ssh_ppp_events_isempty(SshPppEventsInput inq);

void 
ssh_ppp_events_signal(SshPppEventsOutput outq, 
                      SshPppEvent event);

SshPppEvent 
ssh_ppp_events_peek(SshPppEventsInput inq);

SshPppEvent 
ssh_ppp_events_get(SshPppEventsInput inq);

void 
ssh_ppp_events_flush_output(SshPppEventsOutput outq);

void 
ssh_ppp_events_flush_input(SshPppEventsInput inq);

SshPppEventsInput 
ssh_ppp_events_attach_input(SshPppEvents eventq, 
                            struct SshPppThreadRec *ppp_thread);

SshPppEventsOutput ssh_ppp_events_attach_output(SshPppEvents evs, 
                                                struct SshPppThreadRec 
                                                *ppp_thread);

SshPppEvents ssh_ppp_events_create(void);

void ssh_ppp_events_detach_input(SshPppEventsInput inq,
                                 struct SshPppThreadRec *ppp_thread);

void ssh_ppp_events_detach_output(SshPppEventsOutput outq, 
                                  struct SshPppThreadRec *ppp_thread);

void ssh_ppp_events_reserve(SshPppEventsOutput outq);
void ssh_ppp_events_unreserve(SshPppEventsOutput outq);

char* ssh_ppp_event_to_string(SshPppEvent event);

#endif /* SSHPPP_EVENTS_H */
