/*
 *	Forwarding decision
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	$Id: br_forward.c,v 1.1.1.1 2007-05-25 06:50:00 bruce Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/if_vlan.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

/* Don't forward packets to originating port or forwarding diasabled */
static inline int should_deliver(const struct net_bridge_port *p,
				 const struct sk_buff *skb)
{
	return (skb->dev != p->dev && p->state == BR_STATE_FORWARDING);
}

static inline unsigned packet_length(const struct sk_buff *skb)
{
	return skb->len - (skb->protocol == htons(ETH_P_8021Q) ? VLAN_HLEN : 0);
}

int br_dev_queue_push_xmit(struct sk_buff *skb)
{
	/* drop mtu oversized packets except gso */
	if (packet_length(skb) > skb->dev->mtu && !skb_is_gso(skb))
		kfree_skb(skb);
	else {
		/* ip_refrag calls ip_fragment, doesn't copy the MAC header. */
		if (nf_bridge_maybe_copy_header(skb))
			kfree_skb(skb);
		else {
			skb_push(skb, ETH_HLEN);

			dev_queue_xmit(skb);
		}
	}

	return 0;
}

int br_forward_finish(struct sk_buff *skb)
{
	return NF_HOOK(PF_BRIDGE, NF_BR_POST_ROUTING, skb, NULL, skb->dev,
		       br_dev_queue_push_xmit);

}

static void __br_deliver(const struct net_bridge_port *to, struct sk_buff *skb)
{
	skb->dev = to->dev;
	NF_HOOK(PF_BRIDGE, NF_BR_LOCAL_OUT, skb, NULL, skb->dev,
			br_forward_finish);
}

struct vlan_transparent vlan_trans={0}; 

//bridge forward add flag Bruce Hsu
//we mark some pkt which needs to add vlan tags,
//and we only care about ra(X) --> eth(Y) pkt,
//the opposite will be transfer by wireless driver. (i guess...)

void br_add_flag(char *dst, struct sk_buff *skb)
{
    unsigned char *src=skb->dev->name;
    unsigned char *chk=NULL;
    char *ra_num=NULL;
    int rn=0;
    char **endptr;

                        /*    Function flag
                         *      - bit 0 --> ra0 action
                         *      - bit 1 --> ra1 action
                         *      - bit 2 --> ra2 action
                         *      - bit 3 --> ra3 action
                         *      - bit 4 --> rau0 action
                         *      - bit 5 --> rau1 action
                         *      - bit 6 --> rau2 action
                         *      - bit 7
                         */    
    if(vlan_trans.flag==0){   //if we do not enable any function, quick pass!!    
        return;
    }

    if( (*src == 'r') && (*(src+1) == 'a') && (*dst == 'e') && (*(dst+1) == 't') && (*(dst+2) == 'h') ){
        //we should check this path...
        if(*(src+2) == 'u'){    //5G pkt
            ra_num=src+3;     //skip "rau"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(2>=rn)){   //rn should be 0~2
                if(vlan_trans.ravid[rn+4] == 0){  //this rau do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0xa5;  //magic bytes, means need action
                *(chk+1)=rn+4;    //mark rau number = rau channel + 4
                //printk("TagRau: %02X %02X\n",*chk,*(chk+1));
            }
        }else{  //2.4G pkt
            ra_num=src+2;     //skip "ra"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn] == 0){  //this ra do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0xa5;  //magic bytes, means need action
                *(chk+1)=rn;    //mark ra number
                //printk("TagRa: %02X %02X\n",*chk,*(chk+1));
            }
        }
    }else if( (*src == 'e') && (*(src+1) == 't') && (*(src+2) == 'h') && (*dst == 'r') && (*(dst+1) == 'a') ){
        // Check path:  eth -> ra/rau
        if(*(dst+2) == 'u'){    //5G pkt
            ra_num=dst+3;     //skip "rau"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(2>=rn)){   //rn should be 0~2
                if(vlan_trans.ravid[rn+4] == 0){  //this rau do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0x5a;  //magic bytes, means need action
                *(chk+1)=rn+4;    //mark rau number = rau channel + 4
                //printk("CheckRau: %02X %02X\n",*chk,*(chk+1));
            }
        }else{  //2.4G pkt
            ra_num=dst+2;     //skip "ra"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn] == 0){  //this ra do not need tag...
                	chk=skb->head;
					*chk=0x00; // fanny: don't care about the value. Wish check_vlan_tag() would discard this packet.
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0x5a;  //magic bytes, means need action
                *(chk+1)=rn;    //mark ra number
                //printk("CheckRu: %02X %02X\n",*chk,*(chk+1));
            }
        }
    }

  return;
}

#define RTMP_SET_PACKET_BR_NUM(_p, _num)	(((struct sk_buff *)_p)->cb[47] = _num)
#define RTMP_GET_PACKET_BR_NUM(_p)			(((struct sk_buff *)_p)->cb[47])
static void br_add_interface_num (const struct net_bridge_port *to, struct sk_buff *skb)
{
    RTMP_SET_PACKET_BR_NUM(skb, 0);
    if (to->br && to->br->dev) {
        unsigned char *name = to->br->dev->name;
        if (name[0] == 'b' && name[1] == 'r') {
            /* br0 - 1, br1 - 2 */
            RTMP_SET_PACKET_BR_NUM(skb, (name[2] - 0x2F));
            //printk("br_add_interface_num: name = %s, idx = %d\n", to->br->dev, RTMP_GET_PACKET_BR_NUM(skb));
        }   
    }
        
}

static void __br_forward(const struct net_bridge_port *to, struct sk_buff *skb)
{
	struct net_device *indev;
    //printk("%s -> %s\n",skb->dev->name, to->dev->name);
    br_add_flag(to->dev->name, skb);    //bruce
    br_add_interface_num(to, skb);
	indev = skb->dev;
	skb->dev = to->dev;
	skb->ip_summed = CHECKSUM_NONE;

	NF_HOOK(PF_BRIDGE, NF_BR_FORWARD, skb, indev, skb->dev,
			br_forward_finish);
}

/* called with rcu_read_lock */
void br_deliver(const struct net_bridge_port *to, struct sk_buff *skb)
{
	if (should_deliver(to, skb)) {
		__br_deliver(to, skb);
		return;
	}

	kfree_skb(skb);
}

/* called with rcu_read_lock */
void br_forward(const struct net_bridge_port *to, struct sk_buff *skb)
{
	if (should_deliver(to, skb)) {
		__br_forward(to, skb);
		return;
	}

	kfree_skb(skb);
}

/* called under bridge lock */
static void br_flood(struct net_bridge *br, struct sk_buff *skb, int clone,
	void (*__packet_hook)(const struct net_bridge_port *p,
			      struct sk_buff *skb))
{
	struct net_bridge_port *p;
	struct net_bridge_port *prev;

	if (clone) {
		struct sk_buff *skb2;

		if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
			br->statistics.tx_dropped++;
			return;
		}

		skb = skb2;
	}

	prev = NULL;

	list_for_each_entry_rcu(p, &br->port_list, list) {
		if (should_deliver(p, skb)) {
			if (prev != NULL) {
				struct sk_buff *skb2;

				if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
					br->statistics.tx_dropped++;
					kfree_skb(skb);
					return;
				}

				__packet_hook(prev, skb2);
			}

			prev = p;
		}
	}

	if (prev != NULL) {
		__packet_hook(prev, skb);
		return;
	}

	kfree_skb(skb);
}


/* called with rcu_read_lock */
void br_flood_deliver(struct net_bridge *br, struct sk_buff *skb, int clone)
{
	br_flood(br, skb, clone, __br_deliver);
}

/* called under bridge lock */
void br_flood_forward(struct net_bridge *br, struct sk_buff *skb, int clone)
{
	br_flood(br, skb, clone, __br_forward);
}
