/*
 * Copyright (C) 2007 Mindspeed Technologies Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/hardware.h>

/* See also ARM Ref. Man. */
#define C1_MMU          (1<<0)          /* mmu off/on */
#define C1_ALIGN        (1<<1)          /* alignment faults off/on */
#define C1_DC           (1<<2)          /* dcache off/on */
#define C1_WB           (1<<3)          /* merging write buffer on/off */
#define C1_BIG_ENDIAN   (1<<7)  /* big endian off/on */
#define C1_SYS_PROT     (1<<8)          /* system protection */
#define C1_ROM_PROT     (1<<9)          /* ROM protection */
#define C1_IC           (1<<12)         /* icache off/on */
#define C1_HIGH_VECTORS (1<<13) /* location of vectors: low/high addresses */
#define RESERVED_1      (0xf << 3)      /* must be 111b for R/W */

u32 CFG_HZ_CLOCK;     /* Allow for AHB clock frequency to be chosen at run time */

#define COMCERTO_PART_MAGIC 0x4D535044

void apply_ddr_setting(void);
static void config_board1(void);
static void config_board2(void);
static void config_board3(void);

void arm_write64(u64 data,volatile u64 *p)
{
        int *tptr = &data;
        register int reg_0 __asm__ ("r3");
        register int reg_1 __asm__ ("r4");

        __asm__ __volatile__ (
                "ldmia     %0, {%1,%2}     \n\t"
                : "+r" (tptr), "=r" (reg_0), "=r" (reg_1)
        );
        __asm__ __volatile__ (
                "stmia     %0, {%1,%2}     \n\t"
                : "+r" (p), "=r" (reg_0), "=r" (reg_1)
        );
}

#if 0
void arm_write64(u64 data,volatile u64 *p)
{
        __asm__ __volatile__ (    "STRD %0, [%1]\n"
                  :
                  :"r"(data),"r"(p)
                  );
}
#endif


/* read co-processor 15, register #1 (control register) */
static unsigned long read_p15_c1 (void)
{
        unsigned long value;

        __asm__ __volatile__(
                                                "mrc    p15, 0, %0, c1, c0, 0   @ read control reg\n"
                                                : "=r" (value)
                                                :
                                                : "memory");
        return value;
}

/* write to co-processor 15, register #1 (control register) */
static void write_p15_c1 (unsigned long value)
{
        __asm__ __volatile__(
                                                "mcr    p15, 0, %0, c1, c0, 0   @ write it back\n"
                                                :
                                                : "r" (value)
                                                : "memory");

        read_p15_c1 ();
}

static void cp_delay (void)
{
        volatile int i;

        /* Many OMAP regs need at least 2 nops  */
        for (i = 0; i < 100; i++);
}


void icache_enable (void)
{
        ulong reg;

        reg = read_p15_c1 ();   /* get control reg. */
        cp_delay ();
        write_p15_c1 (reg | C1_IC);
}

void SoC_APB_setup(void)
{
	// Setting APB Bus Wait states to 0, set post write
	*(volatile u32*)(APB_ACCESS_WS_REG) = __cpu_to_le32(0x40);
}

void SoC_ARAM_setup(void)
{
	// enable ARAM collision detection
	// read mod write to set bit 2 to enable.  Keep default of 32 byte mask
	*(volatile u32*)(MEMCORE_BASEADDR+0x4) |= __cpu_to_le32(4);
	// disable ARAM pipeline mode
	*(volatile u32*)(MEMCORE_BASEADDR+0x38) = __cpu_to_le32(0);
	// set ARAM port/fbp priority to 1 
	*(volatile u32*)(MEMCORE_BASEADDR+0x0C) = __cpu_to_le32(3); 
}

void SoC_Partno_setup(void)
{
        *(volatile u64*)(ARAM_BASE_ADDR+0x10000) = __cpu_to_le64(COMCERTO_PART_MAGIC);
        memcpy((volatile u8*)(ARAM_BASE_ADDR+0x10100),COMCERTO_PART_NO,8);

}

void SoC_AHB_setup(void)
{
	// DDR size 512MB, starting from 80000000
	*(volatile u32*) AHB_HIGHMEM_ADDR_TRANSLATION = __cpu_to_le32(0x9FFF); 	//(AHB_BASEADDR + 0x04)
	// Disable REMAP (Low Mem region is mapped to DDR base)
	*(volatile u32*) AHB_LOWMEM_REMAP_CTRL = __cpu_to_le32(0x0);		//(AHB_BASEADDR + 0x00)
	// 2nd stage level muxs are configured to round-robin
	*(volatile u32*) AHB_STAGE2_ARBITER_CFG	= __cpu_to_le32(0x0);	//(AHB_BASEADDR + 0x08)
}

void SoC_PLL_init(int cfg)
{
	struct c1000_cfg_clk c1000_cfg_clk_table[] =
	{
		[CFG_CLK_650_187_375] = {650000000, 3, 4, 750000000, 2, 2, 250000000, 3, 8, 0, 650000000, 0, 187500000, 375000000, 250000000}, /* M8326x, 375MHz */
		[CFG_CLK_600_165_330] = {600000000, 1, 4, 660000000, 1, 4, 250000000, 3, 8, 0, 600000000, 0, 165000000, 330000000, 110000000}, /* M83160 */
		[CFG_CLK_534_178_330] = {660000000, 1, 4, 534000000, 1, 4, 250000000, 3, 8, 1, 534000000, 1, 178000000, 330000000, 178000000}, /* M8325x */
		[CFG_CLK_450_165_330] = {450000000, 1, 4, 660000000, 1, 4, 250000000, 3, 8, 0, 450000000, 0, 165000000, 330000000, 110000000}, /* M8324x */
		[CFG_CLK_400_165_330] = {400000000, 2, 6, 660000000, 1, 4, 250000000, 3, 8, 0, 400000000, 0, 165000000, 330000000, 110000000}, /* M83240 */
		[CFG_CLK_650_200_200] = {650000000, 3, 4, 800000000, 3, 2, 250000000, 3, 8, 0, 650000000, 0, 200000000, 200000000, 266666666}, /* M8326x, 200MHz */
		[CFG_CLK_650_187_325] = {650000000, 3, 4, 750000000, 2, 2, 250000000, 3, 8, 0, 650000000, 1, 187500000, 325000000, 250000000}, /* Asic */
		[CFG_CLK_610_186_372] = {610000000, 3, 4, 744000000, 2, 2, 250000000, 3, 8, 0, 610000000, 0, 186000000, 372000000, 248000000} /* Asic */
	};
	struct c1000_cfg_clk *cfg_clk = &c1000_cfg_clk_table[cfg];

	// *(volatile u32*)(GPIO_ARM_MEMORY_SENSE_REGS) = __cpu_to_le32(0x1B6DB6DB); /* Margin = 3 */
	//*(volatile u32*)(GPIO_ARM_MEMORY_SENSE_REGS) = __cpu_to_le32(0x24924924); /* Margin = 4 */
	*(volatile u32*)(GPIO_ARM_MEMORY_SENSE_REGS) = __cpu_to_le32(0x2DB6DB6D);       /* Margin = 5 */

	// Make sure DDR controller is reset
	*(volatile u32*)(CLKCORE_BLK_RESET) &= __cpu_to_le32(~(BLK_RESET_DDR_REF_RESET_N | BLK_RESET_DDR_AHB_RESET_N));

	HAL_all_pll_bypass();

	// Take DDR controller out of reset
	*(volatile u32*)(CLKCORE_BLK_RESET) |= __cpu_to_le32(BLK_RESET_DDR_REF_RESET_N | BLK_RESET_DDR_AHB_RESET_N);

	HAL_set_arm_pll(cfg_clk->arm_pll, cfg_clk->arm_pll_nr, cfg_clk->arm_pll_od);
	HAL_set_ahb_pll(cfg_clk->ahb_pll, cfg_clk->ahb_pll_nr, cfg_clk->ahb_pll_od);
	HAL_set_phy_pll(cfg_clk->phy_pll, cfg_clk->phy_pll_nr, cfg_clk->phy_pll_od);

	HAL_set_arm_clk(cfg_clk->arm_clk, cfg_clk->arm_clk, cfg_clk->arm_from_sclk2);
	HAL_set_ahb_ddr_clk(cfg_clk->ahb_clk, cfg_clk->ddr_clk, cfg_clk->ddr_from_fclk);
	HAL_set_phy_gem_clk(CFG_GEM0_CLOCK, CFG_GEM1_CLOCK, CFG_PHY_CLOCK);
	HAL_set_ipsec_clk(cfg_clk->ipsec_clk);

	HAL_all_pll_remove_bypass();

	CFG_HZ_CLOCK = HAL_get_ahb_clk();
}


void SoC_nand_init(void)
{
        // TODO: verify timings

        // Configure CS4 - 8 bits - NAND
	*(volatile u32*)(EX_CS4_CFG_REG) = __cpu_to_le32(EX_MEM_BUS_8 | EX_NAND_MODE);
        *(volatile u32*)(EX_CSEN_REG)  |= __cpu_to_le32(EX_CS4_EN);
        *(volatile u32 *)EX_CLOCK_DIV_REG = __cpu_to_le32(0x6); //EXP bus at 66MHz
        *(volatile u32*)(EX_CS4_TMG1_REG) = __cpu_to_le32(0x01010001);
        *(volatile u32*)(EX_CS4_TMG2_REG) = __cpu_to_le32(0x01010001);

	*(volatile u32 *) GPIO_UP_ALIGN_ACCESS_LOGIC |= __cpu_to_le32(1);

	*(volatile u32 *) USB_PHY_CONF_REG = __cpu_to_le32(0x002D64C2);
}

void nand_hw_init(void)
{
        SoC_gpio_cfg(29, GPIO_TYPE_OUTPUT);
        SoC_gpio_cfg(30, GPIO_TYPE_OUTPUT);
        SoC_gpio_cfg(31, GPIO_TYPE_OUTPUT);
        *(volatile u32 *) GPIO_PIN_SELECT_REG &= __cpu_to_le32(~((1 << 31)|(1 << 30)|(1 << 29)|(1 << 21)|(1 << 20)|(1 << 5)));
        *(volatile u32 *) GPIO_PIN_SELECT_REG |= __cpu_to_le32((1 << 6));
}


int SoC_gpio_cfg(int gpio, int mode)
{
        if ((gpio < 0) || (gpio > 31))
                goto err;

        switch (mode) {
        case GPIO_TYPE_INPUT:
        default:
		*(volatile u32 *) GPIO_OE_REG &= __cpu_to_le32(~(1 << gpio));
                break;

        case GPIO_TYPE_OUTPUT:
		*(volatile u32 *) GPIO_OE_REG |= __cpu_to_le32(1 << gpio);
                break;
        }

        return 0;

err:
        return -1;
}

void apply_ddr_setting(void)
{
	switch (CFG_CLK) {
	default:
	case CFG_CLK_650_187_375:
	case CFG_CLK_534_178_330:
	case CFG_CLK_450_165_330:
	        config_board1();
		break;

	case CFG_CLK_650_200_200:
		config_board2();
		break;
	}
	
	/* training is required for 375MHz only,
         * but nice to have for 200MHz also */
#if defined(DDR_TRAINING)
		/* Do training */
		start_training();
#endif
}


static void config_board1(void)
{
  // Configure DDR
	*(volatile u64*)(DENALI_CTL_00_DATA) = __cpu_to_le64((u64)DENALI_CTL_00_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_01_DATA) = __cpu_to_le64((u64)DENALI_CTL_01_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_02_DATA) = __cpu_to_le64((u64)DENALI_CTL_02_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_03_DATA) = __cpu_to_le64((u64)DENALI_CTL_03_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_04_DATA) = __cpu_to_le64((u64)DENALI_CTL_04_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_05_DATA) = __cpu_to_le64((u64)DENALI_CTL_05_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_06_DATA) = __cpu_to_le64((u64)DENALI_CTL_06_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_07_DATA) = __cpu_to_le64((u64)DENALI_CTL_07_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_08_DATA) = __cpu_to_le64((u64)DENALI_CTL_08_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_09_DATA) = __cpu_to_le64((u64)DENALI_CTL_09_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_10_DATA) = __cpu_to_le64((u64)DENALI_CTL_10_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_11_DATA) = __cpu_to_le64((u64)DENALI_CTL_11_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_12_DATA) = __cpu_to_le64((u64)DENALI_CTL_12_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_13_DATA) = __cpu_to_le64((u64)DENALI_CTL_13_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_14_DATA) = __cpu_to_le64((u64)DENALI_CTL_14_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_15_DATA) = __cpu_to_le64((u64)DENALI_CTL_15_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_16_DATA) = __cpu_to_le64((u64)DENALI_CTL_16_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_17_DATA) = __cpu_to_le64((u64)DENALI_CTL_17_VAL_CFG1);
#ifdef CFG_DDR_16BIT
	*(volatile u64*)(DENALI_CTL_18_DATA) = __cpu_to_le64((u64)DENALI_CTL_18_VAL_CFG1 | (1ULL << 24));
#else
	*(volatile u64*)(DENALI_CTL_18_DATA) = __cpu_to_le64((u64)DENALI_CTL_18_VAL_CFG1);
#endif
	*(volatile u64*)(DENALI_CTL_19_DATA) = __cpu_to_le64((u64)DENALI_CTL_19_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_20_DATA) = __cpu_to_le64((u64)DENALI_CTL_20_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_21_DATA) = __cpu_to_le64((u64)DENALI_CTL_21_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_22_DATA) = __cpu_to_le64((u64)DENALI_CTL_22_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_23_DATA) = __cpu_to_le64((u64)DENALI_CTL_23_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_24_DATA) = __cpu_to_le64((u64)DENALI_CTL_24_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_25_DATA) = __cpu_to_le64((u64)DENALI_CTL_25_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_26_DATA) = __cpu_to_le64((u64)DENALI_CTL_26_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_27_DATA) = __cpu_to_le64((u64)DENALI_CTL_27_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_28_DATA) = __cpu_to_le64((u64)DENALI_CTL_28_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_29_DATA) = __cpu_to_le64((u64)DENALI_CTL_29_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_30_DATA) = __cpu_to_le64((u64)DENALI_CTL_30_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_31_DATA) = __cpu_to_le64((u64)DENALI_CTL_31_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_32_DATA) = __cpu_to_le64((u64)DENALI_CTL_32_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_33_DATA) = __cpu_to_le64((u64)DENALI_CTL_33_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_34_DATA) = __cpu_to_le64((u64)DENALI_CTL_34_VAL_CFG1);
	//*(volatile u64*)(DENALI_CTL_35_DATA) = __cpu_to_le64((u64)DENALI_CTL_35_VAL_CFG1);
	arm_write64(DENALI_CTL_35_VAL_CFG1,DENALI_CTL_35_DATA); //should be 64bit write
	*(volatile u64*)(DENALI_CTL_36_DATA) = __cpu_to_le64((u64)DENALI_CTL_36_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_37_DATA) = __cpu_to_le64((u64)DENALI_CTL_37_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_38_DATA) = __cpu_to_le64((u64)DENALI_CTL_38_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_39_DATA) = __cpu_to_le64((u64)DENALI_CTL_39_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_40_DATA) = __cpu_to_le64((u64)DENALI_CTL_40_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_41_DATA) = __cpu_to_le64((u64)DENALI_CTL_41_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_42_DATA) = __cpu_to_le64((u64)DENALI_CTL_42_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_43_DATA) = __cpu_to_le64((u64)DENALI_CTL_43_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_44_DATA) = __cpu_to_le64((u64)DENALI_CTL_44_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_45_DATA) = __cpu_to_le64((u64)DENALI_CTL_45_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_46_DATA) = __cpu_to_le64((u64)DENALI_CTL_46_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_47_DATA) = __cpu_to_le64((u64)DENALI_CTL_47_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_48_DATA) = __cpu_to_le64((u64)DENALI_CTL_48_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_49_DATA) = __cpu_to_le64((u64)DENALI_CTL_49_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_50_DATA) = __cpu_to_le64((u64)DENALI_CTL_50_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_51_DATA) = __cpu_to_le64((u64)DENALI_CTL_51_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_52_DATA) = __cpu_to_le64((u64)DENALI_CTL_52_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_53_DATA) = __cpu_to_le64((u64)DENALI_CTL_53_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_54_DATA) = __cpu_to_le64((u64)DENALI_CTL_54_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_55_DATA) = __cpu_to_le64((u64)DENALI_CTL_55_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_56_DATA) = __cpu_to_le64((u64)DENALI_CTL_56_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_57_DATA) = __cpu_to_le64((u64)DENALI_CTL_57_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_58_DATA) = __cpu_to_le64((u64)DENALI_CTL_58_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_59_DATA) = __cpu_to_le64((u64)DENALI_CTL_59_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_60_DATA) = __cpu_to_le64((u64)DENALI_CTL_60_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_61_DATA) = __cpu_to_le64((u64)DENALI_CTL_61_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_62_DATA) = __cpu_to_le64((u64)DENALI_CTL_62_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_63_DATA) = __cpu_to_le64((u64)DENALI_CTL_63_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_64_DATA) = __cpu_to_le64((u64)DENALI_CTL_64_VAL_CFG1);
//	*(volatile u64*)(DENALI_CTL_65_DATA) = __cpu_to_le64((u64)DENALI_CTL_65_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_66_DATA) = __cpu_to_le64((u64)DENALI_CTL_66_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_67_DATA) = __cpu_to_le64((u64)DENALI_CTL_67_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_68_DATA) = __cpu_to_le64((u64)DENALI_CTL_68_VAL_CFG1);
	*(volatile u64*)(DENALI_CTL_69_DATA) = __cpu_to_le64((u64)DENALI_CTL_69_VAL_CFG1);

	// start DDRC
	*(volatile u64*)(DENALI_CTL_02_DATA) = __cpu_to_le64((u64)DENALI_CTL_02_VAL_CFG1 | (1LL << 32));
	//wait int_status[2] (DRAM init complete)
#ifdef CONFIG_C1KM83240
	while(( __le32_to_cpu(*(volatile u32*)(DENALI_CTL_36_DATA+8)) & 0x40000) == 0);
#else
	while(( __le32_to_cpu(*(volatile u32*)(DENALI_CTL_36_DATA+4)) & 0x4) == 0);
#endif
//	*(volatile u64*)(DENALI_CTL_11_DATA) = __cpu_to_le64(((u64)DENALI_CTL_11_VAL_CFG1 & ~(0x00007F0000000000LL)) | (wr_dqs_shift << 40));

}


static void config_board2(void)
{
  // Configure DDR
	*(volatile u64*)(DENALI_CTL_00_DATA) = __cpu_to_le64((u64)DENALI_CTL_00_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_01_DATA) = __cpu_to_le64((u64)DENALI_CTL_01_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_02_DATA) = __cpu_to_le64((u64)DENALI_CTL_02_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_03_DATA) = __cpu_to_le64((u64)DENALI_CTL_03_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_04_DATA) = __cpu_to_le64((u64)DENALI_CTL_04_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_05_DATA) = __cpu_to_le64((u64)DENALI_CTL_05_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_06_DATA) = __cpu_to_le64((u64)DENALI_CTL_06_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_07_DATA) = __cpu_to_le64((u64)DENALI_CTL_07_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_08_DATA) = __cpu_to_le64((u64)DENALI_CTL_08_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_09_DATA) = __cpu_to_le64((u64)DENALI_CTL_09_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_10_DATA) = __cpu_to_le64((u64)DENALI_CTL_10_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_11_DATA) = __cpu_to_le64((u64)DENALI_CTL_11_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_12_DATA) = __cpu_to_le64((u64)DENALI_CTL_12_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_13_DATA) = __cpu_to_le64((u64)DENALI_CTL_13_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_14_DATA) = __cpu_to_le64((u64)DENALI_CTL_14_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_15_DATA) = __cpu_to_le64((u64)DENALI_CTL_15_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_16_DATA) = __cpu_to_le64((u64)DENALI_CTL_16_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_17_DATA) = __cpu_to_le64((u64)DENALI_CTL_17_VAL_CFG2);
#ifdef CFG_DDR_16BIT
	*(volatile u64*)(DENALI_CTL_18_DATA) = __cpu_to_le64((u64)DENALI_CTL_18_VAL_CFG2 | (1ULL << 24));
#else
	*(volatile u64*)(DENALI_CTL_18_DATA) = __cpu_to_le64((u64)DENALI_CTL_18_VAL_CFG2);
#endif
	*(volatile u64*)(DENALI_CTL_19_DATA) = __cpu_to_le64((u64)DENALI_CTL_19_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_20_DATA) = __cpu_to_le64((u64)DENALI_CTL_20_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_21_DATA) = __cpu_to_le64((u64)DENALI_CTL_21_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_22_DATA) = __cpu_to_le64((u64)DENALI_CTL_22_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_23_DATA) = __cpu_to_le64((u64)DENALI_CTL_23_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_24_DATA) = __cpu_to_le64((u64)DENALI_CTL_24_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_25_DATA) = __cpu_to_le64((u64)DENALI_CTL_25_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_26_DATA) = __cpu_to_le64((u64)DENALI_CTL_26_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_27_DATA) = __cpu_to_le64((u64)DENALI_CTL_27_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_28_DATA) = __cpu_to_le64((u64)DENALI_CTL_28_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_29_DATA) = __cpu_to_le64((u64)DENALI_CTL_29_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_30_DATA) = __cpu_to_le64((u64)DENALI_CTL_30_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_31_DATA) = __cpu_to_le64((u64)DENALI_CTL_31_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_32_DATA) = __cpu_to_le64((u64)DENALI_CTL_32_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_33_DATA) = __cpu_to_le64((u64)DENALI_CTL_33_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_34_DATA) = __cpu_to_le64((u64)DENALI_CTL_34_VAL_CFG2);
	//*(volatile u64*)(DENALI_CTL_35_DATA) = __cpu_to_le64((u64)DENALI_CTL_35_VAL_CFG2);
	arm_write64(DENALI_CTL_35_VAL_CFG2,DENALI_CTL_35_DATA); //should be 64bit write
	*(volatile u64*)(DENALI_CTL_36_DATA) = __cpu_to_le64((u64)DENALI_CTL_36_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_37_DATA) = __cpu_to_le64((u64)DENALI_CTL_37_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_38_DATA) = __cpu_to_le64((u64)DENALI_CTL_38_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_39_DATA) = __cpu_to_le64((u64)DENALI_CTL_39_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_40_DATA) = __cpu_to_le64((u64)DENALI_CTL_40_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_41_DATA) = __cpu_to_le64((u64)DENALI_CTL_41_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_42_DATA) = __cpu_to_le64((u64)DENALI_CTL_42_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_43_DATA) = __cpu_to_le64((u64)DENALI_CTL_43_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_44_DATA) = __cpu_to_le64((u64)DENALI_CTL_44_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_45_DATA) = __cpu_to_le64((u64)DENALI_CTL_45_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_46_DATA) = __cpu_to_le64((u64)DENALI_CTL_46_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_47_DATA) = __cpu_to_le64((u64)DENALI_CTL_47_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_48_DATA) = __cpu_to_le64((u64)DENALI_CTL_48_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_49_DATA) = __cpu_to_le64((u64)DENALI_CTL_49_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_50_DATA) = __cpu_to_le64((u64)DENALI_CTL_50_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_51_DATA) = __cpu_to_le64((u64)DENALI_CTL_51_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_52_DATA) = __cpu_to_le64((u64)DENALI_CTL_52_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_53_DATA) = __cpu_to_le64((u64)DENALI_CTL_53_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_54_DATA) = __cpu_to_le64((u64)DENALI_CTL_54_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_55_DATA) = __cpu_to_le64((u64)DENALI_CTL_55_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_56_DATA) = __cpu_to_le64((u64)DENALI_CTL_56_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_57_DATA) = __cpu_to_le64((u64)DENALI_CTL_57_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_58_DATA) = __cpu_to_le64((u64)DENALI_CTL_58_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_59_DATA) = __cpu_to_le64((u64)DENALI_CTL_59_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_60_DATA) = __cpu_to_le64((u64)DENALI_CTL_60_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_61_DATA) = __cpu_to_le64((u64)DENALI_CTL_61_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_62_DATA) = __cpu_to_le64((u64)DENALI_CTL_62_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_63_DATA) = __cpu_to_le64((u64)DENALI_CTL_63_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_64_DATA) = __cpu_to_le64((u64)DENALI_CTL_64_VAL_CFG2);
//	*(volatile u64*)(DENALI_CTL_65_DATA) = __cpu_to_le64((u64)DENALI_CTL_65_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_66_DATA) = __cpu_to_le64((u64)DENALI_CTL_66_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_67_DATA) = __cpu_to_le64((u64)DENALI_CTL_67_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_68_DATA) = __cpu_to_le64((u64)DENALI_CTL_68_VAL_CFG2);
	*(volatile u64*)(DENALI_CTL_69_DATA) = __cpu_to_le64((u64)DENALI_CTL_69_VAL_CFG2);

	// start DDRC
	*(volatile u64*)(DENALI_CTL_02_DATA) = __cpu_to_le64((u64)DENALI_CTL_02_VAL_CFG2 | (1LL << 32));
	//wait int_status[2] (DRAM init complete)
#ifdef CONFIG_C1KM83240
	while(( __le32_to_cpu(*(volatile u32*)(DENALI_CTL_36_DATA+8)) & 0x40000) == 0);
#else
	while(( __le32_to_cpu(*(volatile u32*)(DENALI_CTL_36_DATA+4)) & 0x4) == 0);
#endif
//	*(volatile u64*)(DENALI_CTL_11_DATA) = __cpu_to_le64(((u64)DENALI_CTL_11_VAL_CFG2 & ~(0x00007F0000000000LL)) | (wr_dqs_shift << 40));

}

#ifdef CONFIG_C1KM83240
void spi_hw_init(void)
{
        *(volatile u32 *) GPIO_PIN_SELECT_REG &= __cpu_to_le32(~((1 << 27)|(1 << 26)|(1 << 25)|(1 << 24)|(1 << 13)|(1 << 12)));
}
#endif


