#!/bin/sh /etc/rc.common
####### Error code table
#101: ERROR: ports can not be empty
#102: ERROR: Start port must smaller than end port
#103: ERROR: Target interface profile is disabled
#104: ERROR: WAN IP alias must have value
#105: ERROR: Public port can not contain system reserved ports

START=98

UCI_CONFIG=firewall
IPTABLES=iptables
DEBUG_PRINT="logger"
#DEBUG_PRINT="echo"
SCRIPT_LOCK="/tmp/web_apply_lock/port_redirection"
UCI_TMP_PATH="/tmp/.uci/$UCI_CONFIG"
BOOT_FILE="/tmp/profile_backup/port_redirection/boot_file"
DEBUG_LOG="/tmp/profile_backup/port_redirection/debug_log"
pre_rule=0
CGI_ERROR_MSG="/tmp/cgi_error_msg"
#reserved ports: these ports are opened for special use, they should not be used in public port
reserved_ports="68 546"
#jtarget="MASQUERADE"
#G46996: use special SNAT to replace MASQUERADE
jtarget="SNAT --to-source 0.0.0.0"

tag2="p2"
tagvpn="pVPN"

DEBUG() {
	msg=$1
	if [ "$DEBUG_PRINT" = "logger" ];then
		$DEBUG_PRINT "${msg}"
	else
		$DEBUG_PRINT ${msg} >/dev/console
	fi
}

handle_1to1() {
	#check reserved ports
	for i in $reserved_ports ;do
		if [ "$public_port_start" = "$i" ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start
	private_port=$private_port_start
	private_port2=:$private_port_start
	#clear non-used option
	uci set $UCI_CONFIG.$1.public_port_end=
}
handle_nto1() {
	#check end port > start port
	if [ "$public_port_start" -gt "$public_port_end" ] ;then
		echo -n "ERROR: Start port must smaller than end port" >$CGI_ERROR_MSG
		is_legal_rule=102
	fi
	#check reserved ports
	for i in $reserved_ports ;do
		if [ $(($i-$public_port_start)) -ge 0 -a $(($public_port_end-$i)) -ge 0 ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start:$public_port_end
	private_port=$private_port_start
	private_port2=:$private_port_start
}
handle_nton() {
	#check end port > start port
	if [ "$public_port_start" -gt "$public_port_end" ] ;then
		echo -n "ERROR: Start port must smaller than end port" >$CGI_ERROR_MSG
		is_legal_rule=102
	fi
	#check reserved ports
	for i in $reserved_ports ;do
		if [ $(($i-$public_port_start)) -ge 0 -a $(($public_port_end-$i)) -ge 0 ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start:$public_port_end
	private_port=$public_port_start:$public_port_end
	private_port2=""
	#clear non-used option
	uci set $UCI_CONFIG.$1.private_port=
}
handle_ptoip(){
	#check reserved ports
	for i in $reserved_ports ;do
		if [ $(($i-$public_port_start)) -ge 0 -a $(($public_port_end-$i)) -ge 0 ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start
	private_port=$private_port_start
	private_port2=:$private_port_start
	port_range=$(($public_port_end-$public_port_start+1))
	# ip: aaa.bbb.ccc.ddd   ip_head: "aaa.bbb.ccc."   ip_tail_start: "ddd"
	ip_tail_start=$(echo $private_ip | awk -F '.' '{print $4}' )
	ip_head=$(echo $private_ip | awk -F '.' -v dot="." '{print $1 dot $2 dot $3 dot}' )
}

setup_parameters() {
	is_legal_rule="yes"
	if [ "$2" = "apply" ] ;then
		status=$(uci get $UCI_CONFIG.$1.status)
		proto=$(uci get $UCI_CONFIG.$1.proto)
		port_redirect_mode=$(uci get $UCI_CONFIG.$1.redirect_mode)
		private_port_start=$(uci get $UCI_CONFIG.$1.private_port)
		private_port_end=$(uci get $UCI_CONFIG.$1.private_port_end)
		private_ip=$(uci get $UCI_CONFIG.$1.private_ip)
		public_intf=$(uci get $UCI_CONFIG.$1.public_intf)
		public_port_end=$(uci get $UCI_CONFIG.$1.public_port_end)
		public_port_start=$(uci get $UCI_CONFIG.$1.public_port)
		use_ipalias=$(uci get $UCI_CONFIG.$1.use_ipalias)
		ip_alias=$(uci get $UCI_CONFIG.$1.ip_alias)
		more_1to1_port=$(uci get $UCI_CONFIG.$1.more_1to1_port)
	else
		config_get status $1 status
		config_get public_intf $1 public_intf
		config_get ip_alias $1 ip_alias
		config_get private_ip $1 private_ip
		config_get proto $1 proto
		config_get port_redirect_mode $1 redirect_mode
		config_get public_port_start $1 public_port
		config_get public_port_end $1 public_port_end
		config_get private_port_start $1 private_port
		config_get private_port_end $1 private_port_end
		config_get use_ipalias $1 use_ipalias
		config_get more_1to1_port $1 more_1to1_port
	fi
	flag=
	[ "$status" != "enable" ] && flag="-u"
	
	if [ "$proto" = "both" ];then
		protocol="-m service --service-flag tcp,udp"
	else
		protocol="-p $proto"
	fi
	
	ip_alias=$(echo $ip_alias | cut -d '/' -f 1)
	
	######## CHECK PORT REDIRECT MODE AND SETUP PORT INDEX ##########
	if [ "$port_redirect_mode" = "1to1" ] ;then
		handle_1to1 $1
	elif [ "$port_redirect_mode" = "nto1" ] ;then
		handle_nto1 $1
	elif [ "$port_redirect_mode" = "nton" ] ;then
		handle_nton $1
	elif [ "$port_redirect_mode" = "ptoip" ] ;then
		handle_ptoip $1
	else	#set to default for versions before 1132
		DEBUG "(WebUI port redirection), profile:$1 version upgraded completed."
		#if origianl case is likely an n-to-? case
		if [ ! "$public_port_start" = "0" ] && [ ! "$public_port_end" = "0" ] && [ ! "$private_port_start" = "0" ] && [ ! "$public_port_start" = "$public_port_end" ] ;then
			#echo "profile=$1, n-to-?" >/dev/console
			if [ "$private_port_end" = "0" ] ;then
				#echo "profile=$1, n-to-1" >/dev/console
				uci set $UCI_CONFIG.$1.redirect_mode="nto1"
				#empty port is not allowed
				if [ "$public_port_start" = "" ] && [ "$public_port_end" = "" ] && [ "$private_port_start" = "" ] ;then
					echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
					is_legal_rule=101
				fi
				handle_nto1 $1
			else
				#echo "profile=$1, n-to-n" >/dev/console
				uci set $UCI_CONFIG.$1.redirect_mode="nton"
				#empty port is not allowed
				if [ "$public_port_start" = "" ] && [ "$public_port_end" = "" ] ;then
					echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
					is_legal_rule=101
				fi
				handle_nton $1
			fi
		else
			#echo "profile=$1, 1-to-1" >/dev/console
			uci set $UCI_CONFIG.$1.redirect_mode="1to1"
			#empty port is not allowed
			if [ "$public_port_start" = "" ] && [ "$private_port_start" = "" ] ;then
				echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
				is_legal_rule=101
			fi
			handle_1to1 $1
		fi
	fi

	############## SETUP PUBLIC IP according to WAN INTERFACE ##############
	#ipset:
	#all_interface: contains all of ipalias_wan#
	#ipalias_wan#: contains all alias of a single WAN including itself
	#ip_wan#: WAN IP
	if [ "$public_intf" = "All" ] ;then
		uci set $UCI_CONFIG.$1.use_ipalias=
		uci set $UCI_CONFIG.$1.ip_alias=
		dip="-m set --set all_interface dst"
	fi
	if [ "$public_intf" != "All" ] ;then
		#if ipset:ip_$public_intf does not exist(i.e.,:WAN profile is disabled or someone destroy the ipset)
		ipset -L ip_$public_intf >/dev/null 2>/dev/null
		if [ "$?" != "0" ] ;then
			echo -n "ERROR: Target interface profile is disabled or not existed" >$CGI_ERROR_MSG
			is_legal_rule=103
		else
			#if "Use IP Alias"="no", we include the WAN's IP only
			if [ "$use_ipalias" = "no" ] || [ "$use_ipalias" = "wan" ] || [ "$use_ipalias" = "disable" ] ;then
				uci set $UCI_CONFIG.$1.ip_alias=
				uci set $UCI_CONFIG.$1.use_ipalias=no
				dip="-m set --set ip_$public_intf dst"
			elif [ "$use_ipalias" = "ip_alias" ] ;then	#if "Use IP Alias"="Single_Alias", we include only one of the WAN's alias but not itself
				#use the WAN IP as default if option value is empty or out-of-date version
				if [ "$ip_alias" = "" ] ;then
					echo -n "ERROR: WAN IP alias must have value" >$CGI_ERROR_MSG
					is_legal_rule=104
				else
					dip="-d $ip_alias"
				fi
			elif [ "$use_ipalias" = "all" ] ; then	#if "Use IP Alias"="All", we include the WAN IP and its all alias
				uci set $UCI_CONFIG.$1.ip_alias=
				dip="-m set --set ipalias_$public_intf dst"
			else	#set to "no" as a default for versions before 1132
				uci set $UCI_CONFIG.$1.use_ipalias=no
				uci set $UCI_CONFIG.$1.ip_alias=
				dip="-m set --set ip_$public_intf dst"
			fi
		fi
 	fi
}

## Use counter to store how many rules are using this ip.
## To preventing remove the ip are using by other rule.
ipset_add(){
	private_ip=$1
	#get ip use counter
	ip_mod=$(echo $private_ip | sed 's/\./_/g')
	ip_count=$(json get prd_ipset.$ip_mod)

	if [ "$ip_count" = "" ] || [ "$ip_count" = "0" ];then
		json set prd_ipset $ip_mod=1
		#add related IPset
		ipset -A nat_ptre $private_ip 2>/dev/null
	else
		#add the counter
		count_add=$(($ip_count+1))
		json set prd_ipset $ip_mod=$count_add
	fi
}
# check no one use the ip before remove
ipset_delete(){
	delete_private_ip=$1
	#get ip use counter
	ip_mod=$(echo $delete_private_ip | sed 's/\./_/g')
	ip_count=$(json get prd_ipset.$ip_mod)
	/usr/sbin/conntrack -D -s $delete_private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -d $delete_private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -r $delete_private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -q $delete_private_ip >/dev/null 2>/dev/null
	
	if [ "$ip_count" = "1" ];then
		json set prd_ipset $ip_mod=0
		#delete related IPset
		ipset -D nat_ptre $delete_private_ip 2>/dev/null
	elif [ "$ip_count" = "" ] || [ "$ip_count" = "0" ];then
		#ERROR
		ipset -D nat_ptre $delete_private_ip 2>/dev/null
	else
		#minus the counter
		count_minus=$(($ip_count-1))
		json set prd_ipset $ip_mod=$count_minus
	fi
}

### G42434: 1to1 mode support more port setting
add_more_1to1_port_boot(){
	list_flag=1
	more_public_p=$(echo $1|awk -F ' ' '{print $1}')
	more_private_p=$(echo $1|awk -F ' ' '{print $2}')

	if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
		echo "-A $chain $flag $protocol $dip --dport $more_public_p -j DNAT --to-destination ${private_ip}:${more_private_p}" >> $BOOT_FILE
	else
		$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $more_public_p -j DNAT --to-destination ${private_ip}:${more_private_p}
	fi

	if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
		if [ "$redir_vpn" = "1" ]; then
			echo "-A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar" >> $BOOT_FILE
			echo "-A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar" >> $BOOT_FILE
		else
			echo "-A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget" >> $BOOT_FILE
			echo "-A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget" >> $BOOT_FILE
		fi
	else
		if [ "$redir_vpn" = "1" ]; then
			$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar
			$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar
		else
			$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget
			$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget
		fi
	fi
}

add_more_1to1_port(){
	list_flag=1
	more_public_p=$(echo $1|awk -F ' ' '{print $1}')
	more_private_p=$(echo $1|awk -F ' ' '{print $2}')

	### make rule
	$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $more_public_p -j DNAT --to-destination ${private_ip}:${more_private_p}
	if [ "$redir_vpn" = "1" ]; then
		$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar
		$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $more_private_p -j $nat_portredirect_jtar
	else
		$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget
		$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $more_private_p -j $jtarget
	fi
}


### G47653: Range-to-Range(IP)
### many rule in 1 chain (1 profile)
### WanIP:port1 <-> private_ip1:port
###       port2 <-> private_ip2
###       port3 <-> private_ip3
add_ptoip_rules(){
	count=0
	while [ $count -lt $port_range ]
	do
		private_ip=$ip_head$ip_tail_start
		### make rule
		$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		if [ "$redir_vpn" = "1" ]; then
			$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
			$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
		else
			$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
			$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
		fi
		if [ "$flag" != "-u" ] ;then
			#add related IPset
			ipset_add $private_ip
		fi
		###
		#next rule
		count=$(($count+1))
		public_port=$(($public_port+1))
		ip_tail_start=$(($ip_tail_start+1))
	done
}
add_ptoip_rules_boot(){
	count=0
	while [ $count -lt $port_range ]
	do
		private_ip=$ip_head$ip_tail_start
		### make rule
		if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
			echo "-A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}" >> $BOOT_FILE
		else
			$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		fi
		
		if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
			if [ "$redir_vpn" = "1" ]; then
				echo "-A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar" >> $BOOT_FILE
				echo "-A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar" >> $BOOT_FILE
			else
				echo "-A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget" >> $BOOT_FILE
				echo "-A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget" >> $BOOT_FILE
			fi
		else
			if [ "$redir_vpn" = "1" ]; then
				$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
				$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
			else
				$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
				$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
			fi
		fi
		if [ "$flag" != "-u" ] ;then
			#add related IPset
			ipset_add $private_ip
		fi
		###
		#next rule
		count=$(($count+1))
		public_port=$(($public_port+1))
		ip_tail_start=$(($ip_tail_start+1))
	done
}
delete_ptoip_ipset(){
	old_private_ip=$(uci oget $UCI_CONFIG.$1.private_ip)
	old_public_port_end=$(uci oget $UCI_CONFIG.$1.public_port_end)
	old_public_port_start=$(uci oget $UCI_CONFIG.$1.public_port)
	old_port_range=$(($old_public_port_end-$old_public_port_start+1))
	# ip: aaa.bbb.ccc.ddd   ip_head: "aaa.bbb.ccc."   ip_tail_start: "ddd"
	old_ip_tail_start=$(echo $old_private_ip | awk -F '.' '{print $4}' )
	old_ip_head=$(echo $old_private_ip | awk -F '.' -v dot="." '{print $1 dot $2 dot $3 dot}' )

	count=0
	while [ $count -lt $old_port_range ]
	do
		### delete related IPset
		ipset_delete $old_ip_head$old_ip_tail_start
		###
		#next rule
		count=$(($count+1))
		old_ip_tail_start=$(($old_ip_tail_start+1))
	done
}

# chain nameing rule:
#     prd_XXX in table nat_portredirect
#   p2prd_XXX in table nat_portredirect2
# pVPNprd_XXX in table nat_portredirectVPN
get_chain(){
	chain=`hashmap -g $UCI_CONFIG$1 $1`
	if [ -z "$chain" ] ;then
		cid=`get_nat_portrd_chainid $1`
		
		chain=prd_$cid
		chain2=$tag2$chain
		chainVPN=$tagvpn$chain
		#create new sub-chain
		iptables -t nat -N $chain
		iptables -t nat -N $chain2
		iptables -t nat -N $chainVPN
		
		#put to hashmap <profile,chain> for later query
		hashmap -p $UCI_CONFIG$1 $1 $chain
		#echo "create a new chain" >/dev/console
	else
		chain2=$tag2$chain
		chainVPN=$tagvpn$chain
	fi
}

add_rule_boot() {
	setup_parameters $1 boot

	#get chain
	get_chain $1
	#add new created sub-chain into the main chain
	if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
		echo "-A nat_portredirect $flag -j ${chain}" >> $BOOT_FILE
		echo "-A nat_portredirect2 $flag -j ${chain2}" >> $BOOT_FILE
		echo "-A nat_portredirectVPN $flag -j ${chainVPN}" >> $BOOT_FILE
	else
		iptables -t nat -A nat_portredirect $flag -j $chain
		iptables -t nat -A nat_portredirect2 $flag -j $chain2
		iptables -t nat -A nat_portredirectVPN $flag -j $chainVPN
	fi

	#check port redirect to VPN
	redir_vpn=$(ipset -T exception_subnet_set $private_ip > /dev/null && echo 1 || echo 0)
	if [ "$redir_vpn" = "1" ]; then
		redir_src=$(ip route get $private_ip | grep src | awk -F "src " '{print $2}')
		[ -z "$redir_src" ] && redir_src=$(json get network.lan1.ipaddr)
		nat_portredirect_jtar="SNAT --to-source $redir_src"
	fi

	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		if [ "$port_redirect_mode" = "ptoip" ] ;then
			# make Range-to-many boot rules
			add_ptoip_rules_boot	
		else
			#port redirection (PREROUTING): If the packet want to access public ip:port(redirected)
			if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
				echo "-A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}" >> $BOOT_FILE
			else
				$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
			fi

			#port redirection (POSTROUTING) for loop back: If the packet want to access public ip:port(redirected)
				#[G38092]: support special NAT loop back which packet's source IP is not in our LAN's subnet but is registered by static route
			if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
				if [ "$redir_vpn" = "1" ]; then
					echo "-A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar" >> $BOOT_FILE
					echo "-A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar" >> $BOOT_FILE
				else
					echo "-A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget" >> $BOOT_FILE
					echo "-A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget" >> $BOOT_FILE
				fi
			else
				if [ "$redir_vpn" = "1" ]; then
					$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
					$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
				else
					$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
					$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
				fi
			fi
	
			#save private ip to ipset:nat_ptre, this is used in "User Management":base_configuration.init
				#Reason: prevent port redirection traffic being dropped by User Management mechanism
			if [ "$flag" != "-u" ] ;then
				ipset_add $private_ip
			fi
			
			## 1to1 mode more port setting
			if [ "$more_1to1_port" != "" -a "$port_redirect_mode" == "1to1" ];then
				list_flag=0
				config_list_foreach $1 "more_1to1_port" add_more_1to1_port_boot
				if [ "$list_flag" == "0" ]; then
					# only one more_1to1_port
					add_more_1to1_port_boot "$more_1to1_port"
				fi
			fi
			
		fi
	else
		#use a dummy rule instead
		if [ -f "$BOOT_FILE" -a "$fastboot" = "1" ] ;then
			echo "-A $chain -u -j RETURN" >> $BOOT_FILE
			echo "-A $chain2 -u -j RETURN" >> $BOOT_FILE
			echo "-A $chainVPN -u -j RETURN" >> $BOOT_FILE
		else
			$IPTABLES -t nat -A $chain -u -j RETURN
			$IPTABLES -t nat -A $chain2 -u -j RETURN
			$IPTABLES -t nat -A $chainVPN -u -j RETURN
		fi

		uci set $UCI_CONFIG.$1.status="disable"
	fi
}

add_rule() {
	#echo "add_rule: $1" >/dev/console
	setup_parameters $1 apply

	#get chain
	get_chain $1
	
	#add new created sub-chain into the main chain
	iptables -t nat -A nat_portredirect $flag -j $chain
	iptables -t nat -A nat_portredirect2 $flag -j $chain2
	iptables -t nat -A nat_portredirectVPN $flag -j $chainVPN
	
	#check port redirect to VPN
	redir_vpn=$(ipset -T exception_subnet_set $private_ip > /dev/null && echo 1 || echo 0)
	if [ "$redir_vpn" = "1" ]; then
		redir_src=$(ip route get $private_ip | grep src | awk -F "src " '{print $2}')
		[ -z "$redir_src" ] && redir_src=$(json get network.lan1.ipaddr)
		nat_portredirect_jtar="SNAT --to-source $redir_src"
	fi

	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		if [ "$port_redirect_mode" = "ptoip" ] ;then
			# make Range-to-many rules
			add_ptoip_rules
		else
			#port redirection (PREROUTING): If the packet want to access public ip:port(redirected)
			$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
			if [ "$?" != "0" ] ;then
				echo "add_rule_boot($1): error when creating iptables rule" >>$DEBUG_LOG
			fi
	
			#port redirection (POSTROUTING) for loop back: If the packet want to access public ip:port(redirected)
				#[G38092]: support special NAT loop back which packet's source IP is not in our LAN's subnet but is registered by static route
			if [ "$redir_vpn" = "1" ]; then
				$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
				$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
			else
				$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
				$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
			fi
			if [ "$?" != "0" ] ;then
				echo "add_rule_boot($1): error when creating iptables rule" >>$DEBUG_LOG
			fi
	
			#save private ip to ipset:nat_ptre, this is used in "User Management":base_configuration.init
				#Reason: prevent port redirection traffic being dropped by User Management mechanism
			if [ "$flag" != "-u" ] ;then
				ipset_add $private_ip
			fi
			
			## 1to1 mode more port setting
			if [ "$more_1to1_port" != "" -a "$port_redirect_mode" == "1to1" ];then
				list_flag=0
				config_list_foreach $1 "more_1to1_port" add_more_1to1_port
				if [ "$list_flag" == "0" ]; then
					# only one more_1to1_port
					add_more_1to1_port "$more_1to1_port"
				fi
			fi
		fi
	else
		#use a dummy rule instead
		$IPTABLES -t nat -A $chain -u -j RETURN
		$IPTABLES -t nat -A $chain2 -u -j RETURN
		$IPTABLES -t nat -A $chainVPN -u -j RETURN
		uci set $UCI_CONFIG.$1.status="disable"
	fi
}

swap_rule() {
	#echo "swap_rule: $1" >/dev/console
	setup_parameters $1 apply
	
	#get chain
	get_chain $1
	
	local orig_pos
	local orig_pos2
	local orig_posVPN
	orig_pos=`iptables -t nat -L nat_portredirect --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chain`
	orig_pos2=`iptables -t nat -L nat_portredirect2 --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chain2`
	orig_posVPN=`iptables -t nat -L nat_portredirectVPN --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chainVPN`

	[ -n "$orig_pos" ]&&{
		#remove self
		iptables -t nat -D nat_portredirect $orig_pos  2>/dev/null
		#insert into dest position
		iptables -t nat -I nat_portredirect $uciid $flag -j $chain  2>/dev/null
	}
	[ -n "$orig_pos2" ]&&{
		iptables -t nat -D nat_portredirect2 $orig_pos2  2>/dev/null
		iptables -t nat -I nat_portredirect2 $uciid $flag -j $chain2  2>/dev/null
	}
	[ -n "$orig_posVPN" ]&&{
		iptables -t nat -D nat_portredirectVPN $orig_posVPN  2>/dev/null
		iptables -t nat -I nat_portredirectVPN $uciid $flag -j $chainVPN  2>/dev/null
	}
}

rename_rule() {
	#get chain
	get_chain $1
	hashmap -d $UCI_CONFIG$1 $1
	hashmap -p $UCI_CONFIG$2 $2 $chain
}

replace_rule() {
	#echo "replace_rule: param1=$1" >/dev/console
	setup_parameters $1 apply
	
	#get chain
	get_chain $1
	
	#remove old rules in chain
	iptables -t nat -F $chain 2>/dev/null
	iptables -t nat -F $chain2 2>/dev/null
	iptables -t nat -F $chainVPN 2>/dev/null

# if Range-to-many 
	#delete related IPset
	old_port_redirect_mode=$(uci oget $UCI_CONFIG.$1.redirect_mode)
	if [ "$old_port_redirect_mode" = "ptoip" ] ;then
		delete_ptoip_ipset $1
	else
		old_private_ip=$(uci oget $UCI_CONFIG.$1.private_ip)
		ipset_delete $old_private_ip
	fi
	
	#check port redirect to VPN
	redir_vpn=$(ipset -T exception_subnet_set $private_ip > /dev/null && echo 1 || echo 0)
	if [ "$redir_vpn" = "1" ]; then
		redir_src=$(ip route get $private_ip | grep src | awk -F "src " '{print $2}')
		[ -z "$redir_src" ] && redir_src=$(json get network.lan1.ipaddr)
		nat_portredirect_jtar="SNAT --to-source $redir_src"
	fi
	#add new rules in chain
	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		if [ "$port_redirect_mode" = "ptoip" ] ;then
			# make Range-to-many rules
			add_ptoip_rules
		else
			$IPTABLES -t nat -A $chain $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
			if [ "$redir_vpn" = "1" ]; then
				$IPTABLES -t nat -A $chain2 -u $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
				$IPTABLES -t nat -A $chainVPN $flag $protocol -m set --set exception_subnet_set dst -d $private_ip --dport $private_port -j $nat_portredirect_jtar
			else
				$IPTABLES -t nat -A $chain2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
				$IPTABLES -t nat -A $chainVPN -u $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j $jtarget
			fi
			if [ "$flag" != "-u" ] ;then
				#add related IPset
				ipset_add $private_ip
			fi
			
			## 1to1 mode more port setting
			if [ "$more_1to1_port" != "" -a "$port_redirect_mode" == "1to1" ];then
				list_flag=0
				config_list_foreach $1 "more_1to1_port" add_more_1to1_port
				if [ "$list_flag" == "0" ]; then
					# only one more_1to1_port
					add_more_1to1_port "$more_1to1_port"
				fi
			fi
			
		fi
	else
		#use a dummy rule instead
		$IPTABLES -t nat -A $chain -u -j RETURN
		$IPTABLES -t nat -A $chain2 -u -j RETURN
		$IPTABLES -t nat -A $chainVPN -u -j RETURN
		uci set $UCI_CONFIG.$1.status="disable"
	fi
}

delete_rule() {
	#echo "delete_rule: $1" >/dev/console
	get_chain $1
	local idx
	local idx2
	local idxVPN
	idx=`iptables -t nat -L nat_portredirect --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chain`
	idx2=`iptables -t nat -L nat_portredirect2 --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chain2`
	idxVPN=`iptables -t nat -L nat_portredirectVPN --line-numbers | awk '{if ($2==profile) print $1 }' profile=$chainVPN`
	
	[ -n "$idx" ]&&{
		iptables -t nat -D nat_portredirect $idx 2>/dev/null
		iptables -t nat -F $chain 2>/dev/null
		iptables -t nat -X $chain 2>/dev/null
	}
	[ -n "$idx2" ]&&{
		iptables -t nat -D nat_portredirect2 $idx2 2>/dev/null
		iptables -t nat -F $chain2 2>/dev/null
		iptables -t nat -X $chain2 2>/dev/null
	}
	[ -n "$idxVPN" ]&&{
		iptables -t nat -D nat_portredirectVPN $idxVPN 2>/dev/null
		iptables -t nat -F $chainVPN 2>/dev/null
		iptables -t nat -X $chainVPN 2>/dev/null
	}
	hashmap -d $UCI_CONFIG$1 $1
	
# if Range-to-many 
	#delete related IPset
	old_port_redirect_mode=$(uci oget $UCI_CONFIG.$1.redirect_mode)
	if [ "$old_port_redirect_mode" = "ptoip" ] ;then
		delete_ptoip_ipset $1
	else
		private_ip=$(uci oget $UCI_CONFIG.$1.private_ip)
		ipset_delete $private_ip
	fi
}

boot() {
	json set filter_idx prd_idx=100
	mkdir /tmp/profile_backup/port_redirection
	touch $DEBUG_LOG
	touch $BOOT_FILE
	start

	#Vincent F. 2014/08/27, fix G50299
	#Flush all non-established conntrack
	conntrack -D -u UNSET
}

start() {
	if [ -f "$BOOT_FILE" ] ;then
		rm -f $BOOT_FILE
		touch $BOOT_FILE
	else
		touch $BOOT_FILE
	fi
	echo "*nat" >> $BOOT_FILE
	fastboot=1
	config_load $UCI_CONFIG
	config_foreach add_rule_boot
	echo "COMMIT" >> $BOOT_FILE
	iptables-restore -n < $BOOT_FILE 2>/dev/null
	#If iptables-restore fails, go back to regular boot
	[ "$?" != "0" ] && {
		echo "port redirection: fast boot failed, use regular boot" >>$DEBUG_LOG
		logger "port redirection: fast boot failed, use regular boot"
		fastboot=0
		config_foreach add_rule_boot
	}
}

stop() {
	all_chain_idx=$(iptables -t nat -nvL nat_portredirect | awk -F 'prd_' '{print $2}'|awk -F ' ' '{print $1}')
	for chain_idx in $all_chain_idx
	do
		iptables -t nat -F prd_$chain_idx 2>/dev/null
		iptables -t nat -X prd_$chain_idx 2>/dev/null
	
		iptables -t nat -F p2prd_$chain_idx 2>/dev/null
		iptables -t nat -X p2prd_$chain_idx 2>/dev/null
		
		iptables -t nat -F pVPNprd_$chain_idx 2>/dev/null
		iptables -t nat -X pVPNprd_$chain_idx 2>/dev/null
	done
	
	$IPTABLES -t nat -F nat_portredirect
	$IPTABLES -t nat -F nat_portredirect2
	$IPTABLES -t nat -F nat_portredirectVPN
	ipset -F nat_ptre
}

apply_rule() {
	config_get uciaction $1 uciaction
	config_get uciid $1 uciid
	#rule index offset for the 1st rule of G36181 in boot()
	uciid=$(($uciid+$pre_rule))
	#echo "uciid: $uciid" >/dev/console
	
	case $uciaction in
		add)
			#echo "action: add" >/dev/console
			add_rule $1
			;;
		modify)
			#inspect what kind of modification is: position swapping or attribute changing
			action_type=`cat $UCI_TMP_PATH`
			#echo "action: modify, config:$1, action_type=$action_type" >/dev/console
			echo $action_type |grep '*' >/dev/null
			if [ "$?" = "0" ] ;then
				#a position swapping action
				#echo "Rule $1 SWAP to $uciid" >/dev/console
				swap_rule $1
			elif [ $(echo $action_type |grep '@') ] ;then
				#a profile renaming action
				old=`echo $action_type|sed 's/.*\.//'|awk '{FS="="; print $1}'`
				rename_rule $old $1 
			else
				#a normal attribute changing action
				#echo "Rule($1) REPLACE index:$uciid" >/dev/console
				replace_rule $1
			fi
			;;
		delete)
			#echo "action: delete" >/dev/console
			delete_rule $1
			;;
	esac
	#Notice: Here flush ALL route cache
		#Todo: flush entry according to the given private IP
	/usr/sbin/flush_route_cache.sh "port_redirection"
	#echo "clear IP:$private_ip conntracks" >/dev/console
	/usr/sbin/conntrack -D -s $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -d $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -r $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -q $private_ip >/dev/null 2>/dev/null
}

apply() {
	lock $SCRIPT_LOCK
	#echo -n "firwall apply start "  >>/tmp/apply_dur.log 2>&1
	#cat /proc/uptime >> /tmp/apply_dur.log 2>&1
	tmpcfg=${UCI_CONFIG}_`cat /proc/uptime | cut -d. -f 1`
	uci fchanges export $UCI_CONFIG > /etc/config/$tmpcfg
	config_load $tmpcfg
	rm -f /etc/config/$tmpcfg
	is_change=`uci fchanges all $UCI_CONFIG`
	
	config_foreach apply_rule
	
	[ -n "$is_change" ] && uci commit $UCI_CONFIG
	lock -u $SCRIPT_LOCK
	
	#Handle return error code
	if [ "$is_legal_rule" != "yes" ] ;then
		exit $is_legal_rule
	fi
	#echo -n "firwall apply end "  >>/tmp/apply_dur.log 2>&1
	#cat /proc/uptime >> /tmp/apply_dur.log 2>&1
}