/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "MLD_Proxy.h"
#include "MLD_forward.h"


/******   manipulate interface list ******/
/* ASM */
int add_interface_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_interface_c");
		return FALSE;
	}
	
	return add_interface(list,grp_addr, iface);
}

int add_interface(forward_t **list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *new, **ifaces,*tmp_iface;
	forward_t *tmp = get_forwarded_group(*list,group);
	time_t now;
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"add_interface : Group not forwarded.\n");
		fprintf(log_file,"add_interface : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	tmp_iface = (*ifaces);
	
	new = (iface_t *)malloc(sizeof(iface_t));
	if(new == NULL)
	{
		perror("malloc new in add_interface");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( is_group_forwarded(*list, group) == FALSE)
	{	
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Group %s is NOT forwarded : cannot add iface %d.\n",grp, iface);
		fprintf(log_file,"Group %s is NOT forwarded : cannot add iface %d.\n",grp, iface);
		return FALSE;
	}
	
	if( is_interface_set(*list, group, iface) == TRUE)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Group %s is already forwarded to %d.\n",grp, iface);
		fprintf(log_file,"Group %s is already forwarded to %d.\n",grp, iface);
		return FALSE;
	}
	
	new->iface_id = iface;
	new->sources = NULL;
	new->filter = MC_EXCLUDE;
	time(&now);
	new->time_out = now+entry_validity;
	new->nb_fwded = 0;

	if( (*ifaces) != NULL)				/* if list not empty */
		while(tmp_iface->next != NULL)	/* go to the end of the list */
			tmp_iface=tmp_iface->next;

	if( (*ifaces) == NULL)
	{
		(*ifaces) = new;
	}
	else
	{
		tmp_iface->next = new;
		new->previous = tmp_iface;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Added iface %d to ASM Group %s in Forwarding Table.\n", iface, grp);
	fprintf(log_file,"Added iface %d to ASM Group %s in Forwarding Table.\n", iface, grp);
	
	return TRUE;
}

int del_interface_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in del_interface_c");
		return FALSE;
	}
	
	return del_interface(list,grp_addr, iface);
}

int del_interface(forward_t **list, struct in6_addr group, int iface)
{
	iface_t **ifaces, *itmp;
	forward_t *tmp = get_forwarded_group(*list,group);
	char grp[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"del_interface : Group not forwarded.\n");
		fprintf(log_file,"del_interface : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	itmp = (*ifaces);
	
	if( is_interface_set(*list,group,iface) == FALSE )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Unable to delete iface %d for %s from Forwarding Table : not in it.\n",iface, grp);
		fprintf(log_file,"Unable to delete iface %d for %s from Forwarding Table : not in it.\n",iface, grp);
		return FALSE;
	}
	
	while(itmp != NULL)
	{
		if( itmp->iface_id == iface)
		{
			if( nb_interface(*list,group) == 1)
			{
					free(itmp);
					(*ifaces) = NULL;
			}else if( itmp == (*ifaces) )
			{
				(*ifaces) = (*ifaces)->next;
				(*ifaces)->previous = NULL;
				free(itmp);
			}
			else
			{
					if(itmp->previous != NULL)
					{
						iface_t *pre_itmp = itmp->previous;
						pre_itmp->next = itmp->next;
					}
					if(itmp->next !=NULL)
					{
						iface_t *post_itmp = itmp->next;
						post_itmp->previous = itmp->previous;
					}
			}
			return TRUE;			
		}
		itmp = itmp->next;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Deleted iface %d to ASM Group %s in Forwarding Table.\n", iface, grp);
	fprintf(log_file,"Deleted iface %d to ASM Group %s in Forwarding Table.\n", iface, grp);
	return FALSE;
}

int is_interface_set_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_interface_set_c");
		return FALSE;
	}
	
	return is_interface_set(list,grp_addr, iface);
}

int is_interface_set(forward_t *list, struct in6_addr group, int iface)
{
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group(list,group);
	
	if( tmp == NULL)
	{
		/* To be done ... */
		/*fprintf(stderr,"nb_interface : Group not forwarded. toto\n");*/
		
		return -1;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		if(ifaces->iface_id == iface)
			return TRUE;
		ifaces = ifaces->next;
	}
	return FALSE;
}

int nb_interface_c(forward_t *list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in nb_interface_c");
		return FALSE;
	}
	
	return nb_interface(list,grp_addr);
}

int nb_interface(forward_t *list, struct in6_addr group)
{
	int size=0;
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group(list,group);
	
	if( tmp == NULL)
	{
		/* fprintf(stderr,"nb_interface : Group not forwarded.\n"); */
		return -1;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		size++;
		ifaces = ifaces->next;
	}
	
	return size;
}

iface_t * get_interface_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_interface_c");
		return FALSE;
	}
	
	return get_interface(list,grp_addr,iface);
}

iface_t * get_interface(forward_t *list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group(list,group);
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_interface : Group %s not forwarded.\n", grp);
		fprintf(log_file,"get_interface : Group %s not forwarded.\n", grp);
		return NULL;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		if(ifaces->iface_id == iface)
			return ifaces;
		ifaces = ifaces->next;
	}
	return NULL;
}

int remove_all_interface_c(forward_t **list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_interface_c");
		return FALSE;
	}
	
	return remove_all_interface(list,grp_addr);
}

int remove_all_interface(forward_t **list, struct in6_addr group)
{
	iface_t **ifaces, *tmp_iface;
	forward_t *tmp = get_forwarded_group(*list,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"remove_all_interface : Group not forwarded.\n");
		fprintf(log_file,"remove_all_interface : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	
	while( (*ifaces) != NULL)
	{
		tmp_iface =  (*ifaces);
		remove_all_filtered_source(list,group,tmp_iface->iface_id);
		(*ifaces) = (*ifaces)->next;
		free(tmp_iface);
	}
	return TRUE;
}

/* SSM */
int add_interface_SSM_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_interface_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_interface_SSM_c");
		return FALSE;
	}
	
	return add_interface_SSM(list,src_addr,grp_addr, iface);
}

int add_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *new, **ifaces, *tmp_iface;
	time_t now;
	forward_t *tmp = get_forwarded_group_SSM(*list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"add_interface_SSM : Group not forwarded.\n");
		fprintf(log_file,"add_interface_SSM : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	tmp_iface = (*ifaces);
	
	new = (iface_t *)malloc(sizeof(iface_t));
	if(new == NULL)
	{
		perror("malloc new in add_interface");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( is_group_forwarded_SSM(*list, source, group) == FALSE)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Group (%s,%s) is NOT forwarded : cannot add iface %d.\n",src,grp,iface);
		fprintf(log_file,"Group (%s,%s) is NOT forwarded : cannot add iface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	if( is_interface_SSM_set(*list, source, group, iface) == TRUE)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Group (%s,%s) is already forwarded to %d : cannot add iface.\n",src,grp, iface);
		fprintf(log_file,"Group (%s,%s) is already forwarded to %d : cannot add iface.\n",src,grp, iface);
		return FALSE;
	}
	
	new->iface_id = iface;
	new->sources = NULL;
	new->filter = MC_EXCLUDE;			/* not used */
	time(&now);
	new->time_out = now+entry_validity;
	new->nb_fwded = 0;

	if( (*ifaces) != NULL)				/* if list not empty */
		while(tmp_iface->next != NULL)	/* go to the end of the list */
			tmp_iface=tmp_iface->next;

	if( (*ifaces) == NULL)
	{
		(*ifaces) = new;
	}
	else
	{
		tmp_iface->next = new;
		new->previous = tmp_iface;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added iface %d to SSM Group (%s,%s) in Forwarding Table.\n", iface, src, grp);
	
	return TRUE;
}

int del_interface_SSM_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in del_interface_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in del_interface_SSM_c");
		return FALSE;
	}
	
	return del_interface_SSM(list,src_addr,grp_addr, iface);
}

int del_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	iface_t **ifaces, *itmp;
	forward_t *tmp = get_forwarded_group_SSM(*list,source,group);
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"del_interface_SSM : Group not forwarded.\n");
		fprintf(log_file,"del_interface_SSM : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	itmp = (*ifaces);
	
	if( is_interface_SSM_set(*list,source,group,iface) == FALSE )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Unable to delete iface %d for (%s,%s) from Forwarding Table : not in it.\n",iface, src, grp);
		fprintf(log_file,"Unable to delete iface %d for (%s,%s) from Forwarding Table : not in it.\n",iface, src, grp);
		return FALSE;
	}
	
	while(itmp != NULL)
	{
		if( itmp->iface_id == iface)
		{
			if( nb_interface_SSM(*list,source,group) == 1)
			{
					free(itmp);
					(*ifaces) = NULL;
			}else if( itmp == (*ifaces) )
			{
				(*ifaces) = (*ifaces)->next;
				(*ifaces)->previous = NULL;
				free(itmp);
			}
			else
			{
					if(itmp->previous != NULL)
					{
						iface_t *pre_itmp = itmp->previous;
						pre_itmp->next = itmp->next;
					}
					if(itmp->next !=NULL)
					{
						iface_t *post_itmp = itmp->next;
						post_itmp->previous = itmp->previous;
					}
			}
			return TRUE;			
		}
		itmp = itmp->next;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Deleted iface %d to SSM Group (%s,%s) in Forwarding Table.\n", iface, src, grp);
	fprintf(log_file,"Deleted iface %d to SSM Group (%s,%s) in Forwarding Table.\n", iface, src, grp);
	
	return FALSE;
}

int is_interface_SSM_set_c(forward_t *list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_interface_SSM_set_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in is_interface_SSM_set_c");
		return FALSE;
	}
	
	return is_interface_SSM_set(list,src_addr,grp_addr, iface);
}

int is_interface_SSM_set(forward_t *list, struct in6_addr source, struct in6_addr group, int iface)
{
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group_SSM(list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return -1;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		if(ifaces->iface_id == iface)
			return TRUE;
		ifaces = ifaces->next;
	}
	return FALSE;
}

int nb_interface_SSM_c(forward_t *list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in nb_interface_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in nb_interface_SSM_c");
		return FALSE;
	}
	
	return nb_interface_SSM(list,src_addr,grp_addr);
}

int nb_interface_SSM(forward_t *list, struct in6_addr source, struct in6_addr group)
{
	int size=0;
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group_SSM(list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return -1;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		size++;
		ifaces = ifaces->next;
	}
	
	return size;
}

iface_t * get_interface_SSM_c(forward_t *list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_interface_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in get_interface_SSM_c");
		return FALSE;
	}
	
	return get_interface_SSM(list,src_addr,grp_addr, iface);
}

iface_t * get_interface_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface)
{
	iface_t *ifaces;
	forward_t *tmp = get_forwarded_group_SSM(list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return NULL;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		if(ifaces->iface_id == iface)
			return ifaces;
		ifaces = ifaces->next;
	}
	return NULL;
}

int remove_all_interface_SSM_c(forward_t **list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in remove_all_interface_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in remove_all_interface_SSM_c");
		return FALSE;
	}
	
	return remove_all_interface_SSM(list,src_addr,grp_addr);
}

int remove_all_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group)
{
	iface_t **ifaces, *tmp_iface;
	forward_t *tmp = get_forwarded_group_SSM(*list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"remove_all_interface : Group not forwarded.\n");
		fprintf(log_file,"remove_all_interface : Group not forwarded.\n");
		return FALSE;
	}
	
	ifaces = &(tmp->interfaces);
	
	while( (*ifaces) != NULL)
	{
		tmp_iface =  (*ifaces);
		(*ifaces) = (*ifaces)->next;
		free(tmp);
	}
	return TRUE;
}


/* BOTH */

void printf_interface_c(forward_t *list, MN_t *mobiles, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in printf_interface_c");
		return;
	}
	
	return printf_interface(list,mobiles,grp_addr);
}

void printf_interface(forward_t *list, MN_t *mobiles, struct in6_addr group)
{
	iface_t *ifaces;
	MN_t *tmp_mn = NULL;
	forward_t *tmp = get_forwarded_group(list,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		tmp_mn = get_MN_by_id(mobiles, ifaces->iface_id);
		printf("\n\t\t%d as %s in ", tmp_mn->tunnel_id,tmp_mn->tunnel_name);
		/* filtered sources */
		if(ifaces->filter == MC_EXCLUDE)
			printf("EXCLUDE {");
		else
			printf("INCLUDE {");
		//print filtered sources
		printf_source(list,group,ifaces->iface_id);
		printf("\t}");
		ifaces = ifaces->next;
	}
	printf("\n");
	return;
}

void fprintf_interface_c(forward_t *list, MN_t *mobiles, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in printf_interface_c");
		return;
	}
	
	return fprintf_interface(list,mobiles,grp_addr);
}

void fprintf_interface(forward_t *list, MN_t *mobiles, struct in6_addr group)
{
	iface_t *ifaces;
	MN_t *tmp_mn = NULL;
	forward_t *tmp = get_forwarded_group(list,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		tmp_mn = get_MN_by_id(mobiles, ifaces->iface_id);
		fprintf(log_file,"\n\t\t%d as %s in ", tmp_mn->tunnel_id,tmp_mn->tunnel_name);
		/* filtered sources */
		if(ifaces->filter == MC_EXCLUDE)
			fprintf(log_file,"EXCLUDE {");
		else
			fprintf(log_file,"INCLUDE {");
		//print filtered sources
		fprintf_source(list,group,ifaces->iface_id);
		fprintf(log_file," }");
		ifaces = ifaces->next;
	}
	fprintf(log_file,"\n");
	return;
}
void printf_interface_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in remove_all_interface_SSM_c");
		return;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in remove_all_interface_SSM_c");
		return;
	}
	
	return printf_interface_SSM(list,mobiles,src_addr,grp_addr);
}

void printf_interface_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group)
{
	iface_t *ifaces;
	MN_t *tmp_mn = NULL;
	forward_t *tmp = get_forwarded_group_SSM(list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return;
	}
	
	ifaces=tmp->interfaces;
	
	while(ifaces != NULL)
	{
		tmp_mn = get_MN_by_id(mobiles, ifaces->iface_id);
		printf("\n\t\t%d as %s", tmp_mn->tunnel_id,tmp_mn->tunnel_name);
		ifaces = ifaces->next;
	}
	printf("\n");
	return;
}

void fprintf_interface_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in fprintf_interface_SSM_c");
		return;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in fprintf_interface_SSM_c");
		return;
	}
	
	return fprintf_interface_SSM(list,mobiles,src_addr,grp_addr);
}

void fprintf_interface_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group)
{
	iface_t *ifaces;
	MN_t *tmp_mn = NULL;
	forward_t *tmp = get_forwarded_group_SSM(list,source,group);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_interface : Group not forwarded.\n");
		fprintf(log_file,"nb_interface : Group not forwarded.\n");
		return;
	}
	
	ifaces=tmp->interfaces;
	
	fprintf(log_file,"{");
	while(ifaces != NULL)
	{
		tmp_mn = get_MN_by_id(mobiles, ifaces->iface_id);
		fprintf(log_file,"\n\t\t%d as %s", tmp_mn->tunnel_id,tmp_mn->tunnel_name);
		ifaces = ifaces->next;
	}
	fprintf(log_file,"\t}\n");
	return;
}




/******   manipulate filtered source list   ******/
int add_filtered_source_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_filtered_source_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_filtered_source_c");
		return FALSE;
	}
	
	return add_filtered_source(list,src_addr,grp_addr, iface);
}

int add_filtered_source(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	filtered_src_t **sources,*new, *tmp_sources;
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"add_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"add_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	if( is_filtered_source(*list, source, group, iface) == TRUE)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Source %s is already filtered for Group %s on iface  %d : cannot add.\n",src,grp, iface);
		fprintf(log_file,"Source %s is already filtered for Group %s on iface  %d : cannot add.\n",src,grp, iface);
		return FALSE;
	}
	
	sources=&(tmp->sources);
	tmp_sources = *sources;
	
	new = (filtered_src_t *)malloc(sizeof(filtered_src_t));
	if(new == NULL)
	{
		perror("malloc new in add_filtered_source");
		return FALSE;
	}
	new->src_addr = source;
	
	if( (*sources) != NULL)				/* if list not empty */
		while(tmp_sources->next != NULL)	/* go to the end of the list */
			tmp_sources=tmp_sources->next;
			
	if( (*sources) == NULL)
		(*sources) = new;
	else
	{
		tmp_sources->next = new;
		new->previous = tmp_sources;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Source %s filtered on iface %d for ASM Group %s in Forwarding Table.\n",src,iface,grp);
	fprintf(log_file,"Source %s filtered on iface %d for ASM Group %s in Forwarding Table.\n",src,iface,grp);
	
	return TRUE;
}

int del_filtered_source_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in del_filtered_source_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in del_filtered_source_c");
		return FALSE;
	}
	
	return del_filtered_source(list,src_addr,grp_addr, iface);
}

int del_filtered_source(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	filtered_src_t **sources, *tmp_sources;
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"del_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"del_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	if( is_filtered_source(*list, source, group, iface) == FALSE)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Source %s is not filtered for Group %s on iface  %d : cannot delete.\n",src,grp, iface);
		fprintf(log_file,"Source %s is not filtered for Group %s on iface  %d : cannot delete.\n",src,grp, iface);
		return FALSE;
	}
	
	sources=&(tmp->sources);
	tmp_sources = *sources;
	
	while(tmp_sources != NULL)
	{
		if( IN6_ARE_ADDR_EQUAL(tmp_sources->src_addr.s6_addr32,source.s6_addr32) )
		{
			if( nb_filtered_source(*list,group,iface) == 1)
			{
					free(tmp_sources);
					(*sources) = NULL;
			}else if( tmp_sources == (*sources) )
			{
				(*sources) = (*sources)->next;
				(*sources)->previous = NULL;
				free(tmp_sources);
			}
			else
			{
					if(tmp_sources->previous != NULL)
					{
						filtered_src_t *pre_itmp = tmp_sources->previous;
						pre_itmp->next = tmp_sources->next;
					}
					if(tmp_sources->next !=NULL)
					{
						filtered_src_t *post_itmp = tmp_sources->next;
						post_itmp->previous = tmp_sources->previous;
					}
			}
			return TRUE;			
		}
		tmp_sources = tmp_sources->next;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Source %s no more filtered on iface %d for ASM Group %s in Forwarding Table.\n",src,iface,grp);
	fprintf(log_file,"Source %s no more filtered on iface %d for ASM Group %s in Forwarding Table.\n",src,iface,grp);
	return FALSE;
}

int is_filtered_source_c(forward_t *list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_filtered_source_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in is_filtered_source_c");
		return FALSE;
	}
	
	return is_filtered_source(list,src_addr,grp_addr, iface);
}

int is_filtered_source(forward_t *list, struct in6_addr source, struct in6_addr group, int iface)
{
	filtered_src_t *sources;
	iface_t *tmp = get_interface(list,group,iface);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		fprintf(log_file,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		return FALSE;
	}
	
	sources=tmp->sources;
	
	if(nb_filtered_source(list,group,iface) == 0)
		return FALSE;
	
	while(sources != NULL)
	{
		if( IN6_ARE_ADDR_EQUAL(sources->src_addr.s6_addr32,source.s6_addr32) )
			return TRUE;
		sources = sources->next;
	}
	return FALSE;
}

filtered_src_t * get_filtered_source_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_filtered_source_c");
		return NULL;
	}
	
	return get_filtered_source(list,grp_addr, iface);
}

filtered_src_t * get_filtered_source(forward_t *list, struct in6_addr group, int iface)
{
	iface_t *tmp = get_interface(list,group,iface);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_filtered_source : Group not forwarded to interface %d.\n", iface);
		fprintf(log_file,"get_filtered_source : Group not forwarded to interface %d.\n", iface);
		return NULL;
	}
	
	return tmp->sources;
}

int remove_all_filtered_source_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in remove_all_filtered_source_c");
		return FALSE;
	}
	
	return remove_all_filtered_source(list,grp_addr, iface);
}

int remove_all_filtered_source(forward_t **list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	filtered_src_t **sources, *tmp_sources;
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"remove_all_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"remove_all_filtered_source : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	sources=&(tmp->sources);
	
	if( nb_filtered_source(*list, group, iface) == 0)
	{
		(*sources) = NULL;
		return TRUE;
	}
	
	while( (*sources) != NULL)
	{
		tmp_sources =  (*sources);
		(*sources) = (*sources)->next;
		free(tmp_sources);
	}
	return TRUE;
}

int nb_filtered_source_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in nb_filtered_source_c");
		return -1;
	}
	
	return nb_filtered_source(list,grp_addr, iface);
	
}

int nb_filtered_source(forward_t *list, struct in6_addr group, int iface)
{
	int size=0;
	filtered_src_t *sources;
	iface_t *tmp = get_interface(list,group,iface);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		fprintf(log_file,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		return -1;
	}
	
	sources=tmp->sources;
	
	while(sources != NULL)
	{
		size++;
		sources = sources->next;
	}
	
	return size;
}

void printf_source_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in printf_source_c");
		return;
	}
	
	return printf_source(list,grp_addr, iface);
	
}

void printf_source(forward_t *list, struct in6_addr group, int iface)
{
	filtered_src_t *sources;
	char src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(list,group,iface);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		fprintf(log_file,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		return;
	}
	
	sources=tmp->sources;
	
	if(nb_filtered_source(list,group,iface) == 0)
		return;
	
	while(sources != NULL)
	{
		inet_ntop(AF_INET6,sources->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		printf("\t%s",src);
		sources = sources->next;
	}
	return;
}

void fprintf_source_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in fprintf_source_c");
		return;
	}
	
	return fprintf_source(list,grp_addr, iface);
}

void fprintf_source(forward_t *list, struct in6_addr group, int iface)
{
	filtered_src_t *sources;
	char src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(list,group,iface);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		fprintf(log_file,"nb_filtered_source : Group not forwarded to interface %d.\n", iface);
		return;
	}
	
	sources=tmp->sources;
	
	if(nb_filtered_source(list,group,iface) == 0)
		return;
	
	while(sources != NULL)
	{
		inet_ntop(AF_INET6,sources->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		fprintf(log_file,"\t%s",src);
		sources = sources->next;
	}
	return;
	
}


/******   manipulated forward list   ******/
/* ASM */
int add_forwarded_group_c(forward_t **list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	return add_forwarded_group(list,grp_addr);
}

int add_forwarded_group(forward_t **list, struct in6_addr group)
{
	char grp[INET6_ADDRSTRLEN];
	forward_t *new;
	forward_t *tmp = *list;
	
	new = (forward_t *)malloc(sizeof(forward_t)+sizeof(iface_t)+sizeof(filtered_src_t));
	if(new == NULL)
	{
		perror("malloc new in add_forwarded_group");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( is_group_forwarded(*list, group) )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Group %s is already forwarded.\n",grp);
		fprintf(log_file,"Group %s is already forwarded.\n",grp);
		return FALSE;
	}
	
	new->grp_addr = group;
	new->mode = MC_ASM;
	new->src_addr = in6addr_any;
	new->interfaces = NULL;

	if(*list != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(*list == NULL)
	{
		*list = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Added ASM Group %s in Forwarding Table.\n", grp);
	fprintf(log_file,"Added ASM Group %s in Forwarding Table.\n", grp);
	
	return TRUE;
}

int del_forwarded_group_c(forward_t **list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	return del_forwarded_group(list,grp_addr);
}

int del_forwarded_group(forward_t **list, struct in6_addr group)
{
	forward_t *tmp = *list;
	char grp[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	struct ipv6_mreq grp_data;
	
	if(!is_group_forwarded(*list,group))
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Unable to delete %s from Forwarding Table : not in it.\n", grp);
		fprintf(log_file,"Unable to delete %s from Forwarding Table : not in it.\n", grp);
		return FALSE;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_ASM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) )
			{
				if( nb_forwarded_group(*list) == 1)		/* if it is the last group */
				{
					free(tmp);
					*list = NULL;
				}
				
				/* leave group on LAN interface */
				
				grp_data.ipv6mr_multiaddr = group;
				grp_data.ipv6mr_interface = int_lan_id;
					
				if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_LEAVE_GROUP, (char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Unable to leave (*,%s) on if %d (LAN).\n", grp, int_lan_id );
					fprintf(log_file,"Unable to leave (*,%s) on if %d (LAN).\n", grp, int_lan_id );
				}
				
				else if( tmp == *list)
				{
					*list = tmp->next;
					free(tmp);
					(*list)->previous = NULL;
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"%s deleted successfully from Forwarding Table.\n",grp);
					fprintf(log_file,"%s deleted successfully from Forwarding Table.\n",grp);
					return TRUE;
				}
				else 				/* else adjust pointers */
				{
					if(tmp->previous != NULL)
					{
						forward_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						forward_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"%s deleted successfully from Forwarding Table.\n",grp);
				fprintf(log_file,"%s deleted successfully from Forwarding Table.\n",grp);
				return TRUE;
			}
		}
		tmp = tmp->next;
	}	
	return FALSE;
}

int is_group_forwarded_c(forward_t *list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	return is_group_forwarded(list,grp_addr);
}

int is_group_forwarded(forward_t *list, struct in6_addr group)
{
	forward_t *tmp = list;
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_ASM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) )
				return TRUE;
		}
		tmp = tmp->next;
	}
	return FALSE;
}

forward_t * get_forwarded_group_c(forward_t *list, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	return get_forwarded_group(list,grp_addr);
}

forward_t * get_forwarded_group(forward_t *list, struct in6_addr group)
{
	forward_t *tmp = list;
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_ASM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) )
				return tmp;
		}
		tmp = tmp->next;
	}
	return NULL;
}

/* SSM */
int add_forwarded_group_SSM_c(forward_t **list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	return add_forwarded_group_SSM(list,src_addr,grp_addr);
}

int add_forwarded_group_SSM(forward_t **list, struct in6_addr source, struct in6_addr group)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	forward_t *new;
	forward_t *tmp = *list;
	
	new = (forward_t *)malloc(sizeof(forward_t));
	if(new == NULL)
	{
		perror("malloc new in add_forwarded_group");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( is_group_forwarded_SSM(*list, source, group) )
	{
		if( debug_lvl == VERBOSE)
			fprintf(stderr,"Group (%s,%s) is already forwarded.\n",src,grp);
		fprintf(log_file,"Group (%s,%s) is already forwarded.\n",src,grp);
		return FALSE;
	}
	
	new->grp_addr = group;
	new->mode = MC_SSM;
	new->src_addr = source;
	new->interfaces = NULL;

	if(*list != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(*list == NULL)
	{
		*list = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(stderr,"Added SSM Group (%s,%s) in Forwarding Table.\n", src, grp);
	fprintf(log_file,"Added SSM Group (%s,%s) in Forwarding Table.\n", src, grp);
	
	return TRUE;
}


int del_forwarded_group_SSM_c(forward_t **list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	return del_forwarded_group_SSM(list,src_addr,grp_addr);
}

int del_forwarded_group_SSM(forward_t **list, struct in6_addr source, struct in6_addr group)
{
	
	forward_t *tmp = *list;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if(!is_group_forwarded_SSM(*list,source,group))
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"Unable to delete (%s,%s) from Forwarding Table : not in it.\n", src, grp);
		fprintf(log_file,"Unable to delete (%s,%s) from Forwarding Table : not in it.\n", src, grp);
		return FALSE;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_SSM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) && IN6_ARE_ADDR_EQUAL(tmp->src_addr.s6_addr32,source.s6_addr32) )
			{
				if( nb_forwarded_group(*list) == 1)		/* if it is the last MN */
				{
					free(tmp);
					*list = NULL;
				}
				else if( tmp == *list)
				{
					*list = tmp->next;
					free(tmp);
					(*list)->previous = NULL;
					return TRUE;
				}
				else 			/* else adjust pointers */
				{
					if(tmp->previous != NULL)
					{
						forward_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						forward_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"(%s,%s) deleted successfully from Forwarding Table.\n",src,grp);
				fprintf(log_file,"(%s,%s) deleted successfully from Forwarding Table.\n",src,grp);
				return TRUE;
			}
		}
		tmp = tmp->next;
	}	
	return FALSE;
}

int is_group_forwarded_SSM_c(forward_t *list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	return is_group_forwarded_SSM(list,src_addr,grp_addr);
}

int is_group_forwarded_SSM(forward_t *list, struct in6_addr source, struct in6_addr group)
{
	forward_t *tmp = list;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_SSM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) && IN6_ARE_ADDR_EQUAL(tmp->src_addr.s6_addr32,source.s6_addr32) )
				return TRUE;
		}
		tmp = tmp->next;
	}
	return FALSE;
}
forward_t * get_forwarded_group_SSM_c(forward_t *list, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return FALSE;
	}
	return get_forwarded_group_SSM(list,src_addr,grp_addr);
}

forward_t * get_forwarded_group_SSM(forward_t *list, struct in6_addr source, struct in6_addr group)
{
	forward_t *tmp = list;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_SSM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->grp_addr.s6_addr32,group.s6_addr32) && IN6_ARE_ADDR_EQUAL(tmp->src_addr.s6_addr32,source.s6_addr32) )
				return tmp;
		}
		tmp = tmp->next;
	}
	return NULL;
}

/* BOTH */
void print_forwarded_group_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	return print_forwarded_group_SSM(list,mobiles,src_addr,grp_addr);
}

void print_forwarded_group_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group)
{
	forward_t *tmp = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	
	tmp = get_forwarded_group_SSM(list, source, group);
	
	inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,tmp->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
	printf("\tSSM Group : (%s,%s)\t",src,grp);
	printf("Forwarding on %d interfaces : ",nb_interface_SSM(list,tmp->src_addr,tmp->grp_addr));
	printf_interface_SSM(list,mobiles,tmp->src_addr,tmp->grp_addr);
	printf("\n");
}

void fprint_forwarded_group_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	return fprint_forwarded_group_SSM(list,mobiles,src_addr,grp_addr);	
}

void fprint_forwarded_group_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group)
{
	forward_t *tmp = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	
	tmp = get_forwarded_group_SSM(list, source, group);
	
	inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,tmp->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
	fprintf(log_file,"\tSSM Group : (%s,%s)\t",src,grp);
	fprintf(log_file,"Forwarding on %d interfaces : ",nb_interface_SSM(list,tmp->src_addr,tmp->grp_addr));
	fprintf_interface_SSM(list,mobiles,tmp->src_addr,tmp->grp_addr);
	fprintf(log_file,"\n");	
}
void print_forwarded_group_c(forward_t *list, MN_t *mobiles, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	
	return print_forwarded_group(list,mobiles,grp_addr);
}

void print_forwarded_group(forward_t *list, MN_t *mobiles, struct in6_addr group)
{
	forward_t *tmp = NULL;
	char grp[INET6_ADDRSTRLEN];
	
	tmp = get_forwarded_group(list, group);
	
	inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
	printf("\tASM Group : %s\t",grp);
	printf("Forwarding on %d interfaces : ",nb_interface(list,tmp->grp_addr));
	printf_interface(list, mobiles,tmp->grp_addr);
	printf("\n");
}

void fprint_forwarded_group_c(forward_t *list, MN_t *mobiles, char *group)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_forwarded_group_c");
		return;
	}
	
	return fprint_forwarded_group(list,mobiles,grp_addr);
}

void fprint_forwarded_group(forward_t *list, MN_t *mobiles, struct in6_addr group)
{
	forward_t *tmp = NULL;
	char grp[INET6_ADDRSTRLEN];
	
	tmp = get_forwarded_group(list, group);
	
	inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
	fprintf(log_file,"\tASM Group : %s\t",grp);
	fprintf(log_file,"Forwarding on %d interfaces : ",nb_interface(list,tmp->grp_addr));
	fprintf_interface(list,mobiles,tmp->grp_addr);
	fprintf(log_file,"\n");
}

void print_all_forwarded_group(forward_t *list, MN_t *mobiles)
{
	forward_t *tmp = list;
	char group[INET6_ADDRSTRLEN],source[INET6_ADDRSTRLEN];

	printf("\nIn the forwarding table there are %d groups.\n", nb_forwarded_group(list));
	if( list == NULL)
	{
		return;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_ASM)
		{
			inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,group,INET6_ADDRSTRLEN);
			printf("\tASM Group : %s\t",group);
			printf("Forwarding on %d interfaces :\t",nb_interface(list,tmp->grp_addr));
			printf_interface(list,mobiles,tmp->grp_addr);
		}
		else if(tmp->mode==MC_SSM)
		{
			inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,group,INET6_ADDRSTRLEN);
			inet_ntop(AF_INET6,tmp->src_addr.s6_addr,source,INET6_ADDRSTRLEN);
			printf("\tSSM Group : (%s,%s)\t",source,group);
			printf("Forwarding on %d interfaces : ",nb_interface_SSM(list,tmp->src_addr,tmp->grp_addr));
			printf_interface_SSM(list,mobiles,tmp->src_addr,tmp->grp_addr);
		}
		tmp = tmp->next;
	}
	printf("\n");
}

void fprint_all_forwarded_group(forward_t *list, MN_t *mobiles)
{
	forward_t *tmp = list;
	char group[INET6_ADDRSTRLEN],source[INET6_ADDRSTRLEN];

	fprintf(log_file,"\nIn the forwarding table there are %d groups.\n", nb_forwarded_group(list));
	if( list == NULL)
	{
		return;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode==MC_ASM)
		{
			inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,group,INET6_ADDRSTRLEN);
			fprintf(log_file,"\tASM Group : %s\t",group);
			fprintf(log_file,"Forwarding on %d interfaces : ",nb_interface(list,tmp->grp_addr));
			fprintf_interface(list,mobiles,tmp->grp_addr);
		}
		else if(tmp->mode==MC_SSM)
		{
			inet_ntop(AF_INET6,tmp->grp_addr.s6_addr,group,INET6_ADDRSTRLEN);
			inet_ntop(AF_INET6,tmp->src_addr.s6_addr,source,INET6_ADDRSTRLEN);
			fprintf(log_file,"\tSSM Group : (%s,%s)\t",source,group);
			fprintf(log_file,"Forwarding on %d interfaces : ",nb_interface_SSM(list,tmp->src_addr,tmp->grp_addr));
			fprintf_interface_SSM(list,mobiles,tmp->src_addr,tmp->grp_addr);
		}
		tmp = tmp->next;
	}
	fprintf(log_file,"\n");
}

int nb_forwarded_group(forward_t *list)
{
	int size = 0;
	forward_t *tmp = list;
	
	while(tmp != NULL)
	{
		size++;
		tmp = tmp->next;
	}
	
	return size;
}

void free_forwarded_group(forward_t **list)
{
	forward_t *tmp = *list, *next;
	int length = nb_forwarded_group(*list), i;
	
	if(debug_lvl == VERBOSE)
		fprintf(stderr,"\nFreeing forwarding table...\n");
	fprintf(log_file,"\nFreeing forwarding table...\n");
	
	if(*list == NULL)
		return;
	
	if(length == 0)
	{
		*list = NULL;
		return;
	}
	
	for(i=0;i<length;i++)
	{
		if(tmp->mode == MC_ASM)
			remove_all_interface(list,tmp->grp_addr);
		else
			remove_all_interface_SSM(list,tmp->src_addr, tmp->grp_addr);
		next = tmp->next;
		free(tmp);
		tmp = next;
	}

	*list = NULL;
	return;
}

/* TIMERS */

/**
 * Reset the timer for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in reset_timer_SSM_c");
		return FALSE;
	}
	
	return reset_timer(list,grp_addr,iface);
}
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer(forward_t **list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	time_t now;
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"reset_timer : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"reset_timer : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	time(&now);
	tmp->time_out = now + entry_validity;
	
	return TRUE;
}
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_SSM_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_filtered_source_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in is_filtered_source_c");
		return FALSE;
	}
	
	return reset_timer_SSM(list,src_addr,grp_addr, iface);
}
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	time_t now;
	iface_t *tmp = get_interface_SSM(*list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"reset_timer_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file,"reset_timer_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	time(&now);
	tmp->time_out = now + entry_validity;
	
	return TRUE;
}
/**
 * Get the time out for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in reset_timer_SSM_c");
		return FALSE;
	}
	
	return get_timer(list,grp_addr,iface);
}
/**
 * Get the time out for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer(forward_t *list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_timer : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"get_timer : Group %s not forwarded to interface %d.\n",grp,iface);
		return -1;
	}
	
	return tmp->time_out;
	
}
/**
 * Get the time out for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_SSM_c(forward_t *list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_timer_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in get_timer_SSM_c");
		return FALSE;
	}
	
	return get_timer_SSM(list,src_addr,grp_addr, iface);
}
/**
 * Get the time out for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface_SSM(list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"reset_timer_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file,"reset_timer_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	return tmp->time_out;
}
/**
 * Checks all the timers  in the forwarding table and deletes the corresponding entries if the timer has expired
 * @param list The forwarding table
 * @return TRUE if success, FALSE otherwise
 */
int check_timers(forward_t **list)
{
	time_t now;
	forward_t *tmp_fwd = *list;
	iface_t *tmp_iface = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	int length;
	struct ipv6_mreq grp_data;
		
	time(&now);
		
	if(*list == NULL)
	{
		if(debug_lvl == VERBOSE)
		{
			// fprintf(log_file,"No groups forwarded -> cannot check timers (time is %d).\n", (int)now);
			// fprintf(stderr,"No groups forwarded -> cannot check timers (time is %d).\n", (int)now);
		}
		return FALSE;
	}
	
	while(tmp_fwd != NULL)
	{
		tmp_iface = tmp_fwd->interfaces;
		inet_ntop(AF_INET6,tmp_fwd->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,tmp_fwd->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		while(tmp_iface != NULL)
		{
			if( tmp_fwd->mode == MC_ASM)
			{
				if( get_timer(*list, tmp_fwd->grp_addr, tmp_iface->iface_id)-now <= DELAY_BEFORE_NEW_QUERY 
				    && get_timer(*list, tmp_fwd->grp_addr, tmp_iface->iface_id)-now > DELAY_BEFORE_NEW_QUERY-1)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr, "--- %sSending query to if %d before timer expires for %s\n", 
							ctime(&now), tmp_iface->iface_id, grp);
										
					fprintf(log_file, "--- %sSending query to if %d before timer expires for %s\n", 
							ctime(&now), tmp_iface->iface_id, grp);
					
					length = sizeof(struct sockaddr_ll);
					last_query_ll.sll_ifindex = tmp_iface->iface_id;
					
					if( sendto( sock_fwd, last_query, last_query_size, 0, (struct sockaddr *)&last_query_ll, length ) == -1 )
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Problem while forwarding Query to %d\n",tmp_iface->iface_id);
						fprintf(log_file,"Problem while forwarding Query to %d\n",tmp_iface->iface_id);
					}
				}
			
				if(get_timer(*list, tmp_fwd->grp_addr, tmp_iface->iface_id) <= now)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Forwarding of ASM Group %s toward %d -> TIMER EXPIRED\n", grp, tmp_iface->iface_id);
					fprintf(log_file,"Forwarding of ASM Group %s toward %d -> TIMER EXPIRED\n", grp, tmp_iface->iface_id);
										
					/* leave group on interface */
				
					grp_data.ipv6mr_multiaddr = tmp_fwd->grp_addr;
					grp_data.ipv6mr_interface = tmp_iface->iface_id;
					
					if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_LEAVE_GROUP, (char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
					{
						if( debug_lvl == VERBOSE)
							fprintf(stderr,"Unable to leave (*,%s) on if %d.\n", grp, tmp_iface->iface_id );
						fprintf(log_file,"Unable to leave (*,%s) on if %d.\n", grp, tmp_iface->iface_id );
					}
					
					if( del_interface (list, tmp_fwd->grp_addr, tmp_iface->iface_id) != TRUE)
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Deleting of forwarding of ASM Group %s toward %d FAILED\n", grp, tmp_iface->iface_id);
						fprintf(log_file,"Deleting of forwarding of ASM Group %s toward %d FAILED\n", grp, tmp_iface->iface_id);
						return FALSE;
					}
				}
				// else
					// fprintf(stderr, "Timer is ok %d %d\n", 
					// (int)get_timer(*list, tmp_fwd->grp_addr, tmp_iface->iface_id), (int)now);
			}
			else if( tmp_fwd->mode == MC_SSM)
			{
				if( get_timer_SSM(*list, tmp_fwd->src_addr, tmp_fwd->grp_addr, tmp_iface->iface_id)-now <= DELAY_BEFORE_NEW_QUERY 
				    && get_timer_SSM(*list, tmp_fwd->src_addr, tmp_fwd->grp_addr, tmp_iface->iface_id)-now > DELAY_BEFORE_NEW_QUERY-1)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr, "%sSending query to if %d before timer expires for %s\n", 
							ctime(&now), tmp_iface->iface_id, grp);
					fprintf(log_file, "%sSending query to if %d before timer expires for %s\n", 
						ctime(&now), tmp_iface->iface_id, grp);
					
					length = sizeof(struct sockaddr_ll);
					last_query_ll.sll_ifindex = tmp_iface->iface_id;
					
					if( sendto( sock_fwd, last_query, last_query_size, 0, (struct sockaddr *)&last_query_ll, length ) == -1 )
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Problem while forwarding Query to %d\n",tmp_iface->iface_id);
						fprintf(log_file,"Problem while forwarding Query to %d\n",tmp_iface->iface_id);
					}
				}
			
				if(get_timer_SSM(*list, tmp_fwd->src_addr, tmp_fwd->grp_addr, tmp_iface->iface_id) <= now)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Forwarding of SSM Channel (%s,%s) toward if %d -> TIMER EXPIRED\n", src, grp, tmp_iface->iface_id);
					fprintf(log_file,"Forwarding of SSM Channel (%s,%s) toward if %d -> TIMER EXPIRED\n", src, grp, tmp_iface->iface_id);
					if( del_interface_SSM(list, tmp_fwd->src_addr, tmp_fwd->grp_addr, tmp_iface->iface_id) != TRUE)
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Deleting of forwarding of SSM Channel (%s,%s) toward %d FAILED\n", src, grp, tmp_iface->iface_id);
						fprintf(log_file,"Deleting of forwarding of SSM Channel (%s,%s) toward %d FAILED\n", src, grp, tmp_iface->iface_id);
						return FALSE;
					}
				}
			}
			tmp_iface=tmp_iface->next;
		}
		tmp_fwd=tmp_fwd->next;
	}
	
	return TRUE;
}
/**
 * Checks all entries in the forwarding table, and deletes the ones with no interface to forward to or the ones which are not joined
 * on the LAN interface
 * @param fwd_list The forwarding table
 * @param member_list The membership structure
 * @return TRUE if success, FALSE otherwise
 */
int check_forward(forward_t **fwd_list, membership_t *member_list)
{
	forward_t *tmp_fwd = *fwd_list;
	struct in6_addr group, source;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	
	if(*fwd_list == NULL)
	{
		/*if(debug_lvl == VERBOSE)
			fprintf(log_file,"No groups forwarded -> cannot check entries.\n");*/
		return FALSE;
	}
	
	while(tmp_fwd != NULL)
	{
		source=tmp_fwd->src_addr;
		group=tmp_fwd->grp_addr;
		inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
		
		if( tmp_fwd->mode == MC_ASM)
		{
			tmp_fwd=tmp_fwd->next;
			if( is_group_joined_by_addr(*member_list,group) == FALSE)
			{
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"ASM Group %s not joined on LAN interface -> FORWARDING USELESS\n", grp);
				fprintf(log_file,"ASM Group %s not joined on LAN interface -> FORWARDING USELESS\n", grp);
				if( del_forwarded_group(fwd_list,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of forwarding of ASM Group %s FAILED\n", grp);
					fprintf(log_file,"Deleting of forwarding of ASM Group %s FAILED\n", grp);
					return FALSE;
				}
				
			}
			else if(nb_interface(*fwd_list,group) <= 0)
			{
				/* LEAVE ON LAN INTERFACE */
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"No forwarding of ASM Group %s -> USELESS\n", grp);
				fprintf(log_file,"No forwarding of ASM Group %s -> USELESS\n", grp);
				if( del_forwarded_group(fwd_list,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of forwarding of ASM Group %s FAILED\n", grp);
					fprintf(log_file,"Deleting of forwarding of ASM Group %s FAILED\n", grp);
				}
				if( del_mult_group_by_addr(member_list,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of membership for ASM Group %s FAILED\n", grp);
					fprintf(log_file,"Deleting of membership for ASM Group %s FAILED\n", grp);
				}
			}
		}
		else if( tmp_fwd->mode == MC_SSM)
		{
			tmp_fwd=tmp_fwd->next;
			if( is_group_joined_SSM_by_addr(*member_list,source,group) == FALSE)
			{
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"SSM Channel (%s,%s) not joined on LAN interface -> FORWARDING USELESS\n",src,grp);
				fprintf(log_file,"SSM Channel (%s,%s) not joined on LAN interface -> FORWARDING USELESS\n",src,grp);
				if( del_forwarded_group_SSM(fwd_list,source,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of forwarding of SSM Channel (%s,%s) FAILED\n",src, grp);
					fprintf(log_file,"Deleting of forwarding of SSM Channel (%s,%s) FAILED\n",src, grp);
					return FALSE;
				}
				
			}
			else if(nb_interface_SSM(*fwd_list,source,group) <= 0)
			{
				/* LEAVE ON LAN INTERFACE */
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"No forwarding of SSM Channel (%s,%s) -> USELESS\n",src, grp);
				fprintf(log_file,"No forwarding of SSM Channel (%s,%s) -> USELESS\n",src, grp);
				if( del_forwarded_group_SSM(fwd_list,source,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of forwarding of SSM Channel (%s,%s) FAILED\n",src, grp);
					fprintf(log_file,"Deleting of forwarding of SSM Channel (%s,%s) FAILED\n",src, grp);
				}
				if( del_mult_group_SSM_by_addr(member_list,source,group) != TRUE)
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Deleting of membership for SSM Channel (%s,%s) FAILED\n",src, grp);
					fprintf(log_file,"Deleting of membership for SSM Channel (%s,%s) FAILED\n",src, grp);
				}
			}
		}
	}
	
	return TRUE;
}

/* Stats */
/**
 * Init the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in init_fwd_stat_c");
		return FALSE;
	}
	
	return init_fwd_stat(list,grp_addr,iface);
}
/**
 * Init the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat(forward_t **list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if( debug_lvl == VERBOSE )
			fprintf(stderr,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file, "init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded = 0;
	
	return TRUE;
}
/**
 * Add 1 to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_c(forward_t **list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_fwd_stat_c");
		return FALSE;
	}
	
	return inc_fwd_stat(list,grp_addr,iface);
}
/**
 * Add 1 to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat(forward_t **list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded += 1;
	
	return TRUE;
}
/**
 * Add n to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_c(forward_t **list, char *group, int iface, int n)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_n_fwd_stat_c");
		return FALSE;
	}
	
	return inc_n_fwd_stat(list,grp_addr,iface,n);
}
/**
 * Add n to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat(forward_t **list, struct in6_addr group, int iface, int n)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(*list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded += n;
	
	return TRUE;
}
/**
 * Get the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given ASM Group if success, -1 otherwise
 */
int get_fwd_stat_c(forward_t *list, char *group, int iface)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_fwd_stat_c");
		return FALSE;
	}
	
	return get_fwd_stat(list,grp_addr,iface);
}
/**
 * Get the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given ASM Group if success, -1 otherwise
 */
int get_fwd_stat(forward_t *list, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface(list,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		fprintf(log_file,"init_fwd_stat : Group %s not forwarded to interface %d.\n",grp,iface);
		return -1;
	}
	
	return tmp->nb_fwded;
}

/**
 * Init the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in init_fwd_stat_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in init_fwd_stat_SSM_c");
		return FALSE;
	}
	
	return init_fwd_stat_SSM(list,src_addr,grp_addr, iface);
}
/**
 * Init the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface_SSM(*list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file,"init_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded = 0;
	
	return TRUE;
}
/**
 * Add 1 to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_fwd_stat_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_fwd_stat_SSM_c");
		return FALSE;
	}
	
	return inc_fwd_stat_SSM(list,src_addr,grp_addr, iface);
}
/**
 * Add 1 to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface_SSM(*list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file,"inc_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded += 1;
	
	return TRUE;
}
/**
 * Add n to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface, int n)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_n_fwd_stat_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in inc_n_fwd_stat_SSM_c");
		return FALSE;
	}
	
	return inc_n_fwd_stat_SSM(list,src_addr,grp_addr, iface,n);
}
/**
 * Add n to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface, int n)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface_SSM(*list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_n_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file ,"inc_n_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	tmp->nb_fwded += n;
	
	return TRUE;
}
/**
 * Get the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given SSM Channel if success, -1 otherwise
 */
int get_fwd_stat_SSM_c(forward_t *list, char *source, char *group, int iface)
{
	struct in6_addr grp_addr, src_addr;
	
	if( (inet_pton(AF_INET6, group, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in get_fwd_stat_SSM_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in get_fwd_stat_SSM_c");
		return FALSE;
	}
	
	return get_fwd_stat_SSM(list,src_addr,grp_addr, iface);
}
/**
 * Get the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given SSM Channel if success, -1 otherwise
 */
int get_fwd_stat_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface)
{
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	iface_t *tmp = get_interface_SSM(list,source,group,iface);
	
	inet_ntop(AF_INET6,group.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source.s6_addr,src,INET6_ADDRSTRLEN);
	
	if( tmp == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		fprintf(log_file,"get_fwd_stat_SSM : Group (%s,%s) not forwarded to interface %d.\n",src,grp,iface);
		return FALSE;
	}
	
	return tmp->nb_fwded;
}
/**
 * Print stats on forwarded packets on the standard output
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 */
void print_fwd_stat(forward_t *fwd_list, MN_t *mn_list)
{
	forward_t *tmp_fwd = fwd_list;
	MN_t *tmp_mn = NULL;
	iface_t *tmp_iface = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	
	if(fwd_list == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"No groups forwarded -> cannot print stats.\n");
		return;
	}

	while(tmp_fwd != NULL)
	{
		tmp_iface = tmp_fwd->interfaces;
		inet_ntop(AF_INET6,tmp_fwd->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,tmp_fwd->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		
		if( tmp_fwd->mode == MC_ASM)
		{
			fprintf(stderr,"ASM Group %s :\n", grp);
		}
		else if( tmp_fwd->mode == MC_SSM)
		{
			fprintf(stderr,"SSM Channel (%s,%s) :\n",src,grp);
		}
		
		while(tmp_iface != NULL)
		{
			tmp_mn = get_MN_by_id(mn_list, tmp_iface->iface_id);
			fprintf(stderr,"\tforwarded %d to %s\n",tmp_iface->nb_fwded,tmp_mn->tunnel_name);
			tmp_iface=tmp_iface->next;
		}
		fprintf(stderr,"\n");
		tmp_fwd=tmp_fwd->next;
	}
	
	return;
}
/**
 * Print stats on forwarded packets on the standard output
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 */
void fprint_fwd_stat(forward_t *fwd_list, MN_t *mn_list)
{
	forward_t *tmp_fwd = fwd_list;
	MN_t *tmp_mn = NULL;
	iface_t *tmp_iface = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];	
	
	if(fwd_list == NULL)
	{
		fprintf(log_file,"\tNo groups forwarded -> cannot print stats.\n");
		return;
	}

	while(tmp_fwd != NULL)
	{
		tmp_iface = tmp_fwd->interfaces;
		inet_ntop(AF_INET6,tmp_fwd->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,tmp_fwd->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		
		if( tmp_fwd->mode == MC_ASM)
		{
			fprintf(log_file,"\tASM Group %s :\n", grp);
		}
		else if( tmp_fwd->mode == MC_SSM)
		{
			fprintf(log_file,"\tSSM Channel (%s,%s) :\n",src,grp);
		}
		
		while(tmp_iface != NULL)
		{
			tmp_mn = get_MN_by_id(mn_list, tmp_iface->iface_id);
			fprintf(log_file,"\t\tforwarded %d to %s\n",tmp_iface->nb_fwded,tmp_mn->tunnel_name);
			tmp_iface=tmp_iface->next;
		}
		fprintf(log_file,"\n");
		tmp_fwd=tmp_fwd->next;
	}
	
	return;
}
