/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "MLD_Proxy.h"
#include "MLD_MN.h"


/**
 * Gives the number of MN in the list 
 * @param list The MN List
 * @return The number f MN in the list
 */
int nb_MN(MN_t *list)
{
	int size=1;			/* the supposed lenght is 1 MN */
	MN_t * tmp = list;

	if( tmp == NULL )
		return 0;		/* if the list if empty -> return zero */
	
	while( tmp->next != NULL )	/* if there is another MN in the list -> size++ */
	{
		size++;
		tmp = tmp->next;
	}
	return size;
}

/* check if a MN is in the list */
int is_MN_by_name(MN_t *list, char *name)
{
	MN_t *tmp = list;

	if(list == NULL)
		return FALSE;

	while(tmp != NULL)
	{
		if( strcmp(name,tmp->tunnel_name) == 0)
			return TRUE;
		else
			tmp = tmp->next;
	}
	return FALSE;
		
}

int is_MN_by_id(MN_t *list, int id)
{
	MN_t *tmp = list;

	if(list == NULL)
		return FALSE;

	while(tmp != NULL)
	{
		if( id == tmp->tunnel_id)
	        	return TRUE;
		else
			tmp = tmp->next;
	}
	return FALSE;
}

int is_MN_by_addr(MN_t *list, struct in6_addr addr)
{
	MN_t *tmp = list;

	if(list == NULL)
		return FALSE;

	while(tmp != NULL)
	{
		if( IN6_ARE_ADDR_EQUAL(tmp->home_addr.s6_addr32,addr.s6_addr32) )
	        	return TRUE;
		else
			tmp = tmp->next;
	}
	return FALSE;
}


/* Add a MN in the list */
int add_MN(MN_t **list, char *name )
{
	int id;
	MN_t *tmp=*list, *new=NULL;
	
	if( is_MN_by_name(*list,name) )	/* if the MN is already in the list -> error */
	{
		fprintf(log_file,"ADHOC interface %s is already set in the list.\n",name);
		return FALSE;
	}
	
	id = if_nametoindex(name);	/* sys id of the tunnel/MN */
	if(id == 0)
	{
		fprintf(log_file,"ADHOC interface (%s) does not appear to be a valid interface\n",name);
		if( debug_lvl == VERBOSE )
			fprintf(stderr,"ADHOC interface (%s) does not appear to be a valid interface\n",name);
		return FALSE;
	}

	new = (MN_t *)malloc(sizeof(MN_t));
	if(new == NULL)
	{
		perror("Malloc in add_MN");
		return FALSE;
	}

	/* inits new parameters */
	strcpy(new->tunnel_name,name);
	new->tunnel_id = id;
	new->home_addr = in6addr_any;
	new->nb_fwded = 0;
	new->nb_sent = 0;

	if(*list != NULL)			/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;
	
	if(*list == NULL)
		*list = new;
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added ADHOC interface %s with sys id %d in MFC.\n", new->tunnel_name, new->tunnel_id);
	
	return TRUE;
}

/* Removes a MN from the list */
int del_MN_by_name(MN_t **list, char *name)
{
	MN_t *tmp=*list;
	
	if( !is_MN_by_name(*list,name) )	/* if the MN is not already in the list -> error */
	{
		fprintf(log_file,"Unable to delete %s from MN List : not in it.\n", name);
		return FALSE;
	}
	
	if(*list != NULL)			/* if list not empty */
		while(tmp != NULL)		/* check if MN in list */
		{
			if( !strcmp(tmp->tunnel_name,name) )	/* this is the MN */
			{
				if( tmp == *list)
				{
					*list = tmp->next;
					return TRUE;
				}
				else if( nb_MN(*list) == 1)		/* if it is the last MN */
					*list = NULL;
				else				/* else adjust pointers */
				{
					if(tmp->previous != NULL)
					{
						MN_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						MN_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				free(tmp);
				if(debug_lvl == VERBOSE)
					fprintf(log_file,"Deleted MN %s from MN List\n", name);
				return TRUE;
			}
			else
				tmp = tmp->next;
		}
	return FALSE;
}

int del_MN_by_id(MN_t **list, int id)
{
	MN_t *tmp=*list;
	
	if( !is_MN_by_id(*list,id) )	/* if the MN is not already in the list -> error */
	{
		/* LOG */
		return FALSE;
	}
	
	if(*list != NULL)			/* if list not empty */
		while(tmp != NULL)		/* check if MN in list */
		{
			if( tmp->tunnel_id == id )		/* this is the MN */
			{
				if( nb_MN(*list) == 1)		/* if it is the last MN */
				{
					*list = NULL;
					free(tmp);
				}
				else if( tmp == (*list) )
				{
					(*list) = tmp->next;
					(*list)->previous = NULL;
					free(tmp);
				}
				else				/* else adjust pointers */
				{
					
					if(tmp->previous != NULL)
					{
						MN_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						MN_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				free(tmp);
				if(debug_lvl == VERBOSE)
					fprintf(log_file,"Deleted MN with id %d from MN List\n", id);
				return TRUE;
			}
			else
				tmp = tmp->next;
		}
	return FALSE;
}

/* Gets the structure identifying a MN */
MN_t * get_MN_by_name(MN_t *list, char *name)
{
	MN_t * tmp = list;
	int length = nb_MN(list),i;

	if( length == 0)
		return NULL;

	for(i=0;i<length;i++)
	{
		if( !strcmp( tmp->tunnel_name, name) )
			return tmp;
		else
			tmp=tmp->next;
	}
	return NULL;
}

MN_t * get_MN_by_id(MN_t *list, int id)
{
	MN_t * tmp = list;
	int length = nb_MN(list),i;

	if( length == 0)
		return NULL;

	for(i=0;i<length;i++)
	{
		if( tmp->tunnel_id == id )
			return tmp;
		else
			tmp=tmp->next;
	}
	return NULL;
}

MN_t * get_MN_by_addr(MN_t *list, struct in6_addr addr)
{
	MN_t * tmp = list;
	int length = nb_MN(list),i;

	if( length == 0)
		return NULL;

	for(i=0;i<length;i++)
	{
		if( IN6_ARE_ADDR_EQUAL(tmp->home_addr.s6_addr32,addr.s6_addr32) )
			return tmp;
		else
			tmp=tmp->next;
	}
	return NULL;
}


/* Inits the MN list */
int init_MN_table( MN_t **list, char *name, int nb,int indice)
{
	fprintf(log_file,"Creating MFC cache ...\n");

	if( add_MN(list, name) == FALSE)
	{
		free_MN(list);
		fprintf(log_file,"Unable to add ADHOC if (%s - %d) in the MFC.\n", name, int_adhoc_id);
		fprintf(log_file,"Initialisation Failed.\n");
		fprintf(stderr,"Unable to add ADHOC if (%s - %d) in the MFC.\n", name, int_adhoc_id);
		fprintf(stderr,"Initialisation Failed.\n");
		return FALSE;
	}
	
	return TRUE;
}

/* Frees the complete MN List */
void free_MN( MN_t ** list)
{
	MN_t *tmp = *list, *next;
	int length = nb_MN(*list), i;
	
	if(debug_lvl == VERBOSE)
		fprintf(log_file,"\nFreeing MN list...\n");
	
	if(*list == NULL)
		return;
	
	if(length == 0)
	{
		*list = NULL;
		return;
	}
	
	for(i=0;i<length;i++)
	{
		next = tmp->next;
		free(tmp);
		tmp = next;
	}

	*list = NULL;
	return;
}

/* Prints the MN List */
void print_MN(MN_t * list)
{
	MN_t *tmp = list;
	char addr[INET6_ADDRSTRLEN];

	printf("In the MN list there are %d MN.\n", nb_MN(list));
	if( list == NULL)
	{
		return;
	}
	
	inet_ntop(AF_INET6,tmp->home_addr.s6_addr,addr,INET6_ADDRSTRLEN);
	printf("\t( Tunnel : %s\tID : %d\t Home Addr : %s )\n", tmp->tunnel_name, tmp->tunnel_id, addr);
	while(tmp->next != NULL)
	{
		tmp = tmp->next;
		inet_ntop(AF_INET6,tmp->home_addr.s6_addr,addr,INET6_ADDRSTRLEN);
		printf("\t( Tunnel : %s\tID : %d\t Home Addr : %s )\n", tmp->tunnel_name, tmp->tunnel_id, addr);
	}
	printf("\n");
}

void fprint_MN(MN_t * list)
{
	MN_t *tmp = list;
	char addr[INET6_ADDRSTRLEN];

	fprintf(log_file,"In the MN list there are %d MN.\n", nb_MN(list));
	if( list == NULL)
	{
		return;
	}
	
	inet_ntop(AF_INET6,tmp->home_addr.s6_addr,addr,INET6_ADDRSTRLEN);
	fprintf(log_file,"\t( Tunnel : %s\tID : %d\t Home Addr : %s )\n", tmp->tunnel_name, tmp->tunnel_id, addr);
	while(tmp->next != NULL)
	{
		tmp = tmp->next;
		inet_ntop(AF_INET6,tmp->home_addr.s6_addr,addr,INET6_ADDRSTRLEN);
		fprintf(log_file,"\t( Tunnel : %s\tID : %d\t Home Addr : %s )\n", tmp->tunnel_name, tmp->tunnel_id, addr);
	}
	printf("\n");
}
/* Stats */
/**
 * Init the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_mn_fwded(MN_t **list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_mn_fwded : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"init_mn_fwded : Interface %d not in MN List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_fwded = 0;
	return TRUE;
}
/**
 * Init the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_mn_sent(MN_t **list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"init_mn_sent : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"init_mn_sent : Interface %d not in MFC List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_sent = 0;
	return TRUE;
}
/**
 * Add 1 to the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_mn_fwded(MN_t **list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_mn_fwded : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"inc_mn_fwded : Interface %d not in MFC List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_fwded += 1;
	return TRUE;
}
/**
 * Add 1 to the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_mn_sent(MN_t **list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_mn_sent : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"inc_mn_sent : Interface %d not in MFC List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_sent += 1;
	return TRUE;
}
/**
 * Add n to the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_mn_fwded(MN_t **list, int iface, int n)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_n_mn_fwded : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"inc_n_mn_fwded : Interface %d not in MFC List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_fwded += n;
	return TRUE;
}
/**
 * Add n to the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_mn_sent(MN_t **list, int iface,int n)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(*list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"inc_n_mn_sent : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"inc_n_mn_sent : Interface %d not in MFC List.\n",iface);
		return FALSE;
	}
	
	mobile->nb_sent += n;
	return TRUE;
}
/**
 * Get the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int get_mn_fwded(MN_t *list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_mn_fwded : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"get_mn_fwded : Interface %d not in MFC List.\n",iface);	
		return -1;
	}
	
	return mobile->nb_fwded;
}
/**
 * Get the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int get_mn_sent(MN_t *list, int iface)
{
	MN_t *mobile = NULL;
	mobile = get_MN_by_id(list,iface);
	
	if( mobile == NULL )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"get_mn_sent : Interface %d not in MFC List.\n",iface);
		fprintf(log_file,"get_mn_sent : Interface %d not in MFC List.\n",iface);
		return -1;
	}
	
	return mobile->nb_sent;
}

/**
 * Print stats on the packets setn and received for each MN on the standard output
 * @param list The MN list
 */
void print_mn_stat(MN_t *list)
{
	MN_t *tmp = list;
	
	while(tmp != NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"%s\n\tsent %d packets to %s\n\treceived %d packets from %s\n\n",
		tmp->tunnel_name,tmp->nb_fwded,tmp->tunnel_name,tmp->nb_sent,tmp->tunnel_name);
		tmp=tmp->next;
	}
}
/**
 * Print stats on the packets setn and received for each MN in the log_file
 * @param list The MN list
 */
void fprint_mn_stat(MN_t *list)
{
	MN_t *tmp = list;
	
	while(tmp != NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(log_file,"\t%s\n\tsent %d packets to %s\n\treceived %d packets from %s\n\n",
		tmp->tunnel_name,tmp->nb_fwded,tmp->tunnel_name,tmp->nb_sent,tmp->tunnel_name);
		tmp=tmp->next;
	}
}
