/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "MLD_Proxy.h"
#include "MLD_membership.h"

/* inits the struct membership */
int init_membership(membership_t *list, char *name)
{
	strcpy(list->oif_name,name);
	list->oif_id = if_nametoindex(name);
	
	if( get_v6_addr( list->oif_id, IPV6_LINKLOCAL, &(list->oif_addr) ) == -1 )
	{
		fprintf(log_file, "Unable to fetch LAN Interface (%s) IPv6 link-local address\n", name);
		fprintf(stderr, "Unable to fetch LAN Interface (%s) IPv6 link-local address\n", name);
		return FALSE;
	}
	
	list->joined_groups = NULL;
	return TRUE;
}

/* add group */
int add_mult_group_by_addr_c(membership_t *list, char *addr)
{
	struct in6_addr grp_addr;
	mult_group_t *new;
	mult_group_t *tmp = list->joined_groups;
	
	new = (mult_group_t *)malloc(sizeof(mult_group_t));
	if(new == NULL)
	{
		perror("malloc new in add_mult_group");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, addr, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_mult_group");
		return FALSE;
	}
	
	if( is_group_joined_by_addr(*list, grp_addr) )	/* if the MN is already in the list -> error */
	{
		fprintf(log_file,"Group %s is already joined.\n",addr);
		return FALSE;
	}
	
	new->group_addr = grp_addr;
	if(IN6_IS_ADDR_MC_LINKLOCAL(grp_addr.s6_addr))
		new->scope = MC_LINK;
	else if(IN6_IS_ADDR_MC_GLOBAL(grp_addr.s6_addr))
		new->scope = MC_GLOBAL;
	else if(IN6_IS_ADDR_MC_SITELOCAL(grp_addr.s6_addr))
		new->scope = MC_SITE;
	new->mode = MC_ASM;
	new->source_addr = in6addr_any;

	if(list->joined_groups != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(list->joined_groups == NULL)
	{
		list->joined_groups = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added Group %s.\n", addr);
	
	return TRUE;
}

int add_mult_group_by_addr(membership_t *list, struct in6_addr grp_addr)
{
	char addr[INET6_ADDRSTRLEN];
	mult_group_t *new;
	mult_group_t *tmp = list->joined_groups;
	
	new = (mult_group_t *)malloc(sizeof(mult_group_t));
	if(new == NULL)
	{
		perror("malloc new in add_mult_group");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,grp_addr.s6_addr,addr,INET6_ADDRSTRLEN);
	
	if( is_group_joined_by_addr(*list, grp_addr) )	/* if the MN is already in the list -> error */
	{
		fprintf(log_file,"Group %s is already joined.\n",addr);
		return FALSE;
	}
	
	new->group_addr = grp_addr;
	if(IN6_IS_ADDR_MC_LINKLOCAL(grp_addr.s6_addr))
		new->scope = MC_LINK;
	else if(IN6_IS_ADDR_MC_GLOBAL(grp_addr.s6_addr))
		new->scope = MC_GLOBAL;
	else if(IN6_IS_ADDR_MC_SITELOCAL(grp_addr.s6_addr))
		new->scope = MC_SITE;
	new->mode = MC_ASM;
	new->source_addr = in6addr_any;

	if(list->joined_groups != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(list->joined_groups == NULL)
	{
		list->joined_groups = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added Group %s.\n", addr);
	
	return TRUE;
}

	
int add_mult_group_SSM_by_addr_c(membership_t *list, char *source_addr, char *group_addr)
{
	struct in6_addr grp_addr, src_addr;
	mult_group_t *new;
	mult_group_t *tmp = list->joined_groups;
	
	new = (mult_group_t *)malloc(sizeof(mult_group_t));
	if(new == NULL)
	{
		perror("malloc new in add_mult_group");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, group_addr, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in add_mult_group");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source_addr, &src_addr)) == -1)
	{
		perror("Unable to get v6 address in add_mult_group");
		return FALSE;
	}
	
	if( is_group_joined_SSM_by_addr(*list, src_addr, grp_addr) )	/* if the MN is already in the list -> error */
	{
		fprintf(log_file,"Group (%s,%s) is already joined.\n",source_addr, group_addr);
		return FALSE;
	}

	new->group_addr = grp_addr;
	new->source_addr = src_addr;
	if(IN6_IS_ADDR_MC_LINKLOCAL(grp_addr.s6_addr))
		new->scope = MC_LINK;
	else if(IN6_IS_ADDR_MC_GLOBAL(grp_addr.s6_addr))
		new->scope = MC_GLOBAL;
	else if(IN6_IS_ADDR_MC_SITELOCAL(grp_addr.s6_addr))
		new->scope = MC_SITE;
	new->mode = MC_SSM;

	if(list->joined_groups != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(list->joined_groups == NULL)
	{
		list->joined_groups = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added Group (%s,%s).\n", source_addr, group_addr);
	
	return TRUE;
}

int add_mult_group_SSM_by_addr(membership_t *list, struct in6_addr source_addr, struct in6_addr group_addr )
{
	char src_addr[INET6_ADDRSTRLEN],grp_addr[INET6_ADDRSTRLEN];
	mult_group_t *new;
	mult_group_t *tmp = list->joined_groups;
	
	new = (mult_group_t *)malloc(sizeof(mult_group_t));
	if(new == NULL)
	{
		perror("malloc new in add_mult_group");
		return FALSE;
	}
	
	inet_ntop(AF_INET6,group_addr.s6_addr,grp_addr,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source_addr.s6_addr,src_addr,INET6_ADDRSTRLEN);
	
	if( is_group_joined_SSM_by_addr(*list, source_addr, group_addr) )	/* if the MN is already in the list -> error */
	{
		fprintf(log_file,"Group (%s,%s) is already joined.\n",src_addr, grp_addr);
		return FALSE;
	}

	new->group_addr = group_addr;
	new->source_addr = source_addr;
	if(IN6_IS_ADDR_MC_LINKLOCAL(group_addr.s6_addr))
		new->scope = MC_LINK;
	else if(IN6_IS_ADDR_MC_GLOBAL(group_addr.s6_addr))
		new->scope = MC_GLOBAL;
	else if(IN6_IS_ADDR_MC_SITELOCAL(group_addr.s6_addr))
		new->scope = MC_SITE;
	new->mode = MC_SSM;

	if(list->joined_groups != NULL)				/* if list not empty */
		while(tmp->next != NULL)	/* go to the end of the list */
			tmp=tmp->next;

	if(list->joined_groups == NULL)
	{
		list->joined_groups = new;
	}
	else
	{
		tmp->next = new;
		new->previous = tmp;
		new->next = NULL;
	}

	if(debug_lvl == VERBOSE)
		fprintf(log_file,"Added Group (%s,%s).\n", src_addr, grp_addr);
	
	return TRUE;
}


/* del group */
int del_mult_group_by_addr_c(membership_t *list, char *addr)
{
	struct in6_addr group;
	
	if(!is_group_joined_by_addr_c(*list,addr))
	{
		fprintf(log_file,"Unable to delete %s from membership List : not in it.\n", addr);
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, addr, &group)) == -1)
	{
		perror("Unable to get v6 address in del_mult_group_by_addr_c");
		return FALSE;
	}
	
	return del_mult_group_by_addr(list, group);
}

int del_mult_group_by_addr(membership_t *list, struct in6_addr addr)
{
	mult_group_t *tmp = list->joined_groups;
	char grp[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,addr.s6_addr,grp,INET6_ADDRSTRLEN);
	
	if(!is_group_joined_by_addr(*list,addr))
	{
		fprintf(log_file,"Unable to delete %s from membership List : not in it.\n", grp);
		return FALSE;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_ASM)
		{
			char grp1[INET6_ADDRSTRLEN];
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp1,INET6_ADDRSTRLEN);
			if(!strcmp(grp1,grp))
//			if(tmp->group_addr.s6_addr == addr.s6_addr)
			{
				if( nb_joined_group(*list) == 1)		/* if it is the last MN */
				{
					list->joined_groups = NULL;
					free(tmp);
				}
				else if( tmp == list->joined_groups)
				{
					list->joined_groups = tmp->next;
					(list->joined_groups)->previous = NULL;
					free(tmp);
				}
				else				/* else adjust pointers */
				{
					if(tmp->previous != NULL)
					{
						mult_group_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						mult_group_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				fprintf(log_file,"%s deleted successfully from membership list.\n",grp);
				return TRUE;
			}
		}
		tmp = tmp->next;
	}
	
	return FALSE;
}


int del_mult_group_SSM_by_addr_c(membership_t *list, char *source_addr, char *group_addr)
{
	struct in6_addr group,source;
	
	if(!is_group_joined_SSM_by_addr_c(*list,source_addr,group_addr))
	{
		fprintf(log_file,"Unable to delete (%s,%s) from membership List : not in it.\n", source_addr,group_addr);
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, group_addr, &group)) == -1)
	{
		perror("Unable to get v6 group address in del_mult_group_by_addr_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, source_addr, &source)) == -1)
	{
		perror("Unable to get v6 source address in del_mult_group_by_addr_c");
		return FALSE;
	}
	
	return del_mult_group_SSM_by_addr(list, source,group);
}

int del_mult_group_SSM_by_addr(membership_t *list, struct in6_addr source_addr, struct in6_addr group_addr )
{
	mult_group_t *tmp = list->joined_groups;
	char grp[INET6_ADDRSTRLEN], src[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,group_addr.s6_addr,grp,INET6_ADDRSTRLEN);
	inet_ntop(AF_INET6,source_addr.s6_addr,src,INET6_ADDRSTRLEN);
	
	if(!is_group_joined_SSM_by_addr(*list,source_addr, group_addr))
	{
		fprintf(log_file,"Unable to delete (%s,%s) from membership List : not in it.\n", src,grp);
		return FALSE;
	}
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_SSM)
		{
			char grp1[INET6_ADDRSTRLEN],src1[INET6_ADDRSTRLEN];
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp1,INET6_ADDRSTRLEN);
			inet_ntop(AF_INET6,tmp->source_addr.s6_addr,src1,INET6_ADDRSTRLEN);
			if((!strcmp(grp1,grp)) && (!strcmp(src1,src)))
//			if( (tmp->group_addr.s6_addr == group_addr.s6_addr) && (tmp->source_addr.s6_addr == source_addr.s6_addr) )
			{
				if( nb_joined_group(*list) == 1)		/* if it is the last MN */
				{
					list->joined_groups = NULL;
					free(tmp);
				}
				else if( tmp == list->joined_groups)
				{
					list->joined_groups = tmp->next;
					(list->joined_groups)->previous = NULL;
					free(tmp);
				}
				else 				/* else adjust pointers */
				{
					if(tmp->previous != NULL)
					{
						mult_group_t *pre_tmp = tmp->previous;
						pre_tmp->next = tmp->next;
					}
					if(tmp->next !=NULL)
					{
						mult_group_t *post_tmp = tmp->next;
						post_tmp->previous = tmp->previous;
					}
				}
				fprintf(log_file,"(%s,%s) deleted successfully from membership list.\n",src,grp);
				return TRUE;
			}
		}
		tmp = tmp->next;
	}
	
	return FALSE;
}


int del_all_mult_group(membership_t *list)
{	
	mult_group_t *tmp = list->joined_groups, *next;
	int length = nb_joined_group(*list), i;
	
	if(length == 0)
	{
		list->joined_groups = NULL;
		return TRUE;
	}
	
	for(i=0;i<length;i++)
	{
		next = tmp->next;
		free(tmp);
		tmp = next;
	}

	list->joined_groups = NULL;
	return TRUE;
}


/* print membership */
void print_mult_group(membership_t list)
{
	int size,i;
	mult_group_t *tmp = list.joined_groups;
	char grp_addr[INET6_ADDRSTRLEN];
	char src_addr[INET6_ADDRSTRLEN];
	
	size = nb_joined_group(list);
	printf("In the membership list, there are %d joined groups.\n", size);
	if(size == 0)
		return;
	
	for(i=1;i<=size;i++)
	{
		if(i==1)
			printf("\t{\t");
		if(tmp->mode == MC_ASM)
		{
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp_addr,INET6_ADDRSTRLEN);
			printf("%s\t",grp_addr);
		}
		else if(tmp->mode == MC_SSM)
		{
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp_addr,INET6_ADDRSTRLEN);
			inet_ntop(AF_INET6,tmp->source_addr.s6_addr,src_addr,INET6_ADDRSTRLEN);
			printf("(%s,%s)\t",src_addr,grp_addr);
		}
		if((i%4==0) && (size%4!=0))
			printf("\n\t\t");
		tmp = tmp->next;
	}
	printf("}\n\n");
}

void print_membership(membership_t list)
{
	char addr[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,list.oif_addr.s6_addr,addr,INET6_ADDRSTRLEN);
	printf("\nLAN Interface %s, sys ID %d, IPv6 address %s\n",list.oif_name, list.oif_id, addr);
	print_mult_group(list);
}

void fprint_mult_group(membership_t list)
{
	int size,i;
	mult_group_t *tmp = list.joined_groups;
	char grp_addr[INET6_ADDRSTRLEN];
	char src_addr[INET6_ADDRSTRLEN];
	
	size = nb_joined_group(list);
	if(size == 0)
		return;
	fprintf(log_file,"In the membership list, there are %d joined groups.\n", size);
	for(i=1;i<=size;i++)
	{
		if(i==1)
			fprintf(log_file,"\t{\t");
		if(tmp->mode == MC_ASM)
		{
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp_addr,INET6_ADDRSTRLEN);
			fprintf(log_file,"%s\t",grp_addr);
		}
		else if(tmp->mode == MC_SSM)
		{
			inet_ntop(AF_INET6,tmp->group_addr.s6_addr,grp_addr,INET6_ADDRSTRLEN);
			inet_ntop(AF_INET6,tmp->source_addr.s6_addr,src_addr,INET6_ADDRSTRLEN);
			fprintf(log_file,"(%s,%s)\t",src_addr,grp_addr);
		}
		if((i%4==0) && (size%4!=0))
			fprintf(log_file,"\n\t\t");
		tmp = tmp->next;
	}
	fprintf(log_file,"}\n\n");
}

void fprint_membership(membership_t list)
{
	char addr[INET6_ADDRSTRLEN];
	inet_ntop(AF_INET6,list.oif_addr.s6_addr,addr,INET6_ADDRSTRLEN);
	fprintf(log_file,"\nLAN Interface %s, sys ID %d, IPv6 address %s\n",list.oif_name, list.oif_id, addr);
	fprint_mult_group(list);
}



/* get the number of joined groups */
int nb_joined_group(membership_t list)
{
	int size = 0;
	mult_group_t *tmp = list.joined_groups;
	
	while(tmp != NULL)
	{
		size++;
		tmp = tmp->next;
	}
	
	return size;
}

/* check if group is joined */
int is_group_joined_by_addr_c(membership_t list, char *addr)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, addr, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return FALSE;
	}
	
	return is_group_joined_by_addr(list,grp_addr);
}

int is_group_joined_by_addr(membership_t list, struct in6_addr addr)
{
	mult_group_t *tmp = list.joined_groups;
	
	if(nb_joined_group(list) == 0)
		return FALSE;
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_ASM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->group_addr.s6_addr32,addr.s6_addr32) )
				return TRUE;
		}
		tmp = tmp->next;
	}
	
	return FALSE;
}

int is_group_joined_SSM_by_addr_c(membership_t list, char *src_addr, char *grp_addr)
{
	struct in6_addr group_addr, source_addr;
	
	if( (inet_pton(AF_INET6, grp_addr, &group_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return FALSE;
	}
	
	if( (inet_pton(AF_INET6, src_addr, &source_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return FALSE;
	}
	
	return is_group_joined_SSM_by_addr(list,source_addr,group_addr);
}

int is_group_joined_SSM_by_addr(membership_t list, struct in6_addr src_addr, struct in6_addr grp_addr )
{
	mult_group_t *tmp = list.joined_groups;
	
	if(nb_joined_group(list) == 0)
		return FALSE;
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_SSM)
		{	
			if( IN6_ARE_ADDR_EQUAL(tmp->group_addr.s6_addr32,grp_addr.s6_addr32) && IN6_ARE_ADDR_EQUAL(tmp->source_addr.s6_addr32,src_addr.s6_addr32) )
				return TRUE;
		}
		tmp = tmp->next;
	}
	
	return FALSE;
}


int are_groups_joined(membership_t list)
{
	if(nb_joined_group(list) == 0)
		return FALSE;
	else
		return TRUE;
}

/* get a group */
mult_group_t * get_group_by_addr_c(membership_t list, char *addr)
{
	struct in6_addr grp_addr;
	
	if( (inet_pton(AF_INET6, addr, &grp_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return NULL;
	}
	
	return get_group_by_addr(list,grp_addr);
}

mult_group_t * get_group_by_addr(membership_t list, struct in6_addr addr)
{
	mult_group_t *tmp = list.joined_groups;
	
	if(nb_joined_group(list) == 0)
		return NULL;
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_ASM)
		{
			if( IN6_ARE_ADDR_EQUAL(tmp->group_addr.s6_addr32,addr.s6_addr32) )
				return tmp;
		}
		tmp = tmp->next;
	}
	
	return NULL;
}
mult_group_t * get_group_SSM_by_addr_c(membership_t list, char *src_addr, char *grp_addr)
{
	struct in6_addr group_addr, source_addr;
	
	if( (inet_pton(AF_INET6, grp_addr, &group_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return NULL;
	}
	
	if( (inet_pton(AF_INET6, src_addr, &source_addr)) == -1)
	{
		perror("Unable to get v6 address in is_group_joined_by_addr_c");
		return NULL;
	}
	
	return get_group_SSM_by_addr(list,source_addr,group_addr);
}

mult_group_t * get_group_SSM_by_addr(membership_t list, struct in6_addr src_addr, struct in6_addr grp_addr )
{
	mult_group_t *tmp = list.joined_groups;
	
	if(nb_joined_group(list) == 0)
		return NULL;
	
	while(tmp != NULL)
	{
		if(tmp->mode == MC_SSM)
		{	
			if( IN6_ARE_ADDR_EQUAL(tmp->group_addr.s6_addr32,grp_addr.s6_addr32)
				&& IN6_ARE_ADDR_EQUAL(tmp->source_addr.s6_addr32,src_addr.s6_addr32) )
				return tmp;
		}
		tmp = tmp->next;
	}
	
	return NULL;
}


mult_group_t * get_groups(membership_t list)
{
	return list.joined_groups;
}


void free_membership( membership_t *list)
{
	mult_group_t *tmp = list->joined_groups, *next;
	int length = nb_joined_group(*list), i;
	
	if(debug_lvl == VERBOSE)
		fprintf(log_file,"\nFreeing membership list...\n");
	
	if(list == NULL)
		return;
	
	if(length == 0)
	{
		list->joined_groups = NULL;
		return;
	}
	
	for(i=0;i<length;i++)
	{
		next = tmp->next;
		free(tmp);
		tmp = next;
	}

	list->joined_groups = NULL;
	return;
}
