/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef _MLD_PROXY_FORWARD_H_
#define _MLD_PROXY_FORWARD_H_

#include "MLD_Proxy.h"
#include "MLD_membership.h"
#include "MLD_MN.h"
#include <netinet/in.h>
#include <netinet/ip6.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <unistd.h>
#include <stdlib.h>

/** The structure identifying the filtered sources for an interface */
typedef struct filtered_src{
	/** the address of the filtered source */
	struct in6_addr src_addr;
	/** Pointer to the next item */
	struct filtered_src *next;
	/** Pointer to the previous item */
	struct filtered_src *previous;
}filtered_src_t;

/** The structure identifying an interface on which multicast data will be forwarded to */
typedef struct iface{
	/** The interface system id */
	int iface_id;
	/** The filter mode for this interface, MC_INCLUDE or MC_EXCLUDE */
	int filter;
	/** The list of filtered sources, only used in ASM mode, an entry per (src,grp) in SSM mode */
	filtered_src_t * sources;
	/** The time when the entry will become invalid */
	time_t time_out;
	/** The number of multicast data packets forwarded on this interface */
	int nb_fwded;
	/** Pointer to the next item */
	struct iface *next;	
	/** Pointer to the previous item */
	struct iface *previous;
}iface_t;

/** The structure identifying en entry in the frowarding table */
typedef struct forward{
	/** Address of the multicast group to forward */
	struct in6_addr grp_addr;
	/** Address of the source of the SSM multicast channel to forward, in6_addr_any in ASM */
	struct in6_addr src_addr;
	/** The multicast mode of the group, MC_ASM or MC_SSM */
	int mode;
	/** List of interfaces to forward multicast data from this group to */
	iface_t * interfaces;
	/** Pointer to the next item */
	struct forward *next;
	/** Pointer to the previous item */
	struct forward *previous;
}forward_t;


/*   manipulate interface list */
/*   TODO : TIMER   */
/* ASM */

/**
 * Add an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @param iface The system id of the interface to add
 * @return TRUE if the tunnel is successfully added in the list, FALSE otherwise
 */
int add_interface_c(forward_t **list, char *group, int iface);
/**
 * Add an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @param iface The system id of the interface to add
 * @return TRUE if the tunnel is successfully added in the list, FALSE otherwise
 */
int add_interface(forward_t **list, struct in6_addr group, int iface);
/**
 * Delete an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @param iface The system id of the interface to delete
 * @return TRUE if the tunnel is successfully deleted in the list, FALSE otherwise
 */
int del_interface_c(forward_t **list, char *group, int iface);
/**
 * Delete an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @param iface The system id of the interface to delete
 * @return TRUE if the tunnel is successfully deleted in the list, FALSE otherwise
 */
int del_interface(forward_t **list, struct in6_addr group, int iface);
/**
 * Check The forwarding of an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @param iface The system id of the interface to find
 * @return TRUE if the tunnel is found the list, FALSE otherwise
 */
int is_interface_set_c(forward_t *list, char *group, int iface);
/**
 * Check The forwarding of an interface in the list for a given ASM group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @param iface The system id of the interface to find
 * @return TRUE if the tunnel is found in the list, FALSE otherwise
 */
int is_interface_set(forward_t *list, struct in6_addr group, int iface);
/**
 * Give the number of interfaces in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return The number of interfaces for the given group, -1 if an error occurs
 */
int nb_interface_c(forward_t *list, char *group);
/**
 * Give the number of interfaces in the list for a given ASM group 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @return The number of interfaces for the given group, -1 if an error occurs
 */
int nb_interface(forward_t *list, struct in6_addr group);
/**
 * Get an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @param iface The system id of the interface to get
 * @return A pointer to the interface structure if it is found, NULL otherwise
 */
iface_t * get_interface_c(forward_t *list, char *group, int iface);
/**
 * Get an interface in the list for a given ASM group 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @param iface The system id of the interface to get
 * @return A pointer to the interface structure if it is found, NULL otherwise
 */
iface_t * get_interface(forward_t *list, struct in6_addr group, int iface);
/**
 * Delete all the interfaces in the list for a given ASM group 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return TRUE if the interfaces are successfully deleted in the list, FALSE otherwise
 */
int remove_all_interface_c(forward_t **list, char *group);
/**
 * Delete all the interfaces in the list for a given ASM group 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM Group
 * @return TTRUE if the interfaces are successfully deleted in the list, FALSE otherwise
 */
int remove_all_interface(forward_t **list, struct in6_addr group);
/**
 * Print the interfaces in the list for a given ASM group on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group A string identifying the ASM Group
 */
void printf_interface_c(forward_t *list, MN_t *mobiles, char *group);
/**
 * Print the interfaces in the list for a given ASM group on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group The IPv6 address of the ASM Group
 */
void printf_interface(forward_t *list, MN_t *mobiles, struct in6_addr group);
/**
 * Print the interfaces in the list for a given ASM group in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group A string identifying the ASM Group
 */
void fprintf_interface_c(forward_t *list, MN_t *mobiles, char *group);
/**
 * Print the interfaces in the list for a given ASM group in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group The IPv6 address of the ASM Group
 */
void fprintf_interface(forward_t *list, MN_t *mobiles, struct in6_addr group);
/* SSM */
/**
 * Add an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface to add
 * @return TRUE if the tunnel is successfully added in the list, FALSE otherwise
 */
int add_interface_SSM_c(forward_t **list, char *source, char *group, int iface);
/**
 * Add an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface to add
 * @return TRUE if the tunnel is successfully added in the list, FALSE otherwise
 */
int add_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Delete an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface to delete
 * @return TRUE if the tunnel is successfully deleted in the list, FALSE otherwise
 */
int del_interface_SSM_c(forward_t **list, char *source, char *group, int iface);
/**
 * Delete an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface to delete
 * @return TRUE if the tunnel is successfully deleted in the list, FALSE otherwise
 */
int del_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Check if an interface is in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface to find
 * @return TRUE if the tunnel is successfully found in the list, FALSE otherwise
 */
int is_interface_SSM_set_c(forward_t *list, char *source, char *group, int iface);
/**
 * Check if an interface is in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface to find
 * @return TRUE if the tunnel is successfully found in the list, FALSE otherwise
 */
int is_interface_SSM_set(forward_t *list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Give the number of interfaces in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return The number of interfaces for the given SSM Channel, -1 otherwise
 */
int nb_interface_SSM_c(forward_t *list, char *source, char *group);
/**
 * Give the number of interfaces in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return The number of interfaces for the given SSM Channel, -1 otherwise
 */
int nb_interface_SSM(forward_t *list, struct in6_addr source, struct in6_addr group);
/**
 * Get an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface to get
 * @return A pointer to the interface if it is successfully found in the list, FALSE otherwise
 */
iface_t * get_interface_SSM_c(forward_t *list, char *source, char *group, int iface);
/**
 * Get an interface in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface to get
 * @return A pointer to the interface if it is successfully found in the list, FALSE otherwise
 */
iface_t * get_interface_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Remove all the interfaces in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int remove_all_interface_SSM_c(forward_t **list, char *source, char *group);
/**
 * Remove all the interfaces in the list for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int remove_all_interface_SSM(forward_t **list, struct in6_addr source, struct in6_addr group);
/**
 * Print the interfaces in the list for a given SSM Channel on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 */
void printf_interface_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group);
/**
 * Print the interfaces in the list for a given SSM Channel on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 */
void printf_interface_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group);
/**
 * Print the interfaces in the list for a given SSM Channel in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 */
void fprintf_interface_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group);
/**
 * Print the interfaces in the list for a given SSM Channel in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 */
void fprintf_interface_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group);


/******   manipulate filtered source list   ******/
/**
 * Add a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source A string identifying the source to filter
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int add_filtered_source_c(forward_t **list, char *source, char *group, int iface);
/**
 * Add a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source The IPv6 address of the source to filter
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int add_filtered_source(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Delete a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source A string identifying the source to delete
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int del_filtered_source_c(forward_t **list, char *source, char *group, int iface);
/**
 * Delete a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source The IPv6 address of the source to delete
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int del_filtered_source(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Check if a source is in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source A string identifying the source to find
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int is_filtered_source_c(forward_t *list, char *source, char *group, int iface);
/**
 * Check if a source is in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source The IPv6 address of the source to find
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int is_filtered_source(forward_t *list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Get a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source A string identifying the source to get
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return A pointer to the source if success, NULL otherwise
 */
filtered_src_t * get_filtered_source_c(forward_t *list, char *group, int iface);
/**
 * Get a source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source The IPv6 address of the source to get
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return A pointer to the source if success, NULL otherwise
 */
filtered_src_t * get_filtered_source(forward_t *list, struct in6_addr group, int iface);
/**
 * Remove all source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int remove_all_filtered_source_c(forward_t **list, char *group, int iface);
/**
 * Remove all source in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int remove_all_filtered_source(forward_t **list, struct in6_addr group, int iface);
/**
 * Get the number of sources in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return The number of sources if success, -1 otherwise
 */
int nb_filtered_source_c(forward_t *list, char *group, int iface);
/**
 * Get the number of sources in the filter for the given interface and ASM group
 * @param list The forwarding table
 * @param source The IPv6 address of the source to get
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return The number of sources if success, -1 otherwise
 */
int nb_filtered_source(forward_t *list, struct in6_addr group, int iface);
/**
 * Print the sources in the filter for the given interface and ASM group on the standard output
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 */
void printf_source_c(forward_t *list, char *group, int iface);
/**
 * Print the sources in the filter for the given interface and ASM group on the standard output
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 */
void printf_source(forward_t *list, struct in6_addr group, int iface);
/**
 * Print the sources in the filter for the given interface and ASM group in the log file
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 */
void fprintf_source_c(forward_t *list, char *group, int iface);
/**
 * Print the sources in the filter for the given interface and ASM group in the log file
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 */
void fprintf_source(forward_t *list, struct in6_addr group, int iface);


/*   manipulated forward list   */
/* ASM */
/**
 * Add an ASM group in the list 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return TRUE if success, FALSE otherwise
 */
int add_forwarded_group_c(forward_t **list, char *group);
/**
 * Add an ASM group in the list 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @return TRUE if success, FALSE otherwise
 */
int add_forwarded_group(forward_t **list, struct in6_addr group);
/**
 * Delete an ASM group in the list 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return TRUE if success, FALSE otherwise
 */
int del_forwarded_group_c(forward_t **list, char *group);
/**
 * Delete an ASM group in the list 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @return TRUE if success, FALSE otherwise
 */
int del_forwarded_group(forward_t **list, struct in6_addr group);
/**
 * Check if an ASM group is forwarded in the list 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return TRUE if success, FALSE otherwise
 */
int is_group_forwarded_c(forward_t *list, char *group);
/**
 * Check if an ASM group is forwarded in the list 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @return TRUE if success, FALSE otherwise
 */
int is_group_forwarded(forward_t *list, struct in6_addr group);
/**
 * Get an ASM group in the list 
 * @param list The forwarding table
 * @param group A string identifying the ASM Group
 * @return A pointer to the group if success, NULL otherwise
 */
forward_t * get_forwarded_group_c(forward_t *list, char *group);
/**
 * Get an ASM group in the list 
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @return A pointer to the group if success, NULL otherwise
 */
forward_t * get_forwarded_group(forward_t *list, struct in6_addr group);
/* SSM */
/**
 * Add an SSM Channel in the list 
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int add_forwarded_group_SSM_c(forward_t **list, char *source, char *group);
/**
 * Add an SSM Channel in the list 
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int add_forwarded_group_SSM(forward_t **list, struct in6_addr source, struct in6_addr group);
/**
 * Delete an SSM Channel in the list 
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int del_forwarded_group_SSM_c(forward_t **list, char *source, char *group);
/**
 * Delete an SSM Channel in the list 
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int del_forwarded_group_SSM(forward_t **list, struct in6_addr source, struct in6_addr group);
/**
 * Check if an SSM Channel is in the list 
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int is_group_forwarded_SSM_c(forward_t *list, char *source, char *group);
/**
 * Check if an SSM Channel is in the list 
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return TRUE if success, FALSE otherwise
 */
int is_group_forwarded_SSM(forward_t *list, struct in6_addr source, struct in6_addr group);
/**
 * Get an SSM Channel in the list 
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @return A pointer to the SSM Channel if success, FALSE otherwise
 */
forward_t * get_forwarded_group_SSM_c(forward_t *list, char *source, char *group);
/**
 * Get an SSM Channel in the list 
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @return A pointer to the SSM Channel if success, FALSE otherwise
 */
forward_t * get_forwarded_group_SSM(forward_t *list, struct in6_addr source, struct in6_addr group);
/* BOTH */
/**
 * Print the ASM Group in the list for a given ASM group on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group A string identifying the ASM Group
 */
void print_forwarded_group_c(forward_t *list, MN_t *mobiles, char *group);
/**
 * Print the ASM Group in the list for a given ASM group on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group The IPv6 address of the ASM Group
 */
void print_forwarded_group(forward_t *list, MN_t *mobiles, struct in6_addr group);
/**
 * Print the ASM Group in the list for a given ASM group in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group A string identifying the ASM Group
 */
void fprint_forwarded_group_c(forward_t *list, MN_t *mobiles, char *group);
/**
 * Print the ASM Group in the list for a given ASM group in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param group The IPv6 address of the ASM Group
 */
void fprint_forwarded_group(forward_t *list, MN_t *mobiles, struct in6_addr group);
/**
 * Print the  SSM Channel on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 */
void print_forwarded_group_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group);
/**
 * Print the  SSM Channel on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 */
void print_forwarded_group_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group);
/**
 * Print the  SSM Channel in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 */
void fprint_forwarded_group_SSM_c(forward_t *list, MN_t *mobiles, char *source, char *group);
/**
 * Print the  SSM Channel in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 */
void fprint_forwarded_group_SSM(forward_t *list, MN_t *mobiles, struct in6_addr source, struct in6_addr group);
/**
 * Print all the forwarded groups on the standard output
 * @param list The forwarding table
 * @param mobiles The MN list
 */
void print_all_forwarded_group(forward_t *list, MN_t *mobiles);
/**
 * Print all the forwarded groups in the log file
 * @param list The forwarding table
 * @param mobiles The MN list
 */
void fprint_all_forwarded_group(forward_t *list, MN_t *mobiles);
/**
 * Get the number of forwarded groups
 * @param list The forwarding table
 * @return The number of forwarded groups if success, -1 otherwise 
 */
int nb_forwarded_group(forward_t *list);
/**
 * Remove all the forwarded groups
 * @param list The forwarding table
 */
void free_forwarded_group(forward_t **list);

/* Timers */
/**
 * Reset the timer for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_c(forward_t **list, char *group, int iface);
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer(forward_t **list, struct in6_addr group, int iface);
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_SSM_c(forward_t **list, char *source, char *group, int iface);
/**
 * Reset the timer for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int reset_timer_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);

/**
 * Get the time out for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_c(forward_t *list, char *group, int iface);
/**
 * Get the time out for the given ASM Group on a given interface
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer(forward_t *list, struct in6_addr group, int iface);
/**
 * Get the time out for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_SSM_c(forward_t *list, char *source, char *group, int iface);
/**
 * Get the time out for the given SSM Channel on a given interface
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
time_t get_timer_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Checks all the timers  in the forwarding table and deletes the corresponding entries if the timer has expired
 * @param list The forwarding table
 * @return TRUE if success, FALSE otherwise
 */
int check_timers(forward_t **list);

/**
 * Checks all entries in the forwarding table, and deletes the ones with no interface to forward to or the ones which are not joined
 * on the LAN interface
 * @param fwd_list The forwarding table
 * @param member_list The membership structure
 * @return TRUE if success, FALSE otherwise
 */
int check_forward(forward_t **fwd_list, membership_t *member_list);


/* Stats */
/**
 * Init the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_c(forward_t **list, char *group, int iface);
/**
 * Init the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat(forward_t **list, struct in6_addr group, int iface);
/**
 * Add 1 to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_c(forward_t **list, char *group, int iface);
/**
 * Add 1 to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat(forward_t **list, struct in6_addr group, int iface);
/**
 * Add n to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_c(forward_t **list, char *group, int iface, int n);
/**
 * Add n to the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat(forward_t **list, struct in6_addr group, int iface, int n);
/**
 * Get the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group A string identifying the ASM group
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given ASM Group if success, -1 otherwise
 */
int get_fwd_stat_c(forward_t *list, char *group, int iface);
/**
 * Get the number of packets forwarded to an interface for a given ASM Group
 * @param list The forwarding table
 * @param group The IPv6 address of the ASM group
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given ASM Group if success, -1 otherwise
 */
int get_fwd_stat(forward_t *list, struct in6_addr group, int iface);

/**
 * Init the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface);
/**
 * Init the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Add 1 to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface);
/**
 * Add 1 to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface);
/**
 * Add n to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_SSM_c(forward_t **list, char *source, char *group, int iface, int n);
/**
 * Add n to the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @param n The number of packets to add
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_fwd_stat_SSM(forward_t **list, struct in6_addr source, struct in6_addr group, int iface, int n);
/**
 * Get the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source A string identifying the source from the SSM Channel
 * @param group A string identifying the group from the SSM Channel
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given SSM Channel if success, -1 otherwise
 */
int get_fwd_stat_SSM_c(forward_t *list, char *source, char *group, int iface);
/**
 * Get the number of packets forwarded to an interface for a given SSM Channel
 * @param list The forwarding table
 * @param source The IPv6 address of the source from the SSM Channel
 * @param group The IPv6 address of the group from the SSM Channel
 * @param iface The system id of the interface
 * @return The number of packets forwarded to the interface for a given SSM Channel if success, -1 otherwise
 */
int get_fwd_stat_SSM(forward_t *list, struct in6_addr source, struct in6_addr group, int iface);


/**
 * Print stats on forwarded packets on the standard output
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 */
void print_fwd_stat(forward_t *fwd_list, MN_t *mn_list);
/**
 * Print stats on forwarded packets on the standard output
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 */
void fprint_fwd_stat(forward_t *fwd_list, MN_t *mn_list);


/*
	print
	fprint
*/

#endif
