/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/**
 * The header file of the Mobile Node (MN) list
 * A mobile is identified by its mipv6 tunnel
 */

#ifndef _MLD_PROXY_MN_H_
#define _MLD_PROXY_MN_H_

#include "MLD_Proxy.h"
#include <netinet/in.h>
#include <netinet/ip6.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <unistd.h>
#include <stdlib.h>

/** The structure identiying the MN */
typedef struct MN{
	/** Interface Name */
	char tunnel_name[16];
	/** Interface ID */
	int tunnel_id;
	/** MN Home Address */
	struct in6_addr home_addr;
	/** The nulber of packets forwarded to this MN */
	int nb_fwded;
	/** The number of packets sent by this MN */
	int nb_sent;
	/** Pointer to the next item */
	struct MN *next;
	/** Pointer to the previous item */
	struct MN *previous;		
}MN_t;

/**
 * Give the number of MN in the list 
 * @param list The MN List
 * @return The number f MN in the list
 */
int nb_MN(MN_t *list);

/**
 * Check if a MN is in the list 
 * @param list The MN List
 * @param name The name of the mipv6 tunnel
 * @return TRUE if the tunnel is in the list, FALSE otherwise
 */
int is_MN_by_name(MN_t *list, char *name);
/**
 * Check if a MN is in the list 
 * @param list The MN List
 * @param id The system id of the tunnel
 * @return TRUE if the tunnel is in the list, FALSE otherwise
 */
int is_MN_by_id(MN_t *list, int id);
/**
 * Check if a MN is in the list 
 * @param list The MN List
 * @param addr The MN Home Address
 * @return TRUE if the tunnel is in the list, FALSE otherwise
 */
int is_MN_by_addr(MN_t *list, struct in6_addr addr);

/**
 * Add a MN in the list 
 * @param list The MN List
 * @param name The name of the mipv6 tunnel
 * @return TRUE if the tunnel is successfully added in the list, FALSE otherwise
 */
int add_MN(MN_t **list, char *name );

/**
 * Delete a MN in the list 
 * @param list The MN List
 * @param name The name of the mipv6 tunnel
 * @return TRUE if the tunnel is successfully deleted from the list, FALSE otherwise
 */
int del_MN_by_name(MN_t **list, char *name);
/**
 * Adds a MN in the list 
 * @param list The MN List
 * @param id The system id of the tunnel
 * @return TRUE if the tunnel is successfully deleted from the list, FALSE otherwise
 */
int del_mn_by_id(MN_t **list, int id);

/**
 * Get a MN from the list 
 * @param list The MN List
 * @param name The name of the mipv6 tunnel
 * @return A pointer to the MN if the tunnel is in the list, NULL otherwise
 */
MN_t * get_MN_by_name(MN_t *list, char *name);
/**
 * Get a MN from the list 
 * @param id The system id of the tunnel
 * @param name The name of the mipv6 tunnel
 * @return A pointer to the MN if the tunnel is in the list, NULL otherwise
 */
MN_t * get_MN_by_id(MN_t *list, int id);
/**
 * Get a MN from the list 
 * @param list The MN List
 * @param addr The MN Home Address
 * @return A pointer to the MN if the tunnel is in the list, NULL otherwise
 */
MN_t * get_MN_by_addr(MN_t *list, struct in6_addr addr);

/**
 * Initialisation of the MN list 
 * @param list The MN List
 * @param name The name of the mipv6 tunnel prefixe
 * @param nb The nulber of tunnels, e.g. "eth" in "eth0" 
 * @param indice The index of the first tunnel, e.g. "0" in "eth0"
 * @return TRUE if the list is successfully initialiazed, FALSE otherwise
 */
int init_MN_table( MN_t **list, char *name, int nb,int indice);

/**
 * Print the MN List to the standard output 
 * @param list The MN List
 */
void print_MN(MN_t * list);
/**
 * Print the MN List in the log file
 * @param list The MN List
 */
void fprint_MN(MN_t * list);

/**
 * Free the MN List
 * @param list The MN List
 */
void free_MN( MN_t ** list);

/* Stats */
/**
 * Init the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_mn_fwded(MN_t **list, int iface);
/**
 * Init the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int init_mn_sent(MN_t **list, int iface);
/**
 * Add 1 to the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_mn_fwded(MN_t **list, int iface);
/**
 * Add 1 to the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_mn_sent(MN_t **list, int iface);
/**
 * Add n to the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_mn_fwded(MN_t **list, int iface, int n);
/**
 * Add n to the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int inc_n_mn_sent(MN_t **list, int iface,int n);
/**
 * Get the number of packets received on an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int get_mn_fwded(MN_t *list, int iface);
/**
 * Get the number of packets sent by an interface
 * @param list The MN list
 * @param iface The system id of the interface
 * @return TRUE if success, FALSE otherwise
 */
int get_mn_sent(MN_t *list, int iface);

/**
 * Print stats on the packets setn and received for each MN on the standard output
 * @param list The MN list
 */
void print_mn_stat(MN_t *list);
/**
 * Print stats on the packets setn and received for each MN in the log_file
 * @param list The MN list
 */
void fprint_mn_stat(MN_t *list);


#endif
