#include <linux/module.h>
#include <linux/fs.h>	
#include <linux/poll.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include "lte_driver.h"
#include "lte_user_if.h"
static LIST_HEAD(dev_list);

static LIST_HEAD(app_event_q);
static spinlock_t app_event_q_lock = SPIN_LOCK_UNLOCKED;
static int app_event_q_count = 0;
static atomic_t lte_ch_open_count = ATOMIC_INIT(0);
static DECLARE_WAIT_QUEUE_HEAD(lte_ch_poll_wait_q);
#define MAX_LTE_EVENT_QUEUE_LEN 20

static void parse_app_data(struct lte_modem_st * modem, struct lte_event_st * event)
{
	switch(event->type)
	{
		case LTE_PACKET:
			printk("modem %s Tx Packet, len = %d, 0x%02X 0x%02X 0x%02X 0x%02X\n", event->if_name, event->data.packet.len,
					event->data.packet.body[0], event->data.packet.body[1], event->data.packet.body[2], 
					event->data.packet.body[3]);
			break;

		case LTE_GET_CONFIG:
			printk("modem %s configuration\n", event->if_name);
			modem->network_mode = event->data.lte_config.network_mode;
			strcpy(modem->apn, event->data.lte_config.apn);
			strcpy(modem->pin_code, event->data.lte_config.pin);
			LTE_PRINT(("Network Mode = %d, APN = %s, PIN = XXXX\n",
						modem->network_mode, modem->apn));
			break;

		case LTE_GET_MY_DNS_IPS:
			printk("modem %s My IP = %d.%d.%d.%d DNS IP = %d.%d.%d.%d\n", event->if_name, 
				event->data.ips.my_ip[0], event->data.ips.my_ip[1], event->data.ips.my_ip[2], event->data.ips.my_ip[3],
				event->data.ips.dns_ip[0], event->data.ips.dns_ip[1], event->data.ips.dns_ip[2], event->data.ips.dns_ip[3]);
			memcpy(modem->my_ip, event->data.ips.my_ip, 4);
			memcpy(modem->dns_ip, event->data.ips.dns_ip, 4);
			modem->my_ip_dns_ip_valid = 1;
			setup_ping_dns_packet(modem);
			break;

		default:
			printk("modem %s unknown data[%d]\n", event->if_name, event->type);
	}
}

static int lte_enq_event(struct lte_event_st * event)
{
	unsigned long flags;
	spin_lock_irqsave(&app_event_q_lock, flags);
	list_add_tail(&event->list, &app_event_q);
	app_event_q_count++;

	if(app_event_q_count >= MAX_LTE_EVENT_QUEUE_LEN) printk("WARNING!!! app event queue too long!\n");

	spin_unlock_irqrestore(&app_event_q_lock, flags);
	return 0;
}

static struct lte_event_st * lte_deq_event(void)
{
	struct lte_event_st * event;
	unsigned long flags;
	spin_lock_irqsave(&app_event_q_lock, flags);
	if(list_empty(&app_event_q))
	{
		spin_unlock_irqrestore(&app_event_q_lock, flags);
		return 0;
	}
	event = (struct lte_event_st *)list_entry(app_event_q.next, struct lte_event_st, list);
	list_del(&event->list);
	if(app_event_q_count != 0)
	{
		app_event_q_count--;
	}
	else
	{
		printk("Try to decrease queue count while it is 0, should not happen!!\n");
	}

	spin_unlock_irqrestore(&app_event_q_lock, flags);
	return event;
}

static void fulsh_app_event_q(void)
{
	struct list_head * p;
	struct lte_event_st * event;
	unsigned long flags;

	spin_lock_irqsave(&app_event_q_lock, flags);
	list_for_each(p, &app_event_q)
	{
		event = list_entry(p, struct lte_event_st, list);
		kfree(event);
	}
	spin_unlock_irqrestore(&app_event_q_lock, flags);
	return;
}

void lte_tell_app(struct lte_event_st * event)
{
	if((atomic_read(&lte_ch_open_count) == 0) && (app_event_q_count >= MAX_LTE_EVENT_QUEUE_LEN))  
	{
		kfree(event);
		return;
	}
	
    //set polling mask and wakeup application that waiting in select() 
	lte_enq_event(event);
	wake_up_interruptible(&lte_ch_poll_wait_q);  
}

static int lte_ch_open (struct inode *inode, struct file *filp)
{
    LTE_TRACE_USER_IF(("---->lte_ch_open()\n"));
    atomic_inc(&lte_ch_open_count);
    LTE_TRACE_USER_IF(("<----lte_ch_open()\n"));
	return 0;
}

static int lte_ch_release(struct inode *inode, struct file *filp)
{
    LTE_TRACE_USER_IF(("---->lte_ch_release()\n"));
    atomic_dec(&lte_ch_open_count);
	fulsh_app_event_q();
	LTE_TRACE_USER_IF(("<----lte_ch_release()\n"));
	return 0;
}


static ssize_t lte_ch_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	struct lte_event_st * event;
    LTE_TRACE_USER_IF(("---->lte_ch_read()\n"));
	if(count < sizeof(struct lte_event_st))
	{
		printk("<--- buffer too small in lte_ch_read()\n");
		return 0;
	}

	event = lte_deq_event();
	if(event == 0)
	{
//		LTE_TRACE_USER_IF(("<----lte_ch_read(), no evnet to read\n"));
		return 0;
	}

	copy_to_user(buf, (char *) event, sizeof(struct lte_event_st));
	kfree(event);
    LTE_TRACE_USER_IF(("<----lte_ch_read()\n"));
    return 1;
}

static ssize_t lte_ch_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
	struct lte_modem_st * modem;
 	struct lte_event_st event;
    LTE_TRACE_USER_IF(("---->lte_ch_write()\n"));
	if(count > sizeof(struct lte_event_st))
	{
		printk("<--- buffer too big in lte_ch_write()\n");
		return 0;
	}
	copy_from_user(&event, buf, count);
	modem = if_name_to_modem(event.if_name);

	if(!modem)
	{
		printk("Serious error!!!! Can not find modem in lte_ch_write()\n");
		return 0;
	}

	parse_app_data(modem, &event);
    LTE_TRACE_USER_IF(("<----lte_ch_write()\n"));
    return 1;
}

static unsigned int lte_ch_poll(struct file *filp, poll_table *wait)
{
    unsigned int mask = 0;

    LTE_TRACE_USER_IF(("---->lte_ch_poll()\n"));
	poll_wait(filp, &lte_ch_poll_wait_q,  wait);

	if(!list_empty(&app_event_q)) mask = POLLIN | POLLRDNORM;	// read notification

    LTE_TRACE_USER_IF(("<----lte_ch_poll()\n"));
	return mask;
}


//our file operations
struct file_operations lte_ch_fops = {
	.owner	 = THIS_MODULE,
	.open	 = lte_ch_open,
	.release = lte_ch_release,
	.read	 = lte_ch_read,
	.write	 = lte_ch_write,
    .ioctl   = 0,
    .poll    = lte_ch_poll
};
