#!/bin/sh 
# Copyright (C) 2008 John Crispin <blogic@openwrt.org>

. /etc/functions.sh

IPTABLES="echo iptables"
IPTABLES=iptables

config_clear
include /lib/network
scan_interfaces

CONFIG_APPEND=1
config_load firewall

# load user firewall setting for traffic control
myfw=$(uci -q get myfirewall.rules.enable 2>/dev/null)
myfw="${myfw:-0}"
if [ "$myfw" == "1" ]; then
	config_load myfirewall
fi


config fw_zones
ZONE_LIST=$CONFIG_SECTION

CUSTOM_CHAINS=1
DEF_INPUT=DROP
DEF_OUTPUT=DROP
DEF_FORWARD=DROP

load_policy() {
	config_get input $1 input
	config_get output $1 output
	config_get forward $1 forward

	DEF_INPUT="${input:-$DEF_INPUT}"
	DEF_OUTPUT="${output:-$DEF_OUTPUT}"
	DEF_FORWARD="${forward:-$DEF_FORWARD}"
}

create_zone() {
	local exists
	
	[ "$1" == "loopback" ] && return

	config_get exists $ZONE_LIST $1
	[ -n "$exists" ] && return
	config_set $ZONE_LIST $1 1 

	$IPTABLES -N zone_$1
	$IPTABLES -N zone_$1_MSSFIX
	$IPTABLES -N zone_$1_ACCEPT
	$IPTABLES -N zone_$1_DROP
	$IPTABLES -N zone_$1_REJECT
	$IPTABLES -N zone_$1_forward
	[ "$4" ] && $IPTABLES -A output -j zone_$1_$4
	$IPTABLES -N zone_$1_nat -t nat
	$IPTABLES -N zone_$1_prerouting -t nat
	[ "$6" == "1" ] && $IPTABLES -t nat -A POSTROUTING -j zone_$1_nat
	[ "$7" == "1" ] && $IPTABLES -I FORWARD 1 -j zone_$1_MSSFIX
}


addif() {
	local dev
	local wan_protocol
	local out_if
	local wan_mtu
	
	#config_get dev core $2
	dev=`uci -q -P /var/state get firewall.core.$2 || echo ''`
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		config_get src_ip lan ipaddr 
		config_get src_mask lan netmask
		config_get wan_mtu $2 mtu
		config_get wan_protocol $2 proto
		if [ "$2" == "wan" ]; then
			delif "ppp1" "3gwan"
		else
			delif "${dev:-br-wan}" "wan"
		fi
	}
	[ -n "$dev" -a "$dev" != "$1" ] && delif "$dev" "$2"
	[ -n "$dev" -a "$dev" == "$1" ] && {
		logger -p 160.5 "[addif] interface $2: $1 exist."
		#return
	}
	[ "$wan_protocol" == "l2tp" -o "$wan_protocol" == "pptp" ] && {
		out_if="ppp0"
	} || {
		out_if="$1"
	}
	logger "adding $out_if to firewall zone $2"
	lock /var/lock/vtss_firewall
	$IPTABLES -D input -i $1 -j zone_$2 2>/dev/null
	$IPTABLES -A input -i $1 -j zone_$2
	
	# for VPN traffic to zone_wan
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		$IPTABLES -D input -i ipsec+ -j zone_$2 2>/dev/null
		$IPTABLES -A input -i ipsec+ -j zone_$2
		$IPTABLES -D input -i ppp3+ -j zone_$2 2>/dev/null
		$IPTABLES -A input -i ppp3+ -j zone_$2
		$IPTABLES -D input -i ppp6+ -j zone_$2 2>/dev/null
		$IPTABLES -A input -i ppp6+ -j zone_$2
	}
	
	[ -z "$wan_mtu" ] && {
	$IPTABLES -D zone_$2_MSSFIX -o $out_if -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu 2>/dev/null
	$IPTABLES -I zone_$2_MSSFIX 1 -o $out_if -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu
	} || {
	let "wan_mss = wan_mtu - 40"
	$IPTABLES -I zone_$2_MSSFIX 1 -o $out_if -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $wan_mss
	}
	$IPTABLES -D zone_$2_ACCEPT -o $1 -j ACCEPT 2>/dev/null
	$IPTABLES -I zone_$2_ACCEPT 1 -o $1 -j ACCEPT
	$IPTABLES -D zone_$2_DROP -o $1 -j DROP 2>/dev/null
	$IPTABLES -I zone_$2_DROP 1 -o $1 -j DROP
	$IPTABLES -D zone_$2_REJECT -o $1 -j reject 2>/dev/null
	$IPTABLES -I zone_$2_REJECT 1 -o $1 -j reject
	$IPTABLES -D zone_$2_ACCEPT -i $1 -j ACCEPT 2>/dev/null
	$IPTABLES -I zone_$2_ACCEPT 1 -i $1 -j ACCEPT
	$IPTABLES -D zone_$2_DROP -i $1 -j DROP 2>/dev/null
	$IPTABLES -I zone_$2_DROP 1 -i $1 -j DROP
	$IPTABLES -D zone_$2_REJECT -i $1 -j reject 2>/dev/null
	$IPTABLES -I zone_$2_REJECT 1 -i $1 -j reject
	
	add_VPN_interface() {
		$IPTABLES -D zone_$2_ACCEPT -o $1 -j ACCEPT 2>/dev/null
		$IPTABLES -A zone_$2_ACCEPT -o $1 -j ACCEPT
		$IPTABLES -D zone_$2_DROP -o $1 -j DROP 2>/dev/null
		$IPTABLES -A zone_$2_DROP -o $1 -j DROP
		$IPTABLES -D zone_$2_REJECT -o $1 -j reject 2>/dev/null
		$IPTABLES -A zone_$2_REJECT -o $1 -j reject
		$IPTABLES -D zone_$2_ACCEPT -i $1 -j ACCEPT 2>/dev/null
		$IPTABLES -A zone_$2_ACCEPT -i $1 -j ACCEPT
		$IPTABLES -D zone_$2_DROP -i $1 -j DROP 2>/dev/null
		$IPTABLES -A zone_$2_DROP -i $1 -j DROP
		$IPTABLES -D zone_$2_REJECT -i $1 -j reject 2>/dev/null
		$IPTABLES -A zone_$2_REJECT -i $1 -j reject
	}
	
	# for VPN traffic in zone_wan_XXX
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		add_VPN_interface ipsec+ $2
		add_VPN_interface ppp3+ $2
		add_VPN_interface ppp6+ $2
	}
	
	# address mapping before MASQUERADE
	set_addr_map()
	{
		config_get EN_MAP	$1 enable
		config_get WIP_IDX $1 wan_ip_idx
		[ "$EN_MAP" == "1" -a "$WIP_IDX" != "0" ] && {
			config_get PIP	$1 priv_ip
			config_get PMASK	$1 mask
			WIP=$(uci -q get wan_ip_alias.wa"$WIP_IDX".alias_ip || echo 0.0.0.0)
			EN_WIP=$(uci -q get wan_ip_alias.wa"$WIP_IDX".enable)
			if [ "$EN_WIP" = "1" -a "$WIP" != "0.0.0.0" ]; then
				config_get PROTO $1 protocol
				if [ "$PROTO" = "1" ]; then
					$IPTABLES -D zone_$3_nat -t nat -p tcp -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null
					$IPTABLES -I zone_$3_nat -t nat -p tcp -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null
				elif [ "$PROTO" = "2" ]; then
					$IPTABLES -D zone_$3_nat -t nat -p udp -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null
					$IPTABLES -I zone_$3_nat -t nat -p udp -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null			
				else
					$IPTABLES -D zone_$3_nat -t nat -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null
					$IPTABLES -I zone_$3_nat -t nat -o $2 ${PIP:+-s $PIP/$PMASK} -j SNAT --to-source $WIP 2>/dev/null
				fi
			fi
		}
	}
	[ "$2" != "lan" ] && {
		config_load "addr_map"
		config_foreach set_addr_map addrmap $1 $2
	}
	
	$IPTABLES -D zone_$2_nat -t nat -o $1 ${src_ip:+-s $src_ip/$src_mask} -j MASQUERADE 2>/dev/null
	$IPTABLES -A zone_$2_nat -t nat -o $1 ${src_ip:+-s $src_ip/$src_mask} -j MASQUERADE 
	$IPTABLES -D PREROUTING -t nat -i $1 -j zone_$2_prerouting 2>/dev/null
	$IPTABLES -I PREROUTING 1 -t nat -i $1 -j zone_$2_prerouting 
	$IPTABLES -D forward -i $1 -j zone_$2_forward 2>/dev/null
	$IPTABLES -A forward -i $1 -j zone_$2_forward
	# for VPN traffic to zone_wan_forward
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		$IPTABLES -D forward -i ipsec+ -j zone_$2_forward 2>/dev/null
		$IPTABLES -A forward -i ipsec+ -j zone_$2_forward
		$IPTABLES -D forward -i ppp3+ -j zone_$2_forward 2>/dev/null
		$IPTABLES -A forward -i ppp3+ -j zone_$2_forward
		$IPTABLES -D forward -i ppp6+ -j zone_$2_forward 2>/dev/null
		$IPTABLES -A forward -i ppp6+ -j zone_$2_forward
	}
	uci_set_state firewall core "$2" "$1"
	lock -u /var/lock/vtss_firewall
}

delif() {
	local wan_mtu
	logger "removing $1 from firewall zone $2"
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		config_get src_ip lan ipaddr 
		config_get src_mask lan netmask
		config_get wan_mtu $2 mtu
	}
	lock /var/lock/vtss_firewall
	$IPTABLES -D input -i $1 -j zone_$2 2>/dev/null
	[ -z "$wan_mtu" ] && {
	$IPTABLES -D zone_$2_MSSFIX -o $1 -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu 2>/dev/null
	} || {
	let "wan_mss = wan_mtu - 40"
	$IPTABLES -D zone_$2_MSSFIX -o $1 -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $wan_mss 2>/dev/null
	}	
	$IPTABLES -D zone_$2_ACCEPT -o $1 -j ACCEPT 2>/dev/null
	$IPTABLES -D zone_$2_DROP -o $1 -j DROP 2>/dev/null
	$IPTABLES -D zone_$2_REJECT -o $1 -j reject 2>/dev/null
	$IPTABLES -D zone_$2_ACCEPT -i $1 -j ACCEPT 2>/dev/null
	$IPTABLES -D zone_$2_DROP -i $1 -j DROP 2>/dev/null
	$IPTABLES -D zone_$2_REJECT -i $1 -j reject 2>/dev/null
	
	del_VPN_interface() {
		$IPTABLES -D zone_$2_ACCEPT -o $1 -j ACCEPT 2>/dev/null
		$IPTABLES -D zone_$2_DROP -o $1 -j DROP 2>/dev/null
		$IPTABLES -D zone_$2_REJECT -o $1 -j reject 2>/dev/null
		$IPTABLES -D zone_$2_ACCEPT -i $1 -j ACCEPT 2>/dev/null
		$IPTABLES -D zone_$2_DROP -i $1 -j DROP 2>/dev/null
		$IPTABLES -D zone_$2_REJECT -i $1 -j reject 2>/dev/null
	}
	# for VPN traffic
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		del_VPN_interface ipsec+ $2
		del_VPN_interface ppp3+ $2
		del_VPN_interface ppp6+ $2

		$IPTABLES -D input -i ipsec+ -j zone_$2 2>/dev/null
		$IPTABLES -D input -i ppp3+ -j zone_$2 2>/dev/null
		$IPTABLES -D input -i ppp6+ -j zone_$2 2>/dev/null

	}

	$IPTABLES -D zone_$2_nat -t nat -o $1 ${src_ip:+-s $src_ip/$src_mask} -j MASQUERADE 2>/dev/null
	$IPTABLES -D PREROUTING -t nat -i $1 -j zone_$2_prerouting 2>/dev/null
	$IPTABLES -D forward -i $1 -j zone_$2_forward 2>/dev/null
	# for VPN traffic
	[ "$2" == "wan" -o "$2" == "3gwan" ] && {
		$IPTABLES -D forward -i ipsec+ -j zone_$2_forward 2>/dev/null
		$IPTABLES -D forward -i ppp3+ -j zone_$2_forward 2>/dev/null
		$IPTABLES -D forward -i ppp6+ -j zone_$2_forward 2>/dev/null
	}
	uci_revert_state firewall core "$2"
	lock -u /var/lock/vtss_firewall
}

load_synflood() {
	local en_synflood=$(uci -q get dos_firewall.synflood.syn_flood || echo '1')
	[ "$en_synflood" == "1" ] && {
		local rate=${1:-$(uci -q get dos_firewall.synflood.syn_rate || echo 50)}
		local burst=${2:-$(uci -q get dos_firewall.synflood.syn_burst || echo 100)}

		echo "Loading synflood protection"
		$IPTABLES -N syn_flood
		$IPTABLES -A syn_flood -p tcp --syn -m limit --limit ${rate:-50}/second --limit-burst ${burst:-100} -j RETURN
		$IPTABLES -A syn_flood -j DROP
		$IPTABLES -A INPUT -p tcp --syn -j syn_flood
	}
}

fw_set_chain_policy() {
	local chain=$1
	local target=$2
	[ "$target" == "REJECT" ] && {
		$IPTABLES -A $chain -j reject
		target=DROP
	}
	$IPTABLES -P $chain $target
}

fw_defaults() {
	local wan_mtu
	[ -n "$DEFAULTS_APPLIED" ] && {
		echo "Error: multiple defaults sections detected"
		return;
	}
	DEFAULTS_APPLIED=1

	load_policy "$1"

	echo 1 > /proc/sys/net/ipv4/tcp_syncookies
	for f in /proc/sys/net/ipv4/conf/*/accept_redirects 
	do
		echo 0 > $f
	done
	for f in /proc/sys/net/ipv4/conf/*/accept_source_route 
	do
		echo 0 > $f
	done                                                                   
	
	uci_revert_state firewall core
	uci_set_state firewall core "" firewall_state 

	$IPTABLES -P INPUT DROP
	$IPTABLES -P OUTPUT DROP
	$IPTABLES -P FORWARD DROP

	$IPTABLES -F
	$IPTABLES -t mangle -F
	$IPTABLES -t nat -F
	$IPTABLES -t mangle -X
	$IPTABLES -t nat -X
	$IPTABLES -X
	
	config_get_bool drop_invalid $1 drop_invalid 1

	[ "$drop_invalid" -gt 0 ] && {
		$IPTABLES -A INPUT -m state --state INVALID -j DROP
		$IPTABLES -A OUTPUT -m state --state INVALID -j DROP
		$IPTABLES -A FORWARD -m state --state INVALID -j DROP
	}

	$IPTABLES -A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT
	$IPTABLES -A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT

	wan_proto=$(uci -q get network.wan.proto)
	if [ "$wan_proto" == "pppoe" ]; then
		config_get wan_mtu wan mtu
		wan_mtu="${wan_mtu:-1492}"
		let "wan_mss = wan_mtu - 40"
		$IPTABLES -A FORWARD -i ppp0 -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $wan_mss
		#$IPTABLES -A FORWARD -o ppp0 -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu
	fi
	
	# chain for CSM before state ESTABLISHED ACCEPT
	$IPTABLES -N forward_csm
	$IPTABLES -A FORWARD -j forward_csm
	
	$IPTABLES -A FORWARD -m state --state RELATED,ESTABLISHED -j ACCEPT
	
	$IPTABLES -A INPUT -i lo -j ACCEPT
	$IPTABLES -A OUTPUT -o lo -j ACCEPT

	config_get syn_flood $1 syn_flood
	config_get syn_rate $1 syn_rate
	config_get syn_burst $1 syn_burst
	[ "$syn_flood" == "1" ] && load_synflood $syn_rate $syn_burst

	$IPTABLES -N input
	$IPTABLES -N output
	$IPTABLES -N forward

	$IPTABLES -A INPUT -j input
	$IPTABLES -A OUTPUT -j output
	$IPTABLES -A FORWARD -j forward

	$IPTABLES -N reject
	$IPTABLES -A reject -p tcp -j REJECT --reject-with tcp-reset
	$IPTABLES -A reject -j REJECT --reject-with icmp-port-unreachable

	echo "Adding custom chains"
	fw_custom_chains

	$IPTABLES -A INPUT -i ! br-lan -j DROP

	fw_set_chain_policy INPUT "$DEF_INPUT"
	fw_set_chain_policy OUTPUT "$DEF_OUTPUT"
	fw_set_chain_policy FORWARD "$DEF_FORWARD"
}

fw_zone_defaults() {
	local name
	local network
	local masq

	config_get name $1 name
	config_get network $1 network
	config_get_bool masq $1 masq "0"
	config_get_bool conntrack $1 conntrack "0"
	config_get_bool mtu_fix $1 mtu_fix 0

	load_policy $1
	[ "$forward" ] && $IPTABLES -A zone_${name}_forward -j zone_${name}_${forward}
	[ "$input" ] && $IPTABLES -A zone_${name} -j zone_${name}_${input}
}

fw_zone() {
	local name
	local network
	local masq

	config_get name $1 name
	config_get network $1 network
	config_get masq $1 masq
	config_get_bool mtu_fix $1 mtu_fix 0
	load_policy $1

	[ -z "$network" ] && network=$name
	create_zone "$name" "$network" "$input" "$output" "$forward" "$masq" "$mtu_fix"
	fw_custom_chains_zone "$name"
}

fw_rule() {
	local enable
	local src 
	local src_ip
	local src_mac
	local src_port
	local src_mac
	local dest
	local dest_ip
	local dest_port
	local proto
	local icmp_type
	local target
	local ruleset
	local del_rule
	local xt_time

	del_rule=$2
	config_get enable $1 enable
	config_get src $1 src
	config_get src_ip $1 src_ip
	config_get src_mac $1 src_mac
	config_get src_port $1 src_port
	config_get dest $1 dest
	config_get dest_ip $1 dest_ip
	config_get dest_port $1 dest_port
	config_get proto $1 proto
	config_get icmp_type $1 icmp_type
	config_get target $1 target
	config_get ruleset $1 ruleset
	config_get time_obj $1 time_obj -1
	En3gBackUp=$(uci -q get network.3gwan.backup || echo 0)
	Is3gWANUP=$(uci -q -P /var/state get network.3gwan.up || echo 0)
	IsWANUP=$(uci -q -P /var/state get network.wan.up || echo 0)
	[ "$dest" == "wan" -a "$En3gBackUp" == "1" -a "$Is3gWANUP" == "1"  -a "$IsWANUP" == "0" ] && {
		dest="3gwan"
	}	
	
	if [ -n "$enable" -a "$enable" == "0" ]; then
		return
	fi
	src_port_first=${src_port%-*}
	src_port_last=${src_port#*-}
	[ "$src_port_first" -ne "$src_port_last" ] && { \
		src_port="$src_port_first:$src_port_last"; } || src_port="$src_port_first"

	dest_port_first=${dest_port%-*}
	dest_port_last=${dest_port#*-}
	[ "$dest_port_first" -ne "$dest_port_last" ] && { \
		dest_port="$dest_port_first:$dest_port_last"; } || dest_port="$dest_port_first"

	ZONE=input
	TARGET=$target
	[ -z "$target" ] && target=DROP
	[ -n "$src" -a -z "$dest" ] && ZONE=zone_$src
	[ -n "$src" -a -n "$dest" ] && ZONE=zone_${src}_forward
	[ -n "$dest" ] && TARGET=zone_${dest}_$target

	$IPTABLES -Z ${ZONE}_tc 2>/dev/null || {
		$IPTABLES -N ${ZONE}_tc
		[ "$ZONE" == "zone_lan_forward" ] && {
			ENABLE_IMP2P=$(uci -q get imp2p.rules.enable || echo 0)
			wcf_enable=$(uci -q get wcf.wcf.enable || echo 0)
			if [ "$ENABLE_IMP2P" == "1" -o "$wcf_enable" == "1" ]; then	
				$IPTABLES -I forward_csm 1 -i br-lan -j ${ZONE}_tc
			else
				$IPTABLES -I $ZONE 1 -j ${ZONE}_tc
			fi
		} || 
			$IPTABLES -I $ZONE 1 -j ${ZONE}_tc
	}
	ZONE=${ZONE}_tc		# zone_???_tc for traffic control rules
	
	eval 'RULE_COUNT=$((++RULE_COUNT_'$ZONE'))'

	xt_time=""
	trim() { echo $1; }
	[ "$time_obj" == "-1" ] || {		#set time obj
		delete=`uci -q get time_obj.$time_obj.delete || echo 0`
		[ "$delete" == "1" ] && xt_time="" ||
		{
			dstart=`uci -q get time_obj.$time_obj.datestart`
			dstop=`uci -q get time_obj.$time_obj.datestop`
			tsatrt=`uci -q get time_obj.$time_obj.timestart`
			tstop=`uci -q get time_obj.$time_obj.timestop`
			weekdays=`uci -q get time_obj.$time_obj.weekdays`
			xt_time=$(trim "${dstart:+--datestart $dstart} ${dstop:+--datestop $dstop} ${tsatrt:+--timestart $tsatrt} ${tstop:+--timestop $tstop} ${weekdays:+--weekdays $weekdays}")
		}
	}
	
	add_rule() {
		[ "$del_rule" == "delete" ] && $IPTABLES -D $ZONE $RULE_COUNT 		# delete before insert
		$IPTABLES -I $ZONE $RULE_COUNT \
			${proto:+-p $proto} \
			${icmp_type:+--icmp-type $icmp_type} \
			${src_ip:+-s $src_ip} \
			${src_port:+--sport $src_port} \
			${src_mac:+-m mac --mac-source $src_mac} \
			${dest_ip:+-d $dest_ip} \
			${dest_port:+--dport $dest_port} \
			${xt_time:+-m time $xt_time --tz $(date +%z | cut -c1-3):$(date +%z | cut -c4-5)} \
			-j $TARGET 
	}
	[ "$proto" == "tcpudp" ] && {
		proto=tcp
		add_rule
		eval 'RULE_COUNT=$((++RULE_COUNT_'$ZONE'))'
		proto=udp
		add_rule
		return
	}
	add_rule
}

fw_forwarding() {
	local src
	local dest
	local masq

	config_get src $1 src
	config_get dest $1 dest
	[ -n "$src" ] && z_src=zone_${src}_forward || z_src=forward
	[ -n "$dest" ] && z_dest=zone_${dest}_ACCEPT || z_dest=ACCEPT
	$IPTABLES -I $z_src 1 -j $z_dest
}

fw_redirect() {
	local src
	local src_ip
	local src_port
	local src_dport
	local src_mac
	local dest_ip
	local dest_port dest_port2
	local proto
	
	config_get src $1 src
	config_get src_ip $1 src_ip
	config_get src_dip $1 src_dip
	config_get src_port $1 src_port
	config_get src_dport $1 src_dport
	config_get src_mac $1 src_mac
	config_get dest_ip $1 dest_ip
	config_get dest_port $1 dest_port
	config_get proto $1 proto
	[ -z "$src" -o -z "$dest_ip" ] && { \
		echo "redirect needs src and dest_ip"; return ; }
	
	src_port_first=${src_port%-*}
	src_port_last=${src_port#*-}
	[ "$src_port_first" != "$src_port_last" ] && { \
		src_port="$src_port_first:$src_port_last"; }

	src_dport_first=${src_dport%-*}
	src_dport_last=${src_dport#*-}
	[ "$src_dport_first" != "$src_dport_last" ] && { \
		src_dport="$src_dport_first:$src_dport_last"; }

	dest_port2=${dest_port:-$src_dport}
	dest_port_first=${dest_port2%-*}
	dest_port_last=${dest_port2#*-}
	[ "$dest_port_first" != "$dest_port_last" ] && { \
		dest_port2="$dest_port_first:$dest_port_last"; }

	add_rule() {
		$IPTABLES -A zone_${src}_prerouting -t nat \
			${proto:+-p $proto} \
			${src_ip:+-s $src_ip} \
			${src_dip:+-d $src_dip} \
			${src_port:+--sport $src_port} \
			${src_dport:+--dport $src_dport} \
			${src_mac:+-m mac --mac-source $src_mac} \
			-j DNAT --to-destination $dest_ip${dest_port:+:$dest_port}

		$IPTABLES -I zone_${src}_forward 1 \
			${proto:+-p $proto} \
			-d $dest_ip \
			${src_ip:+-s $src_ip} \
			${src_port:+--sport $src_port} \
			${dest_port2:+--dport $dest_port2} \
			${src_mac:+-m mac --mac-source $src_mac} \
			-j ACCEPT 
	}
	[ "$proto" == "tcpudp" -o -z "$proto" ] && {
		proto=tcp
		add_rule
		proto=udp
		add_rule
		return
	}
	add_rule
}

fw_include() {
	local path
	config_get path $1 path
	[ -e $path ] && . $path
}

fw_addif() {
	local up
	local ifname
	config_get up $1 up
	config_get ifname $1 ifname
	[ -n "$up" ] || return 0
	(ACTION="ifup" INTERFACE="$1" . /etc/hotplug.d/iface/20-firewall)
}

fw_custom_chains() {
	[ -n "$CUSTOM_CHAINS" ] || return 0
	$IPTABLES -N input_rule
	$IPTABLES -N output_rule
	$IPTABLES -N forwarding_rule
	$IPTABLES -N prerouting_rule -t nat
	$IPTABLES -N postrouting_rule -t nat
			
	$IPTABLES -A INPUT -j input_rule
	$IPTABLES -A OUTPUT -j output_rule
	$IPTABLES -A FORWARD -j forwarding_rule
	$IPTABLES -A PREROUTING -t nat -j prerouting_rule
	$IPTABLES -A POSTROUTING -t nat -j postrouting_rule
}

fw_custom_chains_zone() {
	local zone="$1"

	[ -n "$CUSTOM_CHAINS" ] || return 0
	$IPTABLES -N input_${zone}
	$IPTABLES -N forwarding_${zone}
	$IPTABLES -N prerouting_${zone} -t nat
	$IPTABLES -I zone_${zone} 1 -j input_${zone}
	$IPTABLES -I zone_${zone}_forward 1 -j forwarding_${zone}
	$IPTABLES -I zone_${zone}_prerouting 1 -t nat -j prerouting_${zone}
}

fw_init() {
	DEFAULTS_APPLIED=

	lock /var/lock/vtss_firewall
	echo "Loading defaults"
	config_foreach fw_defaults defaults
	echo "Loading zones"
	config_foreach fw_zone zone
	echo "Loading forwarding"
	config_foreach fw_forwarding forwarding
	echo "Loading redirects"
	config_foreach fw_redirect redirect
	echo "Loading rules"
	config_foreach fw_rule rule
	echo "Loading includes"
	config_foreach fw_include include
	echo "Loading zone defaults"
	config_foreach fw_zone_defaults zone
	uci_set_state firewall core loaded 1
	lock -u /var/lock/vtss_firewall
	unset CONFIG_APPEND
	config_load network
	config_foreach fw_addif interface
}

# to reorder the traffic control rules or reload rules ONLY!!
fw_rule_del_ins() {
	lock /var/lock/vtss_firewall
	$IPTABLES -F input_tc 2>/dev/null
	$IPTABLES -F zone_lan_tc 2>/dev/null
	$IPTABLES -F zone_lan_forward_tc 2>/dev/null
	$IPTABLES -F zone_wan_tc 2>/dev/null
	$IPTABLES -F zone_wan_forward_tc 2>/dev/null
	$IPTABLES -F zone_3gwan_tc 2>/dev/null
	$IPTABLES -F zone_3gwan_forward_tc 2>/dev/null	
	logger "Reload or change order of traffic control rules..."
	config_foreach fw_rule rule
	lock -u /var/lock/vtss_firewall
}

fw_stop() {
	lock /var/lock/vtss_firewall
	$IPTABLES -F
	$IPTABLES -t mangle -F
	$IPTABLES -t nat -F
	$IPTABLES -t mangle -X
	$IPTABLES -t nat -X
	$IPTABLES -X
	$IPTABLES -P INPUT ACCEPT
	$IPTABLES -P OUTPUT ACCEPT
	$IPTABLES -P FORWARD ACCEPT
	uci_revert_state firewall core
	lock -u /var/lock/vtss_firewall
}
